<?php

namespace Tests\Unit;

use Tests\TestCase;
use Illuminate\Foundation\Testing\RefreshDatabase;
use App\Services\LeaveBalanceService;
use App\Models\Employee;
use App\Models\Leave;
use App\Models\LeaveType;
use App\Models\Holiday;
use App\Models\LeaveBalanceChange;
use Carbon\Carbon;

class LeaveBalanceServiceTest extends TestCase
{
    use RefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();
        // run the migrations
        $this->artisan('migrate', ['--env' => 'testing']);

        // seed leave types
        $this->seed(\Database\Seeders\LeaveTypesSeeder::class);
    }

    public function test_calculate_deducted_days_excludes_weekly_offs_and_holidays()
    {
        $employee = Employee::factory()->create([
            'monthly_leave_days_allowed' => 2,
            'weekly_off_days' => ['friday', 'saturday'],
            'hire_date' => Carbon::now()->subYear()->toDateString(),
        ]);

        // create a paid holiday in the middle
        Holiday::create(['date' => Carbon::now()->addDays(2)->toDateString(), 'is_paid' => true]);

        $leave = Leave::create([
            'employee_id' => $employee->id,
            'leave_type' => 'unpaid',
            'leave_type_id' => LeaveType::where('code', 'unpaid')->first()?->id,
            'start_date' => Carbon::now()->toDateString(),
            'end_date' => Carbon::now()->addDays(4)->toDateString(),
            'days_count' => 5,
            'status' => 'approved',
            'approved_at' => now(),
        ]);

        $service = new LeaveBalanceService();
        $deducted = $service->calculateDeductedDays($leave);

        // In the 5-day period there are 2 weekend days (fri,sat) and 1 paid holiday => deducted should be 2
        $this->assertEquals(2, $deducted);
    }

    public function test_apply_approval_creates_balance_change_and_snapshot_for_annual_leave()
    {
        $employee = Employee::factory()->create([
            'monthly_leave_days_allowed' => 2,
            'annual_entitlement' => 12,
            'weekly_off_days' => ['friday','saturday'],
            'hire_date' => Carbon::now()->subYear()->toDateString(),
        ]);

        $leaveType = LeaveType::where('code', 'annual')->first();

        $leave = Leave::create([
            'employee_id' => $employee->id,
            'leave_type' => 'annual',
            'leave_type_id' => $leaveType->id,
            'start_date' => Carbon::now()->toDateString(),
            'end_date' => Carbon::now()->addDays(1)->toDateString(),
            'days_count' => 2,
            'status' => 'pending',
        ]);

        $service = new LeaveBalanceService();
        $leave = $service->applyApproval($leave, 1);

        $this->assertEquals('approved', $leave->status);
        $this->assertNotNull($leave->applied_balance_change_id);
        $this->assertNotNull($leave->balance_snapshot);

        $change = LeaveBalanceChange::find($leave->applied_balance_change_id);
        $this->assertNotNull($change);
        $this->assertEquals(-1 * $leave->deducted_days, $change->days_change);
    }
}
