<?php

namespace Tests\Feature;

use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\WithFaker;
use Tests\TestCase;
use App\Models\User;
use App\Models\Role;
use App\Models\Permission;
use Illuminate\Support\Facades\Hash;

class UserRegistrationTest extends TestCase
{
    use RefreshDatabase, WithFaker;

    protected function setUp(): void
    {
        parent::setUp();
        
        // إنشاء الأدوار والصلاحيات للاختبار
        $this->createRolesAndPermissions();
    }

    private function createRolesAndPermissions()
    {
        // إعداد الصلاحيات كقائمة بسيطة — النموذج Role يستخدم حقل JSON 'permissions'
        $permissions = ['view_dashboard', 'view_employees', 'manage_employees'];

        // إنشاء أدوار أساسية للاختبار مع تخزين الصلاحيات كمصفوفة في الحقل 'permissions'
        $adminRole = Role::create([
            'name' => 'admin',
            'display_name' => 'مدير النظام',
            'description' => 'صلاحيات إدارية كاملة',
            'permissions' => $permissions,
        ]);

        $employeeRole = Role::create([
            'name' => 'employee',
            'display_name' => 'موظف',
            'description' => 'صلاحيات أساسية للموظف',
            'permissions' => ['view_dashboard'], // فقط عرض لوحة التحكم
        ]);
    }

    /** @test */
    public function user_can_register_with_valid_data_and_role()
    {
        $role = Role::where('name', 'employee')->first();
        
        $userData = [
            'name' => $this->faker->name,
            'email' => $this->faker->unique()->safeEmail,
            'password' => 'password123',
            'password_confirmation' => 'password123',
            'role_id' => $role->id,
        ];

        $response = $this->post(route('register'), $userData);

        // التحقق من إعادة التوجيه بنجاح
        $response->assertRedirect('/home');

        // التحقق من إنشاء المستخدم في قاعدة البيانات
        $this->assertDatabaseHas('users', [
            'name' => $userData['name'],
            'email' => $userData['email'],
            'status' => 'active',
        ]);

        // التحقق من ربط المستخدم بالدور
        $user = User::where('email', $userData['email'])->first();
        $this->assertTrue($user->hasRole('employee'));
        
        // التحقق من أن المستخدم لديه الصلاحيات المناسبة
        $this->assertTrue($user->hasPermission('view_dashboard'));
        $this->assertFalse($user->hasPermission('manage_employees'));
    }

    /** @test */
    public function user_registration_fails_without_role()
    {
        $userData = [
            'name' => $this->faker->name,
            'email' => $this->faker->unique()->safeEmail,
            'password' => 'password123',
            'password_confirmation' => 'password123',
            // لا يوجد role_id
        ];

        $response = $this->post(route('register'), $userData);

        // التحقق من وجود خطأ في التحقق
        $response->assertSessionHasErrors('role_id');
        
        // التحقق من عدم إنشاء المستخدم
        $this->assertDatabaseMissing('users', [
            'email' => $userData['email'],
        ]);
    }

    /** @test */
    public function user_registration_fails_with_invalid_role()
    {
        $userData = [
            'name' => $this->faker->name,
            'email' => $this->faker->unique()->safeEmail,
            'password' => 'password123',
            'password_confirmation' => 'password123',
            'role_id' => 999, // دور غير موجود
        ];

        $response = $this->post(route('register'), $userData);

        // التحقق من وجود خطأ في التحقق
        $response->assertSessionHasErrors('role_id');
        
        // التحقق من عدم إنشاء المستخدم
        $this->assertDatabaseMissing('users', [
            'email' => $userData['email'],
        ]);
    }

    /** @test */
    public function admin_user_gets_all_permissions()
    {
        $adminRole = Role::where('name', 'admin')->first();
        
        $userData = [
            'name' => $this->faker->name,
            'email' => $this->faker->unique()->safeEmail,
            'password' => 'password123',
            'password_confirmation' => 'password123',
            'role_id' => $adminRole->id,
        ];

        $response = $this->post(route('register'), $userData);

        $user = User::where('email', $userData['email'])->first();
        
        // التحقق من أن المدير لديه جميع الصلاحيات
        $this->assertTrue($user->hasRole('admin'));
        $this->assertTrue($user->hasPermission('view_dashboard'));
        $this->assertTrue($user->hasPermission('view_employees'));
        $this->assertTrue($user->hasPermission('manage_employees'));
    }

    /** @test */
    public function registration_form_displays_available_roles()
    {
        $response = $this->get(route('register'));

        $response->assertStatus(200);
        $response->assertViewIs('auth.register');
        $response->assertViewHas('roles');
        
        // التحقق من وجود الأدوار في العرض
        $roles = $response->viewData('roles');
        $this->assertCount(2, $roles); // admin و employee
        
        $roleNames = $roles->pluck('name')->toArray();
        $this->assertContains('admin', $roleNames);
        $this->assertContains('employee', $roleNames);
    }

    /** @test */
    public function user_is_automatically_logged_in_after_registration()
    {
        $role = Role::where('name', 'employee')->first();
        
        $userData = [
            'name' => $this->faker->name,
            'email' => $this->faker->unique()->safeEmail,
            'password' => 'password123',
            'password_confirmation' => 'password123',
            'role_id' => $role->id,
        ];

        $response = $this->post(route('register'), $userData);

        // التحقق من تسجيل دخول المستخدم تلقائياً
        $this->assertAuthenticated();
        
        // التحقق من أن المستخدم المسجل دخوله هو نفس المستخدم المسجل
        $user = User::where('email', $userData['email'])->first();
        $this->assertEquals($user->id, auth()->id());
    }

    /** @test */
    public function registration_validates_required_fields()
    {
        $response = $this->post(route('register'), []);

        $response->assertSessionHasErrors([
            'name',
            'email',
            'password',
            'role_id'
        ]);
    }

    /** @test */
    public function registration_validates_email_uniqueness()
    {
        $existingUser = User::factory()->create(['password' => bcrypt('password')]);
        $role = Role::where('name', 'employee')->first();

        $userData = [
            'name' => $this->faker->name,
            'email' => $existingUser->email, // بريد إلكتروني موجود مسبقاً
            'password' => 'password123',
            'password_confirmation' => 'password123',
            'role_id' => $role->id,
        ];

        $response = $this->post(route('register'), $userData);

        $response->assertSessionHasErrors('email');
    }

    /** @test */
    public function registration_validates_password_confirmation()
    {
        $role = Role::where('name', 'employee')->first();

        $userData = [
            'name' => $this->faker->name,
            'email' => $this->faker->unique()->safeEmail,
            'password' => 'password123',
            'password_confirmation' => 'differentpassword', // كلمة مرور مختلفة
            'role_id' => $role->id,
        ];

        $response = $this->post(route('register'), $userData);

        $response->assertSessionHasErrors('password');
    }

    /** @test */
    public function user_role_relationship_is_properly_established()
    {
        $role = Role::where('name', 'admin')->first();
        
        $userData = [
            'name' => $this->faker->name,
            'email' => $this->faker->unique()->safeEmail,
            'password' => 'password123',
            'password_confirmation' => 'password123',
            'role_id' => $role->id,
        ];

        $this->post(route('register'), $userData);

        $user = User::where('email', $userData['email'])->first();
        
        // التحقق من العلاقة في قاعدة البيانات
        $this->assertDatabaseHas('user_roles', [
            'user_id' => $user->id,
            'role_id' => $role->id,
        ]);
        
        // التحقق من العلاقة عبر Eloquent
        $this->assertTrue($user->roles->contains($role));
        $this->assertEquals($role->name, $user->roles->first()->name);
    }
}

