<?php

namespace Tests\Feature;

use Tests\TestCase;
use Illuminate\Foundation\Testing\RefreshDatabase;
use App\Models\Employee;
use App\Models\Attendance;
use App\Models\Payroll;
use Carbon\Carbon;

class PayrollOvertimeIntegrationTest extends TestCase
{
    use RefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();
        $this->artisan('migrate', ['--env' => 'testing']);
    }

    public function test_payroll_includes_overtime_component()
    {
        $employee = Employee::create([
            'employee_id' => 'EMP20250001',
            'name' => 'Test',
            'salary' => 1000,
            'incentives' => 0,
            'monthly_leave_days_allowed' => 0,
            'hire_date' => Carbon::now()->subYear()->toDateString(),
            'status' => 'active',
            'overtime_hourly_rate' => 10.00,
        ]);

        // create an attendance with 2 hours overtime
        Attendance::create([
            'employee_id' => $employee->id,
            'date' => Carbon::now()->startOfMonth()->addDays(1)->toDateString(),
            'check_in' => '09:00:00',
            'check_out' => '19:00:00',
            'scheduled_check_in' => '09:00',
            'scheduled_check_out' => '17:00',
        ]);

        // call payroll generation route/method - we will mimic controller's logic by instantiating controller
        $this->artisan('db:seed', ['--env' => 'testing']); // ensure any required seeders

        // bypass middleware (authentication/role) for the test so the controller can run
        $this->withoutMiddleware();

        // create payroll by calling route if exists
        $response = $this->post(route('payroll.generate-monthly'), [
            'month' => Carbon::now()->month,
            'year' => Carbon::now()->year,
        ]);

        // allow redirect
        $response->assertStatus(302);

        $payroll = Payroll::where('employee_id', $employee->id)->first();
        $this->assertNotNull($payroll);

        $components = $payroll->payrollSalaryComponents()->get();
        $overtimeComp = $components->firstWhere('name', 'Overtime');
        $this->assertNotNull($overtimeComp);
        $this->assertEquals(20.00, round($overtimeComp->value, 2)); // 2 hours * 10.00
    }
}
