<?php

namespace Tests\Feature;

use App\Models\Company;
use App\Models\User;
use App\Models\Employee;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class CompanyIsolationTest extends TestCase
{
    use RefreshDatabase;

    protected $company1;
    protected $company2;
    protected $user1;
    protected $user2;
    protected $employee1;
    protected $employee2;

    protected function setUp(): void
    {
        parent::setUp();

        // إنشاء شركتين
        $this->company1 = Company::create([
            'name' => 'الشركة الأولى',
            'email' => 'company1@test.com',
            'status' => 'active',
        ]);

        $this->company2 = Company::create([
            'name' => 'الشركة الثانية',
            'email' => 'company2@test.com',
            'status' => 'active',
        ]);

        // إنشاء مستخدمين لكل شركة
        $this->user1 = User::create([
            'name' => 'مستخدم الشركة الأولى',
            'email' => 'user1@test.com',
            'password' => bcrypt('password'),
            'company_id' => $this->company1->id,
            'role' => 'admin',
            'status' => 'active',
        ]);

        $this->user2 = User::create([
            'name' => 'مستخدم الشركة الثانية',
            'email' => 'user2@test.com',
            'password' => bcrypt('password'),
            'company_id' => $this->company2->id,
            'role' => 'admin',
            'status' => 'active',
        ]);

        // إنشاء موظفين لكل شركة
        $this->employee1 = Employee::create([
            'employee_id' => 'EMP001',
            'name' => 'موظف الشركة الأولى',
            'email' => 'emp1@test.com',
            'company_id' => $this->company1->id,
            'status' => 'active',
        ]);

        $this->employee2 = Employee::create([
            'employee_id' => 'EMP002',
            'name' => 'موظف الشركة الثانية',
            'email' => 'emp2@test.com',
            'company_id' => $this->company2->id,
            'status' => 'active',
        ]);
    }

    /**
     * اختبار أن المستخدم لا يرى موظفي شركة أخرى
     */
    public function test_user_cannot_see_employees_from_other_company()
    {
        $this->actingAs($this->user1);

        // يجب أن يرى الموظف من شركته
        $employees = Employee::all();
        $this->assertTrue($employees->contains($this->employee1));

        // لا يجب أن يرى الموظف من شركة أخرى
        $this->assertFalse($employees->contains($this->employee2));
    }

    /**
     * اختبار أن كل شركة لها بيانات منفصلة
     */
    public function test_companies_have_isolated_data()
    {
        $this->actingAs($this->user1);

        // عدد الموظفين في الشركة الأولى يجب أن يكون 1
        $company1Employees = Employee::all();
        $this->assertEquals(1, $company1Employees->count());
        $this->assertEquals($this->employee1->id, $company1Employees->first()->id);

        // التبديل إلى المستخدم الثاني
        $this->actingAs($this->user2);

        // عدد الموظفين في الشركة الثانية يجب أن يكون 1
        $company2Employees = Employee::all();
        $this->assertEquals(1, $company2Employees->count());
        $this->assertEquals($this->employee2->id, $company2Employees->first()->id);
    }

    /**
     * اختبار أن Super Admin يرى جميع البيانات
     */
    public function test_super_admin_can_see_all_companies()
    {
        $superAdmin = User::create([
            'name' => 'Super Admin',
            'email' => 'superadmin@test.com',
            'password' => bcrypt('password'),
            'role' => 'super_admin',
            'status' => 'active',
        ]);

        $this->actingAs($superAdmin);

        // يجب أن يرى جميع الشركات
        $companies = Company::all();
        $this->assertEquals(2, $companies->count());
    }

    /**
     * اختبار أن المستخدم لا يستطيع الوصول إلى بيانات شركة أخرى عبر الـ API
     */
    public function test_user_cannot_access_other_company_data_via_query()
    {
        $this->actingAs($this->user1);

        // محاولة الوصول إلى موظف من شركة أخرى
        $employee = Employee::find($this->employee2->id);

        // يجب أن يكون null لأن الـ scope يقيد الوصول
        $this->assertNull($employee);
    }

    /**
     * اختبار أن الـ scope يعمل مع العلاقات
     */
    public function test_scope_works_with_relationships()
    {
        $this->actingAs($this->user1);

        // الحصول على موظفي الشركة الأولى
        $employees = Employee::all();
        
        // يجب أن يكون هناك موظف واحد فقط
        $this->assertEquals(1, $employees->count());
        $this->assertEquals($this->employee1->id, $employees->first()->id);
    }
}
