<?php
// Generates a report of translation keys used in views and their presence in language files
$base = __DIR__ . '/../';
$viewsDir = $base . 'resources/views/';
$langDir = $base . 'resources/lang/';
$langs = array_values(array_filter(scandir($langDir), function($d) use ($langDir){return $d !== '.' && $d !== '..' && is_dir($langDir.$d);}));
$outDir = __DIR__ . '/reports/';
if (!is_dir($outDir)) mkdir($outDir, 0755, true);

function scanViewsForKeys($dir) {
    $keys = [];
    $it = new RecursiveIteratorIterator(new RecursiveDirectoryIterator($dir));
    foreach ($it as $f) {
        if ($f->isFile() && preg_match('/\.blade\.php$/', $f->getFilename())) {
            $contents = file_get_contents($f->getPathname());
            // find keys and capture line numbers
            $lines = preg_split('/\r?\n/', $contents);
            foreach ($lines as $lineno => $line) {
                // patterns: __('app.key'), __("app.key"), @lang('app.key'), trans('app.key')
                if (preg_match_all("/__\\(\\s*'app\.([a-z0-9_]+)'/i", $line, $m)) {
                    foreach ($m[1] as $k) $keys[$k][] = [$f->getPathname(), $lineno+1];
                }
                if (preg_match_all('/__\\(\\s*"app\.([a-z0-9_]+)"/i', $line, $m)) {
                    foreach ($m[1] as $k) $keys[$k][] = [$f->getPathname(), $lineno+1];
                }
                if (preg_match_all("/@lang\(\s*'app\.([a-z0-9_]+)'/i", $line, $m)) {
                    foreach ($m[1] as $k) $keys[$k][] = [$f->getPathname(), $lineno+1];
                }
                if (preg_match_all("/trans\(\s*'app\.([a-z0-9_]+)'/i", $line, $m)) {
                    foreach ($m[1] as $k) $keys[$k][] = [$f->getPathname(), $lineno+1];
                }
            }
        }
    }
    return $keys;
}

function loadLangArray($path) {
    if (!file_exists($path)) return null;
    $arr = include $path;
    if (!is_array($arr)) return null;
    return $arr;
}

$found = scanViewsForKeys($viewsDir);
ksort($found);
$report = [];
foreach ($found as $key => $locations) {
    $row = ['key'=>$key, 'used_in'=>[]];
    foreach ($locations as $loc) $row['used_in'][] = ['file'=>$loc[0], 'line'=>$loc[1]];
    // check languages
    foreach ($langs as $lang) {
        $langFile = $langDir . $lang . '/app.php';
        $arr = loadLangArray($langFile);
        $present = false;
        if (is_array($arr) && array_key_exists($key, $arr)) $present = true;
        $row['langs'][$lang] = $present;
    }
    $report[] = $row;
}

// write JSON
$jsonFile = $outDir . 'translation_report_' . date('YmdHis') . '.json';
file_put_contents($jsonFile, json_encode($report, JSON_PRETTY_PRINT|JSON_UNESCAPED_UNICODE));

// write CSV: key, present_en, present_ar, present_es, present_fr, occurrences_count, files
$csvFile = $outDir . 'translation_report_' . date('YmdHis') . '.csv';
$fh = fopen($csvFile,'w');
$headers = ['key'];
foreach ($langs as $lang) $headers[] = 'present_'.$lang;
$headers[] = 'occurrences';
$headers[] = 'files';
fputcsv($fh, $headers);
foreach ($report as $r) {
    $line = [$r['key']];
    foreach ($langs as $lang) $line[] = ($r['langs'][$lang] ? 'yes' : 'no');
    $line[] = count($r['used_in']);
    $files = array_unique(array_map(function($x){return $x['file'];}, $r['used_in']));
    $line[] = implode('|', $files);
    fputcsv($fh, $line);
}
fclose($fh);

echo "Report generated:\n$jsonFile\n$csvFile\n";
