<?php
// Add missing app.* translation keys to lang files (en/ar/es/fr)
// Usage: php scripts/add_missing_translations.php

$base = __DIR__ . '/../';
$viewsDir = $base . 'resources/views/';
$langDir = $base . 'resources/lang/';
$langs = ['en', 'ar', 'es', 'fr'];

function scanViews($dir) {
    $keys = [];
    $it = new RecursiveIteratorIterator(new RecursiveDirectoryIterator($dir));
    foreach ($it as $f) {
        if ($f->isFile() && preg_match('/\.blade\.php$/', $f->getFilename())) {
            $contents = file_get_contents($f->getPathname());
            // match __('app.key' or __('app.key', ...)
            if (preg_match_all("/__\(\s*'app\.([a-z0-9_]+)'/i", $contents, $m)) {
                foreach ($m[1] as $k) $keys[$k] = true;
            }
            // match @lang('app.key')
            if (preg_match_all("/@lang\(\s*'app\.([a-z0-9_]+)'/i", $contents, $m)) {
                foreach ($m[1] as $k) $keys[$k] = true;
            }
            // match trans('app.key')
            if (preg_match_all("/trans\(\s*'app\.([a-z0-9_]+)'/i", $contents, $m)) {
                foreach ($m[1] as $k) $keys[$k] = true;
            }
            // match __('app.key") with double quotes
            if (preg_match_all('/__\(\s*"app\.([a-z0-9_]+)"/i', $contents, $m)) {
                foreach ($m[1] as $k) $keys[$k] = true;
            }
        }
    }
    return array_keys($keys);
}

function loadLang($path) {
    if (!file_exists($path)) return [];
    $arr = include $path;
    if (!is_array($arr)) return [];
    return $arr;
}

function backupFile($path) {
    $bak = $path . '.bak.' . date('YmdHis');
    copy($path, $bak);
    return $bak;
}

$foundKeys = scanViews($viewsDir);
sort($foundKeys);
echo "Found " . count($foundKeys) . " unique keys in views.\n";

foreach ($langs as $lang) {
    $file = $langDir . $lang . '/app.php';
    $arr = loadLang($file);
    $existing = array_keys($arr);
    $missing = array_diff($foundKeys, $existing);
    if (empty($missing)) {
        echo "[$lang] No missing keys.\n";
        continue;
    }
    echo "[$lang] Missing: " . count($missing) . " keys.\n";
    // backup
    $bak = backupFile($file);
    echo "  backup -> $bak\n";
    // generate defaults
    foreach ($missing as $key) {
        $title = ucwords(str_replace('_', ' ', $key));
        if ($lang === 'en') {
            $arr[$key] = $title;
        } else {
            // copy English title as temporary placeholder
            $arr[$key] = $title;
        }
    }
    // sort keys for consistency
    ksort($arr);
    // write file
    $out = "<?php\n\nreturn [\n";
    foreach ($arr as $k => $v) {
        // escape single quotes
        $vv = str_replace("'", "\\'", $v);
        $out .= "    '" . $k . "' => '" . $vv . "',\n";
    }
    $out .= "];\n";
    file_put_contents($file, $out);
    echo "  updated $file\n";
}

echo "Done. Please clear caches (php artisan view:clear etc.) and verify pages.\n";
