<?php

use Illuminate\Foundation\Inspiring;
use Illuminate\Support\Facades\Artisan;

/*
|--------------------------------------------------------------------------
| Console Routes
|--------------------------------------------------------------------------
|
| This file is where you may define all of your Closure based console
| commands. Each Closure is bound to a command instance allowing a
| simple approach to interacting with each command's IO methods.
|
*/

Artisan::command('inspire', function () {
    $this->comment(Inspiring::quote());
})->purpose('Display an inspiring quote');

Artisan::command('debug:leaves {employee_id} {year} {month}', function ($employee_id, $year, $month) {
    $employee = \App\Models\Employee::findOrFail($employee_id);
    
    $this->info("=== تصحيح حسابات الإجازات ===");
    $this->info("الموظف: {$employee->full_name} (ID: {$employee->id})");
    $this->info("الشهر: {$year}/{$month}");
    $this->newLine();
    
    // 1. فحص بيانات الموظف
    $this->info("📋 معلومات الموظف:");
    $this->line("  تاريخ التعيين: {$employee->hire_date}");
    $this->line("  الإجازات الشهرية المسموحة: {$employee->monthly_leave_days_allowed} يوم");
    $this->line("  الراتب الإجمالي: " . $employee->getTotalSalaryWithComponentsAttribute());
    $this->newLine();
    
    // 2. فحص الإجازات المعتمدة
    $this->info("📅 الإجازات المعتمدة في الشهر:");
    $startOfMonth = \Carbon\Carbon::create($year, $month, 1)->startOfMonth();
    $endOfMonth = (clone $startOfMonth)->endOfMonth();
    $leaves = $employee->leaves()
        ->where('status', 'approved')
        ->ofCodes(['annual', 'emergency', 'sick'])
        ->where(function ($q) use ($startOfMonth, $endOfMonth) {
            $q->where(function ($qq) use ($startOfMonth, $endOfMonth) {
                $qq->where('start_date', '<=', $endOfMonth)
                    ->where('end_date', '>=', $startOfMonth);
            });
        })
        ->get();
    
    if ($leaves->isEmpty()) {
        $this->line("  ❌ لا توجد إجازات معتمدة في هذا الشهر");
    } else {
        foreach ($leaves as $leave) {
            $this->line("  ✅ {$leave->leave_type->name}: من {$leave->start_date} إلى {$leave->end_date} ({$leave->days_count} يوم)");
        }
    }
    $this->newLine();
    
    // 3. فحص الأيام الرسمية
    $this->info("🏛️ الأيام الرسمية في الشهر:");
    $holidays = \App\Models\Holiday::whereBetween('date', [$startOfMonth->format('Y-m-d'), $endOfMonth->format('Y-m-d')])
        ->get();
    
    if ($holidays->isEmpty()) {
        $this->line("  ❌ لا توجد أيام رسمية في هذا الشهر");
    } else {
        foreach ($holidays as $holiday) {
            $this->line("  🎉 {$holiday->name}: {$holiday->date}");
        }
    }
    $this->newLine();
    
    // 4. اختبار الدالة getUsedLeaveDaysInMonth
    $this->info("🔢 أيام الإجازة المستخدمة:");
    $usedDays = $employee->getUsedLeaveDaysInMonth($year, $month);
    $this->line("  المجموع: {$usedDays} يوم");
    $this->newLine();
    
    // 5. عرض جميع التفاصيل الشهرية من calculateLeaveDeductionDetailsNew
    $this->info("📊 جدول التراكم الشهري:");
    $details = $employee->calculateLeaveDeductionDetailsNew($year, $month);
    
    if ($details->isEmpty()) {
        $this->line("  ❌ لا توجد بيانات في الجدول");
    } else {
        foreach ($details as $detail) {
            $this->line("  📌 {$detail['month_name']}");
            $this->line("    • الممنوح: {$detail['granted_this_month']} أيام");
            $this->line("    • المستخدم: {$detail['used_this_month']} أيام");
            $this->line("    • الزائد: {$detail['excess_days']} أيام");
            $this->line("    • الخصم المالي: {$detail['leave_deduction_amount']} ريال");
            $this->line("    • الرصيد (قبل): {$detail['balance_before_deduction']} أيام");
            $this->line("    • الرصيد (بعد): {$detail['balance_after_deduction']} أيام");
            if (isset($detail['debug_info'])) {
                $this->line("    🔍 تصحيح: available={$detail['debug_info']['available']}, used={$detail['debug_info']['used']}, excess={$detail['debug_info']['excess']}");
            }
        }
    }
})->describe('تصحيح وفحص حسابات الإجازات');
