{{-- resources/views/payroll/create-form.blade.php --}}
@extends('layouts.app')

@section('content')
<div class="container-fluid">
    <div class="row">
        <div class="col-12">
            <div class="page-title-box d-flex align-items-center justify-content-between">
                <h4 class="mb-0">{{ __('payroll.create.page_title', ['name' => $employee->name]) }}</h4>
                <div class="page-title-right">
                    <ol class="breadcrumb m-0">
                        <li class="breadcrumb-item"><a href="{{ route('dashboard.index') }}">{{ __('payroll.index.breadcrumb_home') }}</a></li>
                        <li class="breadcrumb-item"><a href="{{ route('payroll.index') }}">{{ __('payroll.index.breadcrumb_payroll') }}</a></li>
                        <li class="breadcrumb-item active">{{ __('payroll.create.breadcrumb_create') }}</li>
                    </ol>
                </div>
            </div>
        </div>
        <div class="col-md-4">
            <div class="mb-3">
                <label class="form-label">{{ __('payroll.create.pending_installments', [], null) ?: 'أقساط القرض المعلقة لهذا الشهر' }}</label>
                <div class="card p-2" style="max-height:200px; overflow:auto;">
                    @if(!empty($pendingInstallments) && $pendingInstallments->isNotEmpty())
                        @foreach($pendingInstallments as $inst)
                            <div class="form-check">
                                <input class="form-check-input installment-checkbox" type="checkbox" value="{{ $inst->id }}" id="inst_{{ $inst->id }}" name="installment_ids[]">
                                <label class="form-check-label" for="inst_{{ $inst->id }}">#{{ $inst->id }} — {{ number_format($inst->amount,2) }} {{ get_currency_code() }} — {{ \Carbon\Carbon::parse($inst->due_date)->toDateString() }} ({{ $inst->loan?->loan_type ?? '' }})</label>
                            </div>
                        @endforeach
                    @else
                        <div class="text-muted">{{ __('payroll.create.no_pending_installments', [], null) ?: 'لا توجد أقساط معلقة لهذا الشهر.' }}</div>
                    @endif
                </div>
            </div>
        </div>
    </div>

    <div class="row mb-3">
        <div class="col-12">
            @if (session()->has('success'))
                <div class="alert alert-success alert-dismissible fade show" role="alert">
                    {{ session('success') }}
                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                </div>
            @endif
            @if (session()->has('error'))
                <div class="alert alert-danger alert-dismissible fade show" role="alert">
                    {{ session('error') }}
                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                </div>
            @endif
        </div>
    </div>

    <div class="row mb-4">
        <div class="col-md-6">
            <div class="card">
                <div class="card-body">
                    <h5 class="card-title">{{ __('payroll.create.employee_info') }}</h5>
                    <div class="row">
                        <div class="col-md-6">
                            <p><strong>{{ __('payroll.create.employee_id') }}:</strong> {{ $employee->employee_id }}</p>
                            <p><strong>{{ __('payroll.create.department') }}:</strong> {{ $employee->department?->name ?? __('payroll.create.not_specified') }}</p>
                            <p><strong>{{ __('payroll.create.position') }}:</strong> {{ $employee->position?->name ?? __('payroll.create.not_specified') }}</p>
                        </div>
                        <div class="col-md-6">
                            <p><strong>{{ __('payroll.create.basic_salary') }}:</strong> {{ number_format($employee->salary, 2) }} {{ get_currency_code() }}</p>
                            <p><strong>{{ __('payroll.create.incentives') }}:</strong> {{ number_format($employee->incentives, 2) }} {{ get_currency_code() }}</p>
                            @php $componentsTotal = $employee->salaryComponents->reduce(function($carry, $c) { return $carry + ($c->pivot->value ?? 0); }, 0); @endphp
                            <div class="mb-2">
                                <label class="form-label"><strong>{{ __('payroll.create.additional_components_total', [], null) ?: 'مجموع المكونات الإضافية' }}:</strong></label>
                                <input type="text" id="components_total" class="form-control" value="{{ number_format($componentsTotal, 2) }} {{ get_currency_code() }}" readonly>
                            </div>
                            <p><strong>{{ __('payroll.create.leave_balance') }}:</strong> {{ $employee->getAvailableBalanceAttribute() }} {{ __('payroll.create.days', [], null) ?: 'يوم' }}</p>
                        </div>
                    </div>
                    @if($employee->salaryComponents->isNotEmpty())
                        <div class="mt-3 border-top pt-3">
                            <h6 class="mb-2 text-primary">{{ __('payroll.create.components_heading') }}</h6>
                            <div class="table-responsive">
                                <table class="table table-sm table-bordered">
                                    <thead>
                                        <tr>
                                            <th scope="col">{{ __('payroll.create.component_name') }}</th>
                                            <th scope="col">{{ __('payroll.create.component_value') }} ({{ get_currency_code() }})</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        @foreach($employee->salaryComponents as $comp)
                                            <tr>
                                                <td>{{ $comp->name }}</td>
                                                <td class="text-end">{{ number_format($comp->pivot->value, 2) }}</td>
                                            </tr>
                                        @endforeach
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    @endif
                </div>
            </div>
        </div>
    </div>

    <div class="row">
        <div class="col-12">
            <div class="card">
                <div class="card-body">
                    <h5 class="card-title">{{ __('payroll.create.form_title') }}</h5>
                    <form action="{{ route('payroll.store') }}" method="POST" id="payrollForm">
                        @csrf
                        <input type="hidden" name="employee_id" value="{{ $employee->id }}">
                        @if($employee->salaryComponents->isNotEmpty())
                            @foreach($employee->salaryComponents as $comp)
                                <input type="hidden" name="salary_component_ids[]" value="{{ $comp->id }}">
                                <input type="hidden" name="salary_component_names[]" value="{{ $comp->name }}">
                                <input type="hidden" name="salary_component_values[]" value="{{ $comp->pivot->value ?? 0 }}">
                            @endforeach
                        @endif

                        <div class="row">
                            <div class="col-md-4">
                                <div class="mb-3">
                                    <label for="year" class="form-label">{{ __('payroll.create.year') }} <span class="text-danger">*</span></label>
                                    <select class="form-control @error('year') is-invalid @enderror" id="year" name="year" required>
                                        @for($y = now()->year - 1; $y <= now()->year + 25; $y++)
                                            <option value="{{ $y }}" {{ old('year', $selectedYear) == $y ? 'selected' : '' }}>{{ $y }}</option>
                                        @endfor
                                    </select>
                                    @error('year') <div class="invalid-feedback">{{ $message }}</div> @enderror
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="mb-3">
                                    <label for="month" class="form-label">{{ __('payroll.create.month') }} <span class="text-danger">*</span></label>
                                    <select class="form-control @error('month') is-invalid @enderror" id="month" name="month" required>
                                        <option value="">اختر الشهر</option>
                                        @for($m = 1; $m <= 12; $m++)
                                            <option value="{{ $m }}" {{ old('month', $selectedMonth) == $m ? 'selected' : '' }}>{{ \Carbon\Carbon::create()->month($m)->translatedFormat('F') }}</option>
                                        @endfor
                                    </select>
                                    @error('month') <div class="invalid-feedback">{{ $message }}</div> @enderror
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="mb-3">
                                    <label for="basic_salary" class="form-label">{{ __('payroll.create.basic_salary') }} <span class="text-danger">*</span></label>
                                    <input type="number" class="form-control @error('basic_salary') is-invalid @enderror" id="basic_salary" name="basic_salary" value="{{ old('basic_salary', $defaultBasicSalary ?? $employee->salary) }}" step="0.01" min="0" required>
                                    @error('basic_salary') <div class="invalid-feedback">{{ $message }}</div> @enderror
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-4">
                                <div class="mb-3">
                                    <label for="incentives" class="form-label">{{ __('payroll.create.incentives') }}</label>
                                    <input type="number" class="form-control @error('incentives') is-invalid @enderror" id="incentives" name="incentives" value="{{ old('incentives', $defaultIncentives ?? $employee->incentives) }}" step="0.01" min="0">
                                    @error('incentives') <div class="invalid-feedback">{{ $message }}</div> @enderror
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="mb-3">
                                    <label for="total_salary" class="form-label">{{ __('payroll.create.total_salary', [], null) ?: 'Total Salary' }}</label>
                                    <input type="text" class="form-control" id="total_salary" name="total_salary" readonly>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="mb-3">
                                    <label for="net_salary" class="form-label">{{ __('payroll.create.net_salary') }}</label>
                                    <input type="text" class="form-control" id="net_salary" name="net_salary" readonly>
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-12 mb-3">
                                <h5>{{ __('payroll.create.components_heading', [], null) ?: 'Additional Components' }}</h5>
                                <div id="componentsContainer">
                                    @php $componentsArray = []; if (isset($defaultComponents) && is_array($defaultComponents) && count($defaultComponents) > 0) { $componentsArray = $defaultComponents; } else { for ($i = 1; $i <= 7; $i++) { $name = $employee->{"component_name_$i"} ?? null; $value = $employee->{"component_$i"} ?? 0; if (!empty($name) && $value > 0) { $componentsArray[$name] = $value; } } } @endphp
                                    @for ($i = 1; $i <= 7; $i++)
                                    <div class="row mb-2 component-row align-items-center">
                                        <div class="col-md-5">
                                            @php $componentName = ''; if (isset($defaultComponents) && is_array($defaultComponents) && count($defaultComponents) > 0) { $componentNames = array_keys($defaultComponents); $componentName = $componentNames[$i-1] ?? ($employee->{"component_name_$i"} ?? ''); } else { $componentName = old('salary_component_names.' . ($i-1), $employee->{"component_name_$i"}); } @endphp
                                            <input type="text" class="form-control component-name" name="salary_component_names[]" placeholder="Component name" value="{{ $componentName }}">
                                        </div>
                                        <div class="col-md-5">
                                            @php $componentValue = ''; if (isset($defaultComponents) && is_array($defaultComponents) && count($defaultComponents) > 0) { $componentValues = array_values($defaultComponents); $componentValue = $componentValues[$i-1] ?? ($employee->{"component_$i"} ?? 0); } else { $componentValue = old('salary_component_values.' . ($i-1), $employee->{"component_$i"}); } @endphp
                                            <input type="number" class="form-control component-value" name="salary_component_values[]" placeholder="Value" step="0.01" min="0" value="{{ $componentValue }}">
                                        </div>
                                        <div class="col-md-2">
                                            <button type="button" class="btn btn-sm btn-outline-danger remove-component">{{ __('payroll.create.remove_component') }}</button>
                                        </div>
                                    </div>
                                    @endfor
                                </div>
                                <button type="button" id="add-component-btn" class="btn btn-info btn-sm mb-3">+ Add</button>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-4">
                                <div class="mb-3">
                                    <label for="active_deductions" class="form-label">{{ __('payroll.create.active_deductions') }}</label>
                                    <input type="number" class="form-control" id="active_deductions" name="active_deductions" value="{{ number_format($activeDeductions ?? 0, 2) }}" step="0.01" min="0" readonly>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="mb-3">
                                    <label for="loan_installments" class="form-label">{{ __('payroll.create.loan_installments') }}</label>
                                    <input type="number" class="form-control" id="loan_installments" name="loan_installments" value="{{ number_format($loanInstallments ?? 0, 2) }}" step="0.01" min="0" readonly>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="mb-3">
                                    <label for="leave_deduction" class="form-label">{{ __('payroll.create.leave_deduction') }}</label>
                                    <input type="number" class="form-control" id="leave_deduction" name="leave_deduction" value="{{ number_format($leaveDeduction ?? 0, 2) }}" step="0.01" min="0" readonly>
                                </div>
                            </div>
                        </div>

                        <div class="row mb-3">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label class="form-label">أيام الحضور المدفوعة</label>
                                    <input type="text" id="paid_off_days" class="form-control" value="{{ isset($paidOffDays) ? number_format($paidOffDays, 0) : 0 }}" readonly>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="paid_off_amount" class="form-label">قيمة أيام الحضور المدفوعة</label>
                                    <div class="input-group">
                                        <input type="number" step="0.01" class="form-control" id="paid_off_amount" name="paid_off_amount" value="{{ old('paid_off_amount', $paidOffAmount ?? $employee->default_paid_off_amount ?? 0.00) }}">
                                        <span class="input-group-text">{{ get_currency_code() }}</span>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-4">
                                <div class="mb-3">
                                    <label for="balance_before_deduction" class="form-label">الرصيد قبل الخصم</label>
                                    <input type="text" class="form-control" id="balance_before_deduction" name="balance_before_deduction" value="{{ number_format($balanceBeforeDeduction ?? 0, 2) }} يوم" readonly>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="mb-3">
                                    <label for="balance_after_deduction" class="form-label">الرصيد بعد الخصم</label>
                                    <input type="text" class="form-control" id="balance_after_deduction" name="balance_after_deduction" value="{{ number_format($balanceAfterDeduction ?? 0, 2) }} يوم" readonly>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="mb-3">
                                    <label for="excess_days" class="form-label">الأيام الزائدة</label>
                                    <input type="text" class="form-control" id="excess_days" name="excess_days" value="{{ number_format($excessDays ?? 0, 2) }} يوم" readonly>
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-12">
                                <div class="mb-3">
                                    <label for="notes" class="form-label">ملاحظات</label>
                                    <textarea class="form-control @error('notes') is-invalid @enderror" id="notes" name="notes" rows="3">{{ old('notes') }}</textarea>
                                    @error('notes') <div class="invalid-feedback">{{ $message }}</div> @enderror
                                </div>
                            </div>
                        </div>

                        <div class="row mb-3">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label class="form-label">ساعات الأوفر تايم</label>
                                    <input type="text" class="form-control" id="attendance_overtime_hours" value="{{ number_format($attendanceOvertimeHours ?? 0, 2) }}" readonly>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label class="form-label">قيمة الأوفر تايم</label>
                                    <div class="input-group">
                                        <input type="text" class="form-control" id="attendance_overtime_amount" value="{{ number_format($overtimeAmount ?? 0, 2) }}" readonly>
                                        <span class="input-group-text">{{ get_currency_code() }}</span>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="row mb-3">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label class="form-label">ساعات النقص</label>
                                    <input type="text" class="form-control" id="attendance_underworked_hours" value="{{ number_format($attendanceUnderworkedHours ?? 0, 2) }}" readonly>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label class="form-label">قيمة خصم النقص</label>
                                    <div class="input-group">
                                        <input type="text" class="form-control" id="attendance_underworked_amount" value="{{ number_format($underworkedAmount ?? 0, 2) }}" readonly>
                                        <span class="input-group-text">{{ get_currency_code() }}</span>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="d-flex gap-2 flex-wrap">
                            <a href="{{ route('payroll.index') }}" class="btn btn-secondary">{{ __('payroll.index.cancel') }}</a>
                            <button type="button" class="btn btn-info" onclick="printPayrollDetails()">
                                🖨️ طباعة تفاصيل الراتب
                            </button>
                            <button type="submit" class="btn btn-success">{{ __('payroll.create.submit') }}</button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>
</div>

@push('scripts')
<script>
document.addEventListener('DOMContentLoaded', function () {
    const dayLabel = "{{ __('payroll.create.day') }}";
    const yearSelect = document.getElementById('year');
    const monthSelect = document.getElementById('month');
    const currencyName = "{{ get_currency_code() }}";

    function parseFormattedNumber(value) {
        if (!value || typeof value !== 'string') { return parseFloat(value) || 0; }
        const cleanValue = value.replace(/[^\d.-]/g, '');
        const parsed = parseFloat(cleanValue);
        return isNaN(parsed) ? 0 : parsed;
    }

    function calculateTotalSalary() {
        const basic = parseFormattedNumber(document.getElementById('basic_salary')?.value);
        const incentives = parseFormattedNumber(document.getElementById('incentives')?.value);
        const paidOffAmount = parseFormattedNumber(document.getElementById('paid_off_amount')?.value);
        const overtimeAmount = parseFormattedNumber(document.getElementById('attendance_overtime_amount')?.value);
        let componentsTotal = 0;
        document.querySelectorAll('input[name="salary_component_values[]"]').forEach(input => {
            componentsTotal += parseFormattedNumber(input.value);
        });
        // إجمالي الراتب = الأساسي + الحوافز + مكونات الراتب + قيمة أيام الحضور المدفوعة + قيمة الأوفرتايم
        return basic + incentives + componentsTotal + paidOffAmount + overtimeAmount;
    }

    function calculateNetSalary() {
        let componentsTotal = 0;
        document.querySelectorAll('input[name="salary_component_values[]"]').forEach(input => {
            componentsTotal += parseFormattedNumber(input.value);
        });
        const componentsEl = document.getElementById('components_total');
        if (componentsEl) { componentsEl.value = componentsTotal.toFixed(2) + ' ' + currencyName; }
        const total = calculateTotalSalary();
        const getNumericValue = (id) => {
            const el = document.getElementById(id);
            if (!el) return 0;
            const val = el.value.replace('جاري التحميل...', '0').replace(dayLabel, '').trim();
            return parseFormattedNumber(val);
        };
        const deductions = getNumericValue('active_deductions');
        const loans = getNumericValue('loan_installments');
        const leaveDed = getNumericValue('leave_deduction');
        const underworkedDedEl = document.getElementById('attendance_underworked_amount');
        const underworkedDed = underworkedDedEl ? parseFormattedNumber(underworkedDedEl.value) : 0;
        const net = total - deductions - loans - leaveDed - underworkedDed;
        document.getElementById('total_salary').value = total.toFixed(2);
        document.getElementById('net_salary').value = net.toFixed(2);
    }

    function fetchCalculations() {
        const year = yearSelect?.value;
        const month = monthSelect?.value;
        if (!year || !month) { console.warn('يجب تحديد السنة والشهر'); return; }
        const loaderFields = ['active_deductions', 'loan_installments', 'leave_deduction', 'balance_before_deduction', 'balance_after_deduction', 'excess_days'];
        loaderFields.forEach(id => {
            const el = document.getElementById(id);
            if (el) el.value = 'جاري التحميل...';
        });
        const selectedInstallments = Array.from(document.querySelectorAll('.installment-checkbox:checked')).map(i => i.value);
        let url = `{{ route('payroll.getCalculations') }}?employee_id={{ $employee->id }}&month=${encodeURIComponent(month)}&year=${encodeURIComponent(year)}`;
        if (selectedInstallments.length > 0) {
            selectedInstallments.forEach(id => {
                url += `&installment_ids[]=${encodeURIComponent(id)}`;
            });
        }
        fetch(url, { method: 'GET', headers: { 'X-CSRF-TOKEN': '{{ csrf_token() }}', 'Accept': 'application/json' } })
        .then(response => response.ok ? response.json() : Promise.reject(new Error('Error')))
        .then(data => {
            if (data.success) {
                if (data.calculations.basic_salary !== undefined) {
                    const basicSalaryInput = document.getElementById('basic_salary');
                    if (basicSalaryInput) { basicSalaryInput.value = parseFormattedNumber(data.calculations.basic_salary).toFixed(2); }
                }
                if (data.calculations.incentives !== undefined) {
                    const incentivesInput = document.getElementById('incentives');
                    if (incentivesInput) { incentivesInput.value = parseFormattedNumber(data.calculations.incentives).toFixed(2); }
                }
                const componentsContainer = document.getElementById('componentsContainer');
                if (componentsContainer) {
                    const componentRows = componentsContainer.querySelectorAll('.component-row');
                    componentRows.forEach(row => {
                        const nameInput = row.querySelector('.component-name');
                        const valueInput = row.querySelector('.component-value');
                        if (nameInput) nameInput.value = '';
                        if (valueInput) valueInput.value = '';
                    });
                    if (data.calculations.components && typeof data.calculations.components === 'object') {
                        let index = 0;
                        for (const [name, value] of Object.entries(data.calculations.components)) {
                            if (index < componentRows.length && name && value > 0) {
                                const row = componentRows[index];
                                const nameInput = row.querySelector('.component-name');
                                const valueInput = row.querySelector('.component-value');
                                if (nameInput) nameInput.value = name;
                                if (valueInput) valueInput.value = parseFormattedNumber(value).toFixed(2);
                                index++;
                            }
                        }
                    }
                }
                document.getElementById('active_deductions').value = parseFormattedNumber(data.calculations.active_deductions).toFixed(2);
                document.getElementById('loan_installments').value = parseFormattedNumber(data.calculations.loan_installments).toFixed(2);
                document.getElementById('leave_deduction').value = parseFormattedNumber(data.calculations.leave_deduction).toFixed(2);
                document.getElementById('balance_before_deduction').value = (data.calculations.balance_before_deduction || 0).toFixed(2) + ' ' + dayLabel;
                document.getElementById('balance_after_deduction').value = (data.calculations.balance_after_deduction || 0).toFixed(2) + ' ' + dayLabel;
                document.getElementById('excess_days').value = (data.calculations.excess_days || 0).toFixed(2) + ' ' + dayLabel;
                if (document.getElementById('attendance_underworked_hours')) {
                    document.getElementById('attendance_underworked_hours').value = (data.calculations.attendance_underworked_hours || 0).toFixed(2);
                }
                if (document.getElementById('attendance_underworked_amount')) {
                    document.getElementById('attendance_underworked_amount').value = (data.calculations.attendance_underworked_deduction || 0).toFixed(2);
                }
                calculateNetSalary();
            }
        })
        .catch(error => {
            const reset = { active_deductions: '0.00', loan_installments: '0.00', leave_deduction: '0.00', balance_before_deduction: '0.00 ' + dayLabel, balance_after_deduction: '0.00 ' + dayLabel, excess_days: '0.00 ' + dayLabel };
            Object.entries(reset).forEach(([id, val]) => {
                const el = document.getElementById(id);
                if (el) el.value = val;
            });
            calculateNetSalary();
        });
    }

    document.addEventListener('change', function(e) {
        if (e.target.classList && e.target.classList.contains('installment-checkbox')) { fetchCalculations(); }
    });

    const inputs = ['basic_salary', 'incentives', 'paid_off_amount'];
    inputs.forEach(id => {
        const el = document.getElementById(id);
        if (el) el.addEventListener('input', calculateNetSalary);
    });

    document.getElementById('componentsContainer').addEventListener('input', function(e) {
        if (e.target.classList.contains('component-value')) { calculateNetSalary(); }
    });

    yearSelect?.addEventListener('change', function() { fetchCalculations(); });
    monthSelect?.addEventListener('change', function() { fetchCalculations(); });

    if (yearSelect?.value && monthSelect?.value) {
        calculateNetSalary();
        setTimeout(fetchCalculations, 500);
    } else {
        calculateNetSalary();
    }

    const addComponentBtn = document.getElementById('add-component-btn');
    if (addComponentBtn) {
        addComponentBtn.addEventListener('click', function() {
            const container = document.getElementById('componentsContainer');
            if (container.children.length >= 7) { alert('لا يمكن إضافة أكثر من 7 مكونات.'); return; }
            const newRow = document.createElement('div');
            newRow.className = 'row mb-2 component-row align-items-center';
            newRow.innerHTML = `<div class="col-md-5"><input type="text" class="form-control component-name" name="salary_component_names[]" placeholder="اسم المكون"></div><div class="col-md-5"><input type="number" class="form-control component-value" name="salary_component_values[]" placeholder="القيمة" step="0.01" min="0"></div><div class="col-md-2"><button type="button" class="btn btn-sm btn-outline-danger remove-component">حذف</button></div>`;
            container.appendChild(newRow);
            newRow.querySelector('.remove-component').addEventListener('click', function() {
                container.removeChild(newRow);
                calculateNetSalary();
            });
        });
    }

    document.addEventListener('click', function(e) {
        if (e.target.classList.contains('remove-component')) {
            e.target.closest('.component-row').remove();
            calculateNetSalary();
        }
    });

    window.printPayrollDetails = function () {
        const year = document.getElementById('year')?.value || '';
        const month = document.getElementById('month')?.value || '';
        const employeeName = "{{ $employee->name }}";
        const employeeId = "{{ $employee->id }}";
        const basic = parseFormattedNumber(document.getElementById('basic_salary')?.value);
        const incentives = parseFormattedNumber(document.getElementById('incentives')?.value);
        const total = parseFormattedNumber(document.getElementById('total_salary')?.value);
        const net = parseFormattedNumber(document.getElementById('net_salary')?.value);
        const deductions = parseFormattedNumber(document.getElementById('active_deductions')?.value);
        const loans = parseFormattedNumber(document.getElementById('loan_installments')?.value);
        const leaveDed = parseFormattedNumber(document.getElementById('leave_deduction')?.value);
        const overtimeHours = document.getElementById('attendance_overtime_hours')?.value || '0';
        const overtimeAmount = parseFormattedNumber(document.getElementById('attendance_overtime_amount')?.value);
        const underworkedHours = document.getElementById('attendance_underworked_hours')?.value || '0';
        const underworkedAmount = parseFormattedNumber(document.getElementById('attendance_underworked_amount')?.value);
        const paidOffDays = document.getElementById('paid_off_days')?.value || '0';
        const paidOffAmount = parseFormattedNumber(document.getElementById('paid_off_amount')?.value);
        const balanceBefore = document.getElementById('balance_before_deduction')?.value || '0';
        const balanceAfter = document.getElementById('balance_after_deduction')?.value || '0';
        const excessDays = document.getElementById('excess_days')?.value || '0';
        const notes = (document.getElementById('notes')?.value || '').trim();
        let componentsRows = '';
        document.querySelectorAll('.component-row').forEach(row => {
            const name = row.querySelector('.component-name')?.value;
            const value = parseFormattedNumber(row.querySelector('.component-value')?.value);
            if (name && value > 0) {
                componentsRows += '<tr><td>' + name + '</td><td class="text-left">' + value.toFixed(2) + ' ' + currencyName + '</td></tr>';
            }
        });
        const w = window.open('', '_blank');
        if (!w) return;
        w.document.write('<!DOCTYPE html><html dir="rtl" lang="ar"><head><meta charset="utf-8"><title>تفاصيل الراتب</title><style>body{margin:0;padding:20px;direction:rtl;font-family:Arial,sans-serif}table{width:100%;margin-bottom:10px;border-collapse:collapse}th,td{padding:8px;border:1px solid #ddd;text-align:right}.text-left{text-align:left}h2,h3{margin:12px 0}</style></head><body><h2 style="text-align:center">تفاصيل الراتب الشهري</h2><table><tr><td><strong>اسم الموظف:</strong></td><td>' + employeeName + '</td></tr><tr><td><strong>الرقم الوظيفي:</strong></td><td>' + employeeId + '</td></tr><tr><td><strong>الشهر:</strong></td><td>' + month + '/' + year + '</td></tr></table><h3>تفاصيل الراتب:</h3><table><tr style="background:#f5f5f5"><th>الراتب الأساسي</th><th class="text-left">' + basic.toFixed(2) + ' ' + currencyName + '</th></tr><tr><td>الحوافز</td><td class="text-left">' + incentives.toFixed(2) + ' ' + currencyName + '</td></tr>' + componentsRows + '<tr style="background:#f5f5f5"><td><strong>الإجمالي</strong></td><td class="text-left"><strong>' + total.toFixed(2) + ' ' + currencyName + '</strong></td></tr></table><h3>الخصومات:</h3><table><tr><td>الخصومات النشطة</td><td class="text-left">' + deductions.toFixed(2) + ' ' + currencyName + '</td></tr><tr><td>أقساط القروض</td><td class="text-left">' + loans.toFixed(2) + ' ' + currencyName + '</td></tr><tr><td>خصم الإجازات</td><td class="text-left">' + leaveDed.toFixed(2) + ' ' + currencyName + '</td></tr><tr><td>خصم نقصان الساعات (' + underworkedHours + ' ساعة)</td><td class="text-left">' + underworkedAmount.toFixed(2) + ' ' + currencyName + '</td></tr></table><h3>الإضافات:</h3><table><tr><td>ساعات الأوفر تايم</td><td class="text-left">' + overtimeHours + ' ساعة</td></tr><tr><td>قيمة الأوفر تايم</td><td class="text-left">' + overtimeAmount.toFixed(2) + ' ' + currencyName + '</td></tr></table><h3>الحضور المدفوع:</h3><table><tr><td>أيام الحضور المدفوعة</td><td class="text-left">' + paidOffDays + '</td></tr><tr><td>قيمة أيام الحضور المدفوعة</td><td class="text-left">' + paidOffAmount.toFixed(2) + ' ' + currencyName + '</td></tr></table><h3>الأرصدة:</h3><table><tr><td>الرصيد قبل الخصم</td><td class="text-left">' + balanceBefore + '</td></tr><tr><td>الرصيد بعد الخصم</td><td class="text-left">' + balanceAfter + '</td></tr><tr><td>الأيام الزائدة</td><td class="text-left">' + excessDays + '</td></tr></table><h3>ملاحظات:</h3><div style="min-height:40px;border:1px solid #ddd;padding:8px">' + (notes || '—') + '</div><table style="margin-top:20px"><tr style="background:#e8f5e8;font-size:18px"><td><strong>الراتب الصافي</strong></td><td class="text-left"><strong>' + net.toFixed(2) + ' ' + currencyName + '</strong></td></tr></table></body></html>');
        w.document.close();
        w.focus();
        w.print();
    };
});
</script>
@endpush
@endsection