@extends('layouts.app')

@section('content')
<div class="container-fluid">
    <div class="row">
        <div class="col-12">
            <div class="page-title-box d-flex align-items-center justify-content-between">
                <h4 class="mb-0">{{ __('loans.form.create_title') }}</h4>
                <div class="page-title-right">
                    <ol class="breadcrumb m-0">
                        <li class="breadcrumb-item"><a href="{{ route('dashboard.index') }}">{{ __('loans.breadcrumb_home') }}</a></li>
                        <li class="breadcrumb-item"><a href="{{ route('loans.index') }}">{{ __('loans.breadcrumb_loans') }}</a></li>
                        <li class="breadcrumb-item active">{{ __('loans.form.create_title') }}</li>
                    </ol>
                </div>
            </div>
        </div>
    </div>

    <div class="row">
        <div class="col-12">
            <div class="card">
                <div class="card-body">
                    <h4 class="card-title mb-4">{{ __('loans.form.create_title') }}</h4>

                    <form action="{{ route('loans.store') }}" method="POST">
                        @csrf
                        <div class="row">
                            @if(auth()->user()->hasRole('admin') || auth()->user()->hasRole('hr') || auth()->user()->hasRole('finance'))
                                <div class="col-md-6 mb-3">
                                    <label for="employee_id" class="form-label">{{ __('loans.form.label_employee') }} <span class="text-danger">*</span></label>
                                    <select class="form-control select2" id="employee_id" name="employee_id" required>
                                        <option value="">{{ __('loans.not_specified') }}</option>
                                        @foreach($employees as $employee)
                                            <option value="{{ $employee->id }}" {{ old('employee_id') == $employee->id ? 'selected' : '' }}>
                                                {{ $employee->name }} - {{ $employee->employee_id }}
                                            </option>
                                        @endforeach
                                    </select>
                                    @error('employee_id')
                                        <div class="text-danger mt-1">{{ $message }}</div>
                                    @enderror
                                </div>
                            @else
                                <!-- employee user: hide select and show readonly -->
                                <input type="hidden" name="employee_id" value="{{ auth()->user()->employee->id ?? '' }}">
                                <div class="col-md-6 mb-3">
                                    <label class="form-label">{{ __('loans.form.label_employee') }}</label>
                                    <input type="text" class="form-control" value="{{ optional(auth()->user()->employee)->name ?? __('loans.not_specified') }}" readonly>
                                </div>
                            @endif

                            <div class="col-md-6 mb-3">
                                <label for="loan_type" class="form-label">{{ __('loans.form.label_type') }} <span class="text-danger">*</span></label>
                                <select class="form-control" id="loan_type" name="loan_type" required>
                                    <option value="">{{ __('loans.not_specified') }}</option>
                                    <option value="personal" {{ old('loan_type') == 'personal' ? 'selected' : '' }}>{{ __('loans.type_labels.personal') }}</option>
                                    <option value="housing" {{ old('loan_type') == 'housing' ? 'selected' : '' }}>{{ __('loans.type_labels.housing') }}</option>
                                    <option value="car" {{ old('loan_type') == 'car' ? 'selected' : '' }}>{{ __('loans.type_labels.car') }}</option>
                                    <option value="emergency" {{ old('loan_type') == 'emergency' ? 'selected' : '' }}>{{ __('loans.type_labels.emergency') }}</option>
                                    <option value="education" {{ old('loan_type') == 'education' ? 'selected' : '' }}>{{ __('loans.type_labels.education') }}</option>
                                </select>
                                @error('loan_type')
                                    <div class="text-danger mt-1">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label for="amount" class="form-label">{{ __('loans.form.label_amount') }} <span class="text-danger">*</span></label>
                                <div class="input-group">
                                    <input type="number" class="form-control" id="amount" name="amount" value="{{ old('amount') }}" step="0.01" min="1" required>
                                    <span class="input-group-text">{{ get_currency_code() }}</span>
                                </div>
                                @error('amount')
                                    <div class="text-danger mt-1">{{ $message }}</div>
                                @enderror
                            </div>
                            <div class="col-md-6 mb-3">
                                <label for="installment_count" class="form-label">{{ __('loans.form.label_installments') }} <span class="text-danger">*</span></label>
                                <input type="number" class="form-control" id="installment_count" name="installment_count" value="{{ old('installment_count', 12) }}" min="1" max="120" required>
                                @error('installment_count')
                                    <div class="text-danger mt-1">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label for="interest_rate" class="form-label">{{ __('loans.form.label_interest_rate') }}</label>
                                <input type="number" class="form-control" id="interest_rate" name="interest_rate" value="{{ old('interest_rate', 0) }}" step="0.01" min="0" max="100">
                                @error('interest_rate')
                                    <div class="text-danger mt-1">{{ $message }}</div>
                                @enderror
                            </div>
                            <div class="col-md-6 mb-3">
                                <label for="start_date" class="form-label">{{ __('loans.form.label_start_date') }} <span class="text-danger">*</span></label>
                                <input type="date" class="form-control" id="start_date" name="start_date" value="{{ old('start_date', date('Y-m-d', strtotime('+1 month'))) }}" required>
                                @error('start_date')
                                    <div class="text-danger mt-1">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>

                        <div class="mb-3">
                            <label for="reason" class="form-label">{{ __('loans.headings.purpose') }} <span class="text-danger">*</span></label>
                            <textarea class="form-control" id="reason" name="reason" rows="3" placeholder="{{ __('loans.form_placeholders.reason') }}" required>{{ old('reason') }}</textarea>
                            @error('reason')
                                <div class="text-danger mt-1">{{ $message }}</div>
                            @enderror
                        </div>

                        <div class="mb-3">
                            <label for="notes" class="form-label">{{ __('loans.headings.additional_notes') }}</label>
                            <textarea class="form-control" id="notes" name="notes" rows="2" placeholder="{{ __('loans.form_placeholders.notes') }}">{{ old('notes') }}</textarea>
                            @error('notes')
                                <div class="text-danger mt-1">{{ $message }}</div>
                            @enderror
                        </div>

                        <div class="d-flex gap-2">
                            <button type="submit" class="btn btn-success">
                                <i class="fas fa-paper-plane"></i> {{ __('loans.form.create_title') }}
                            </button>
                            <a href="{{ route('loans.index') }}" class="btn btn-secondary">
                                <i class="fas fa-arrow-left"></i> {{ __('loans.form.back') }}
                            </a>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>

    <!-- Loan Calculation Preview -->
    <div class="row mt-4">
        <div class="col-md-6">
                    <div class="card">
                        <div class="card-body">
                            <h5 class="card-title">{{ __('loans.preview.title') }}</h5>
                            <div class="table-responsive">
                                <table class="table table-borderless">
                                    <tr>
                                        <th>{{ __('loans.preview.principal') }}:</th>
                                        <td id="preview_amount">0.00 {{ get_currency_code() }}</td>
                                    </tr>
                                    <tr>
                                        <th>{{ __('loans.preview.interest') }}:</th>
                                        <td id="preview_interest">0.00 {{ get_currency_code() }}</td>
                                    </tr>
                                    <tr>
                                        <th>{{ __('loans.preview.total') }}:</th>
                                        <td id="preview_total" class="fw-bold">0.00 {{ get_currency_code() }}</td>
                                    </tr>
                                    <tr>
                                        <th>{{ __('loans.preview.monthly') }}:</th>
                                        <td id="preview_monthly" class="fw-bold">0.00 {{ get_currency_code() }}</td>
                                    </tr>
                                </table>
                            </div>
                        </div>
                    </div>
        </div>

        <!-- Employee Loan History -->
        @if(isset($employees) && !auth()->user()->hasRole(['admin', 'hr', 'finance']))
            @php
                $employee = auth()->user()->employee;
            @endphp
        @elseif(old('employee_id'))
            @php
                $employee = \App\Models\Employee::find(old('employee_id'));
            @endphp
        @else
            @php
                $employee = null;
            @endphp
        @endif

        @if($employee)
        <div class="col-md-6">
            <div class="card">
                <div class="card-body">
                    <h5 class="card-title">سجل القروض السابقة</h5>
                    <div class="table-responsive">
                        <table class="table table-sm">
                            <thead>
                                <tr>
                                    <th>النوع</th>
                                    <th>المبلغ</th>
                                    <th>الحالة</th>
                                </tr>
                            </thead>
                            <tbody>
                                @forelse($employee->loans->take(5) as $prevLoan)
                                <tr>
                                    <td>
                                        @switch($prevLoan->loan_type)
                                            @case('personal') شخصي @break
                                            @case('housing') سكن @break
                                            @case('car') سيارة @break
                                            @case('emergency') طارئ @break
                                            @case('education') تعليم @break
                                            @default {{ $prevLoan->loan_type }}
                                        @endswitch
                                    </td>
                                    <td>{{ number_format($prevLoan->amount, 2) }} {{ $currency_name ?? 'د.ك' }}</td>
                                    <td>
                                        @if($prevLoan->status == 'paid')
                                            <span class="badge bg-success text-white">مدفوع</span>
                                        @elseif($prevLoan->status == 'active')
                                            <span class="badge bg-primary text-white">نشط</span>
                                        @elseif($prevLoan->status == 'rejected')
                                            <span class="badge bg-danger text-white">مرفوض</span>
                                        @elseif($prevLoan->status == 'pending')
                                            <span class="badge bg-warning text-white">معلق</span>
                                        @endif
                                    </td>
                                </tr>
                                @empty
                                <tr>
                                    <td colspan="3" class="text-center">لا توجد قروض سابقة</td>
                                </tr>
                                @endforelse
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
        @endif
    </div>
</div>

@push('scripts')
<script>
document.addEventListener('DOMContentLoaded', function() {
    // تهيئة Select2
    if (typeof $.fn.select2 !== 'undefined') {
        $('.select2').select2({
            placeholder: "اختر موظفاً",
            allowClear: true // يسمح بإزالة الاختيار
        });
    } else {
        console.error('مكتبة Select2 غير معرفة. تأكد من تضمينها.');
    }

    // حساب تفاصيل القرض
    const amountInput = document.getElementById('amount');
    const interestRateInput = document.getElementById('interest_rate');
    const installmentCountInput = document.getElementById('installment_count');
    const previewAmount = document.getElementById('preview_amount');
    const previewInterest = document.getElementById('preview_interest');
    const previewTotal = document.getElementById('preview_total');
    const previewMonthly = document.getElementById('preview_monthly');

    function calculateLoan() {
        const amount = parseFloat(amountInput.value) || 0;
        const interestRate = parseFloat(interestRateInput.value) || 0;
        const installmentCount = parseInt(installmentCountInput.value) || 1;
        
        const interest = amount * (interestRate / 100);
        const totalAmount = amount + interest;
        const monthlyInstallment = totalAmount / installmentCount;
        
        previewAmount.textContent = amount.toFixed(2) + ' {{ $currency_name ?? 'د.ك' }}';
        previewInterest.textContent = interest.toFixed(2) + ' {{ $currency_name ?? 'د.ك' }}';
        previewTotal.textContent = totalAmount.toFixed(2) + ' {{ $currency_name ?? 'د.ك' }}';
        previewMonthly.textContent = monthlyInstallment.toFixed(2) + ' {{ $currency_name ?? 'د.ك' }}';
    }

    if (amountInput) amountInput.addEventListener('input', calculateLoan);
    if (interestRateInput) interestRateInput.addEventListener('input', calculateLoan);
    if (installmentCountInput) installmentCountInput.addEventListener('input', calculateLoan);
    
    // حساب أولي
    calculateLoan();
});
</script>
@endpush
@endsection