{{-- resources/views/leaves/create.blade.php --}}
@extends('layouts.app')

@section('content')
<div class="container-fluid">

    <!-- Page Title -->
    <div class="row">
        <div class="col-12">
            <div class="page-title-box d-flex align-items-center justify-content-between">
                <h4 class="mb-0">{{ __('leaves.create.title') }}</h4>
                <div class="page-title-right">
                    <ol class="breadcrumb m-0">
                        <li class="breadcrumb-item"><a href="{{ route('dashboard.index') }}">{{ __('leaves.breadcrumb.home') }}</a></li>
                        <li class="breadcrumb-item"><a href="{{ route('leaves.index') }}">{{ __('leaves.breadcrumb.leaves') }}</a></li>
                        <li class="breadcrumb-item active">{{ __('leaves.create.breadcrumb') }}</li>
                    </ol>
                </div>
            </div>
        </div>
    </div>

    <!-- Form Card -->
    <div class="row">
        <div class="col-12">
            <div class="card">
                <div class="card-body">
                    <h4 class="card-title mb-4">{{ __('leaves.create.form_title') }}</h4>

                    <form action="{{ route('leaves.store') }}" method="POST">
                        @csrf
                        <div class="row">

                            {{-- Employee Selection --}}
                            @if(auth()->user()->hasRole('admin') || auth()->user()->hasRole('hr') || auth()->user()->hasRole('finance'))
                            <div class="col-md-6 mb-3">
                                <label for="employee_id" class="form-label">{{ __('leaves.form.employee') }} <span class="text-danger">*</span></label>
                                <select class="form-control select2" id="employee_id" name="employee_id" required>
                                    <option value="">{{ __('leaves.form.select_employee') }}</option>
                                    @foreach($employees as $employee)
                                        <option value="{{ $employee->id }}" {{ old('employee_id') == $employee->id ? 'selected' : '' }}>
                                            {{ $employee->name }} - {{ $employee->employee_id }}
                                        </option>
                                    @endforeach
                                </select>
                                @error('employee_id')
                                    <div class="text-danger mt-1">{{ $message }}</div>
                                @enderror
                            </div>
                            @else
                                <!-- إذا كان المستخدم موظفًا عاديًا، نعرض اسمه ونخفي الـ ID -->
                                @php
                                    $currentEmployee = $employees->first();
                                @endphp
                                <input type="hidden" name="employee_id" value="{{ $currentEmployee?->id ?? '' }}">
                                <div class="col-md-6 mb-3">
                                    <label class="form-label">{{ __('leaves.form.employee') }}</label>
                                    <div class="input-group">
                                        <input type="text" class="form-control" value="{{ $currentEmployee?->name ?? __('leaves.not_specified') }}" readonly>
                                        <span class="input-group-text">{{ $currentEmployee?->employee_id ?? '-' }}</span>
                                    </div>
                                </div>
                            @endif

                            {{-- Leave Type --}}
                            <div class="col-md-6 mb-3">
                                <label for="leave_type" class="form-label">{{ __('leaves.form.leave_type') }} <span class="text-danger">*</span></label>
                                <select class="form-control" id="leave_type" name="leave_type" required>
                                    <option value="">{{ __('leaves.form.select_type') }}</option>
                                    @if(isset($leaveTypes) && $leaveTypes->count())
                                        @foreach($leaveTypes as $lt)
                                            <option value="{{ $lt->code }}" {{ old('leave_type') == $lt->code ? 'selected' : '' }}>{{ $lt->name }}</option>
                                        @endforeach
                                    @else
                                        <option value="annual" {{ old('leave_type') == 'annual' ? 'selected' : '' }}>{{ __('leaves.types.annual') }}</option>
                                        <option value="sick" {{ old('leave_type') == 'sick' ? 'selected' : '' }}>{{ __('leaves.types.sick') }}</option>
                                        <option value="emergency" {{ old('leave_type') == 'emergency' ? 'selected' : '' }}>{{ __('leaves.types.emergency') }}</option>
                                        <option value="maternity" {{ old('leave_type') == 'maternity' ? 'selected' : '' }}>{{ __('leaves.types.maternity') }}</option>
                                        <option value="paternity" {{ old('leave_type') == 'paternity' ? 'selected' : '' }}>{{ __('leaves.types.paternity') }}</option>
                                        <option value="unpaid" {{ old('leave_type') == 'unpaid' ? 'selected' : '' }}>{{ __('leaves.types.unpaid') }}</option>
                                    @endif
                                </select>
                                @error('leave_type')
                                    <div class="text-danger mt-1">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>

                        {{-- Dates --}}
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label for="start_date" class="form-label">{{ __('leaves.form.start_date') }} <span class="text-danger">*</span></label>
                                <input type="date" class="form-control" id="start_date" name="start_date" value="{{ old('start_date') }}" required min="{{ date('Y-m-d') }}">
                                @error('start_date')
                                    <div class="text-danger mt-1">{{ $message }}</div>
                                @enderror
                            </div>

                            <div class="col-md-6 mb-3">
                                <label for="end_date" class="form-label">{{ __('leaves.form.end_date') }} <span class="text-danger">*</span></label>
                                <input type="date" class="form-control" id="end_date" name="end_date" value="{{ old('end_date') }}" required min="{{ date('Y-m-d') }}">
                                @error('end_date')
                                    <div class="text-danger mt-1">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>

                        {{-- Days Count --}}
                        <div class="mb-3">
                            <label for="days_count" class="form-label">{{ __('leaves.form.days_count') }} <span class="text-danger">*</span></label>
                            <input type="number" class="form-control" id="days_count" name="days_count" value="{{ old('days_count') }}" required min="1" max="365" readonly>
                            @error('days_count')
                                <div class="text-danger mt-1">{{ $message }}</div>
                            @enderror
                        </div>

                        {{-- Reason --}}
                        <div class="mb-3">
                            <label for="reason" class="form-label">{{ __('leaves.form.reason') }} <span class="text-danger">*</span></label>
                            <textarea class="form-control" id="reason" name="reason" rows="4" placeholder="{{ __('leaves.form.reason_placeholder') }}" required>{{ old('reason') }}</textarea>
                            @error('reason')
                                <div class="text-danger mt-1">{{ $message }}</div>
                            @enderror
                        </div>

                        {{-- Contact Details --}}
                        <div class="mb-3">
                            <label for="contact_details" class="form-label">{{ __('leaves.form.contact_details') }}</label>
                            <input type="text" class="form-control" id="contact_details" name="contact_details" value="{{ old('contact_details') }}" placeholder="{{ __('leaves.form.contact_placeholder') }}">
                            @error('contact_details')
                                <div class="text-danger mt-1">{{ $message }}</div>
                            @enderror
                        </div>

                        {{-- Form Actions --}}
                        <div class="d-flex gap-2">
                            @if(auth()->user()->hasRole('admin') || auth()->user()->hasRole('hr') || auth()->user()->hasRole('finance'))
                                <div class="form-check me-3">
                                    <input class="form-check-input" type="checkbox" value="1" id="auto_approve" name="auto_approve">
                                    <label class="form-check-label" for="auto_approve">
                                        {{ __('leaves.form.auto_approve') ?? 'موافقة تلقائية (للمدير/الموارد البشرية)'}}
                                    </label>
                                </div>
                            @endif
                            <button type="submit" class="btn btn-success">
                                <i class="fas fa-paper-plane"></i> {{ __('leaves.form.submit') }}
                            </button>
                            <a href="{{ route('leaves.index') }}" class="btn btn-secondary">
                                <i class="fas fa-arrow-left"></i> {{ __('leaves.form.back') }}
                            </a>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>

    {{-- Employee Info and Leave Balance --}}
    <div class="row mt-4" id="employee-info-section" style="display: none;">
        <div class="col-md-6">
            <div class="card">
                <div class="card-body">
                    <h5 class="card-title">معلومات الموظف</h5>
                    <div class="d-flex align-items-center mb-3">
                        <div id="employee-photo" class="me-3"></div>
                        <div>
                            <h6 id="employee-name">-</h6>
                            <p class="mb-0 text-muted" id="employee-position">غير محدد</p>
                            <p class="mb-0 text-muted" id="employee-department">غير محدد</p>
                        </div>
                    </div>

                    <div class="row text-center">
                        <div class="col-4">
                            <h6 id="annual_balance">0</h6>
                            <p class="text-muted mb-0">السنوية الممنوحة</p>
                        </div>
                        <div class="col-4">
                            <h6 id="used_balance">0</h6>
                            <p class="text-muted mb-0">المستخدمة</p>
                        </div>
                        <div class="col-4">
                            <h6 id="remaining_balance">0</h6>
                            <p class="text-muted mb-0">المتبقية</p>
                        </div>
                    </div>
                    <!-- Div لعرض الأخطاء -->
                    <div id="error-message" class="text-danger mt-2" style="display: none;"></div>
                </div>
            </div>
        </div>
    </div>

</div>

@push('scripts')
<script>
document.addEventListener('DOMContentLoaded', function() {
    const startDateInput = document.getElementById('start_date');
    const endDateInput = document.getElementById('end_date');
    const daysCountInput = document.getElementById('days_count');
    const employeeSelect = document.getElementById('employee_id');
    const leaveTypeSelect = document.getElementById('leave_type');
    const employeeInfoSection = document.getElementById('employee-info-section');
    const errorMessageDiv = document.getElementById('error-message'); // Div جديد لعرض الأخطاء

    function calculateDays() {
        if (startDateInput.value && endDateInput.value) {
            const start = new Date(startDateInput.value);
            const end = new Date(endDateInput.value);
            if (end >= start) {
                daysCountInput.value = Math.floor((end - start) / (1000 * 3600 * 24)) + 1;
            } else {
                daysCountInput.value = '';
            }
        } else {
            daysCountInput.value = '';
        }
    }

    function fetchEmployeeInfo(employeeId) {
        // أولاً، أخف القسم وامسح رسالة الخطأ
        employeeInfoSection.style.display = 'none';
        errorMessageDiv.style.display = 'none';
        errorMessageDiv.textContent = '';

        if (!employeeId) {
            console.log("No employee ID selected.");
            return;
        }

        console.log("Fetching info for employee ID:", employeeId); // تسجيل ID

        // قم بطلب البيانات
        fetch(`/api/employee-info/${employeeId}`)
            .then(response => {
                console.log("Response status:", response.status); // تسجيل الحالة
                if (!response.ok) {
                    throw new Error(`HTTP error! status: ${response.status}`);
                }
                return response.json();
            })
            .then(data => {
                console.log("Received data:", data); // تسجيل البيانات

                // تحقق مما إذا كانت البيانات تحتوي على الحقول المطلوبة
                if (data.success && data.employee) {
                    const emp = data.employee; // استخدم emp للوصول إلى البيانات

                    // Update photo
                    const photoDiv = document.getElementById('employee-photo');
                    if (emp.photo) {
                        photoDiv.innerHTML = `<img src="${emp.photo}" alt="صورة الموظف" class="rounded-circle" width="60" height="60">`;
                    } else {
                        photoDiv.innerHTML = `<div class="avatar-md"><span class="avatar-title bg-primary text-white rounded-circle">${emp.name.charAt(0)}</span></div>`;
                    }

                    // Update info
                    document.getElementById('employee-name').textContent = emp.name || '-';
                    document.getElementById('employee-position').textContent = emp.position?.name || 'غير محدد';
                    document.getElementById('employee-department').textContent = emp.department?.name || 'غير محدد';

                    // Update leave balances
                    // استخدم الحقول التي تُرجعها دالة getAnnualGranted و getAnnualUsed
                    // تأكد من أن API تُرجعها
                    const granted = emp.annual_granted || emp.granted || 0; // حاول الاسم الجديد أولاً
                    const used = emp.annual_used || emp.used || 0;         // حاول الاسم الجديد أولاً
                    const remaining = granted - used;

                    document.getElementById('annual_balance').textContent = granted;
                    document.getElementById('used_balance').textContent = used;
                    document.getElementById('remaining_balance').textContent = remaining;

                    employeeInfoSection.style.display = 'block'; // أظهر القسم
                } else {
                    // إذا لم تكن الاستجابة success: true
                    throw new Error("API response indicates failure or missing employee data.");
                }
            })
            .catch(error => {
                console.error("Error fetching employee info:", error); // تسجيل الخطأ
                errorMessageDiv.textContent = `حدث خطأ أثناء تحميل معلومات الموظف: ${error.message}`;
                errorMessageDiv.style.display = 'block'; // أظهر رسالة الخطأ
                employeeInfoSection.style.display = 'none'; // أخف القسم في حالة الخطأ
            });
    }

    startDateInput.addEventListener('change', calculateDays);
    endDateInput.addEventListener('change', calculateDays);

    if (employeeSelect) {
        employeeSelect.addEventListener('change', function() {
            if (leaveTypeSelect.value === 'annual' || leaveTypeSelect.value === 'emergency') {
                fetchEmployeeInfo(this.value);
            } else {
                employeeInfoSection.style.display = 'none';
                errorMessageDiv.style.display = 'none'; // إخفاء الخطأ عند تغيير النوع
            }
        });
    }

    leaveTypeSelect.addEventListener('change', function() {
        if ((this.value === 'annual' || this.value === 'emergency') && employeeSelect.value) {
            fetchEmployeeInfo(employeeSelect.value);
        } else {
            employeeInfoSection.style.display = 'none';
            errorMessageDiv.style.display = 'none'; // إخفاء الخطأ عند تغيير النوع
        }
    });

    // Initialize info if already selected and relevant type
    if (employeeSelect && employeeSelect.value && (leaveTypeSelect.value === 'annual' || leaveTypeSelect.value === 'emergency')) {
        fetchEmployeeInfo(employeeSelect.value);
    }

    calculateDays();
});
</script>
@endpush
@endsection