@php
/**
 * Partial: Salary Histories management
 * Variables:
 * - $employee (Employee)
 * - $histories (Collection) optional
 * - $history (optional) when editing
 */
// ✅ جلب آخر سجل تعويض لملء النموذج تلقائياً عند الإضافة
$lastHistory = null;
if (!isset($history)) {
    $lastHistory = $employee->salaryHistories()->orderByDesc('effective_from')->first();
}

// ✅ إذا لم يكن هناك سجل تعويض سابق، استخدم بيانات الموظف الأساسية
if (!isset($history) && !$lastHistory) {
    // إنشاء كائن افتراضي يحتوي على بيانات الموظف
    $defaultHistory = new stdClass();
    $defaultHistory->base_salary = $employee->salary ?? 0;
    $defaultHistory->incentives = $employee->incentives ?? 0;
    $defaultHistory->overtime_rate = $employee->overtime_hourly_rate ?? null;
    $defaultHistory->monthly_leave_entitlement = $employee->monthly_leave_days_allowed ?? null;
    $defaultHistory->annual_leave_entitlement = $employee->annual_entitlement ?? null;
    $defaultHistory->note = null;
    // تحويل hire_date إلى Carbon إذا كان موجوداً
    if ($employee->hire_date) {
        $defaultHistory->effective_from = $employee->hire_date instanceof \Carbon\Carbon 
            ? $employee->hire_date 
            : \Carbon\Carbon::parse($employee->hire_date);
    } else {
        $defaultHistory->effective_from = \Carbon\Carbon::now();
    }
    
    // ✅ جمع المكونات الإضافية من العلاقة salaryComponents
    $components = [];
    if ($employee->relationLoaded('salaryComponents')) {
        foreach ($employee->salaryComponents as $comp) {
            if (!empty($comp->name) && ($comp->pivot->value ?? 0) > 0) {
                $components[$comp->name] = (float) $comp->pivot->value;
            }
        }
    } else {
        // إذا لم تكن العلاقة محمّلة، قم بتحميلها
        $employee->load('salaryComponents');
        foreach ($employee->salaryComponents as $comp) {
            if (!empty($comp->name) && ($comp->pivot->value ?? 0) > 0) {
                $components[$comp->name] = (float) $comp->pivot->value;
            }
        }
    }
    
    // ✅ إضافة المكونات من الحقول الثابتة (component_1 إلى component_7)
    for ($i = 1; $i <= 7; $i++) {
        $nameField = "component_name_$i";
        $valueField = "component_$i";
        $name = $employee->{$nameField};
        $value = (float) ($employee->{$valueField} ?? 0);
        if (!empty($name) && $value > 0) {
            $components[$name] = $value;
        }
    }
    
    $defaultHistory->components = $components;
} else {
    // إذا كان هناك $history (تعديل)، استخدمه، وإلا استخدم $lastHistory
    $defaultHistory = $history ?? $lastHistory;
}
@endphp

<div class="card mt-3">
    <div class="card-header">{{ __('employees.salary_histories.title') }} - {{ $employee->name }}</div>
    <div class="card-body">
        @if(session('success'))
            <div class="alert alert-success">{{ session('success') }}</div>
        @endif
        @if(isset($histories) && $histories->count())
            <div class="table-responsive">
                <table class="table table-sm table-bordered">
                    <thead>
                        <tr>
                            <th>{{ __('employees.salary_histories.table.effective_from') }}</th>
                            <th>{{ __('employees.salary_histories.table.base_salary') }}</th>
                            <th>{{ __('employees.salary_histories.table.incentives') }}</th>
                            <th>{{ __('employees.salary_histories.table.overtime_rate') }}</th>
                            <th>{{ __('employees.salary_histories.table.monthly_leave') }}</th>
                            <th>{{ __('employees.salary_histories.table.annual_leave') }}</th>
                            <th>{{ __('employees.salary_histories.table.components') }}</th>
                            <th>{{ __('employees.salary_histories.table.note') }}</th>
                            <th>{{ __('employees.salary_histories.table.actions') }}</th>
                        </tr>
                    </thead>
                    <tbody>
                        @foreach($histories as $h)
                            <tr>
                                <td>{{ optional($h->effective_from)->toDateString() }}</td>
                                <td>{{ number_format($h->base_salary,2) }}</td>
                                <td>{{ number_format($h->incentives ?? 0,2) }}</td>
                                <td>{{ $h->overtime_rate ?? '-' }}</td>
                                <td>{{ $h->monthly_leave_entitlement ?? '-' }}</td>
                                <td>{{ $h->annual_leave_entitlement ?? '-' }}</td>
                                <td>
                                    @if(is_array($h->components) && count($h->components) > 0)
                                        <ul class="mb-0" style="font-size: 0.85em;">
                                            @foreach($h->components as $name => $value)
                                                <li>{{ $name }}: {{ number_format($value, 2) }}</li>
                                            @endforeach
                                        </ul>
                                    @else
                                        <span class="text-muted">-</span>
                                    @endif
                                </td>
                                <td>{{ Str::limit($h->note ?? '', 50) }}</td>
                                <td>
                                    <a href="{{ route('employees.salary_histories.edit', [$employee->id, $h->id]) }}" class="btn btn-sm btn-primary">{{ __('employees.show.actions.edit') }}</a>
                                    <form action="{{ route('employees.salary_histories.destroy', [$employee->id, $h->id]) }}" method="POST" style="display:inline-block" onsubmit="return confirm('{{ __('employees.salary_histories.messages.delete_confirm') }}');">
                                        @csrf @method('DELETE')
                                        <button class="btn btn-sm btn-danger">{{ __('employees.show.actions.delete') }}</button>
                                    </form>
                                </td>
                            </tr>
                        @endforeach
                    </tbody>
                </table>
            </div>
        @else
            <p>{{ __('employees.salary_histories.no_records') }}</p>
        @endif

        <hr />

        <h5>@if(isset($history)) {{ __('employees.salary_histories.edit_record') }} @else {{ __('employees.salary_histories.add_new') }} @endif</h5>
        <form method="POST" action="@if(isset($history)) {{ route('employees.salary_histories.update', [$employee->id, $history->id]) }} @else {{ route('employees.salary_histories.store', $employee->id) }} @endif" id="salaryHistoryForm">
            @csrf
            @if(isset($history)) @method('PUT') @endif
            
            <div class="row">
                <div class="col-md-3">
                    <label>{{ __('employees.salary_histories.form.effective_from') }} <span class="text-danger">*</span></label>
                    @php
                        $effectiveFromValue = '';
                        if ($defaultHistory && isset($defaultHistory->effective_from)) {
                            if (is_object($defaultHistory->effective_from) && method_exists($defaultHistory->effective_from, 'toDateString')) {
                                $effectiveFromValue = $defaultHistory->effective_from->toDateString();
                            } elseif ($defaultHistory->effective_from instanceof \Carbon\Carbon) {
                                $effectiveFromValue = $defaultHistory->effective_from->toDateString();
                            } elseif (is_string($defaultHistory->effective_from)) {
                                $effectiveFromValue = $defaultHistory->effective_from;
                            } else {
                                $effectiveFromValue = \Carbon\Carbon::parse($defaultHistory->effective_from)->toDateString();
                            }
                        }
                    @endphp
                    <input type="date" name="effective_from" id="effective_from" class="form-control" value="{{ old('effective_from', $effectiveFromValue) }}" required>
                    <small class="text-muted">{{ __('employees.salary_histories.form.effective_from_hint') }}</small>
                </div>
                <div class="col-md-3">
                    <label>{{ __('employees.salary_histories.form.base_salary') }} <span class="text-danger">*</span></label>
                    <input type="number" step="0.01" name="base_salary" class="form-control" value="{{ old('base_salary', $defaultHistory->base_salary ?? '') }}" required>
                </div>
                <div class="col-md-3">
                    <label>{{ __('employees.salary_histories.form.incentives') }}</label>
                    <input type="number" step="0.01" name="incentives" class="form-control" value="{{ old('incentives', $defaultHistory->incentives ?? '') }}">
                </div>
                <div class="col-md-3">
                    <label>{{ __('employees.salary_histories.form.overtime_rate') }}</label>
                    <input type="number" step="0.0001" name="overtime_rate" class="form-control" value="{{ old('overtime_rate', $defaultHistory->overtime_rate ?? '') }}">
                </div>
            </div>
            
            <div class="row mt-2">
                <div class="col-md-3">
                    <label>{{ __('employees.salary_histories.form.monthly_leave_entitlement') }}</label>
                    <input type="number" step="0.1" name="monthly_leave_entitlement" class="form-control" value="{{ old('monthly_leave_entitlement', $defaultHistory->monthly_leave_entitlement ?? '') }}">
                </div>
                <div class="col-md-3">
                    <label>{{ __('employees.salary_histories.form.annual_leave_entitlement') }}</label>
                    <input type="number" step="0.1" name="annual_leave_entitlement" class="form-control" value="{{ old('annual_leave_entitlement', $defaultHistory->annual_leave_entitlement ?? '') }}">
                </div>
            </div>

            <div class="row mt-3">
                <div class="col-md-12">
                    <label class="mb-2">{{ __('employees.salary_histories.form.components') }}</label>
                    <div id="componentsContainer">
                        @php
                            // ✅ استخدام $defaultHistory بدلاً من $history فقط
                            $existingComponents = old('component_names', $defaultHistory && is_array($defaultHistory->components) ? array_keys($defaultHistory->components) : []);
                            $existingValues = old('component_values', $defaultHistory && is_array($defaultHistory->components) ? array_values($defaultHistory->components) : []);
                        @endphp
                        @if(count($existingComponents) > 0)
                            @foreach($existingComponents as $index => $name)
                                <div class="row mb-2 component-row">
                                    <div class="col-md-5">
                                        <input type="text" name="component_names[]" class="form-control" placeholder="{{ __('employees.salary_histories.form.component_name_placeholder') }}" value="{{ $name }}">
                                    </div>
                                    <div class="col-md-5">
                                        <input type="number" step="0.01" name="component_values[]" class="form-control" placeholder="{{ __('employees.salary_histories.form.component_value_placeholder') }}" value="{{ $existingValues[$index] ?? '' }}">
                                    </div>
                                    <div class="col-md-2">
                                        <button type="button" class="btn btn-sm btn-danger remove-component">{{ __('employees.salary_histories.form.remove_component') }}</button>
                                    </div>
                                </div>
                            @endforeach
                        @endif
                    </div>
                    <button type="button" id="addComponentBtn" class="btn btn-sm btn-info mt-2">
                        <i class="fas fa-plus"></i> {{ __('employees.salary_histories.form.add_component') }}
                    </button>
                </div>
            </div>

            <div class="row mt-2">
                <div class="col-md-12">
                    <label>{{ __('employees.salary_histories.form.note') }}</label>
                    <textarea name="note" class="form-control" rows="2">{{ old('note', $defaultHistory->note ?? '') }}</textarea>
                </div>
            </div>
            
            <div class="row mt-3">
                <div class="col-md-12 text-end">
                    <button type="submit" class="btn btn-success">
                        <i class="fas fa-save"></i> {{ isset($history) ? __('employees.salary_histories.form.save') : __('employees.salary_histories.form.add') }}
                    </button>
                </div>
            </div>
        </form>
    </div>
</div>

@push('scripts')
<script>
document.addEventListener('DOMContentLoaded', function() {
    // تحويل effective_from إلى بداية الشهر تلقائيًا
    const effectiveFromInput = document.getElementById('effective_from');
    if (effectiveFromInput) {
        effectiveFromInput.addEventListener('change', function() {
            if (this.value) {
                const date = new Date(this.value);
                const firstDay = new Date(date.getFullYear(), date.getMonth(), 1);
                const formattedDate = firstDay.toISOString().split('T')[0];
                this.value = formattedDate;
            }
        });
    }

    // إضافة مكون جديد
    const addComponentBtn = document.getElementById('addComponentBtn');
    const componentsContainer = document.getElementById('componentsContainer');
    
    if (addComponentBtn && componentsContainer) {
        addComponentBtn.addEventListener('click', function() {
            const newRow = document.createElement('div');
            newRow.className = 'row mb-2 component-row';
            newRow.innerHTML = `
                <div class="col-md-5">
                    <input type="text" name="component_names[]" class="form-control" placeholder="{{ __('employees.salary_histories.form.component_name_placeholder') }}">
                </div>
                <div class="col-md-5">
                    <input type="number" step="0.01" name="component_values[]" class="form-control" placeholder="{{ __('employees.salary_histories.form.component_value_placeholder') }}">
                </div>
                <div class="col-md-2">
                    <button type="button" class="btn btn-sm btn-danger remove-component">{{ __('employees.salary_histories.form.remove_component') }}</button>
                </div>
            `;
            componentsContainer.appendChild(newRow);
        });
    }

    // حذف مكون
    document.addEventListener('click', function(e) {
        if (e.target && e.target.classList.contains('remove-component')) {
            e.target.closest('.component-row').remove();
        }
    });
});
</script>
@endpush
