@extends('layouts.app')

@section('content')
<div class="container-fluid">
    <!-- Page Title & Breadcrumbs -->
    <div class="row">
        <div class="col-12">
            <div class="page-title-box d-flex align-items-center justify-content-between">
                <h4 class="mb-0">{{ __('app.edit') }} {{ __('app.employee') }}</h4>
                <div class="page-title-right">
                    <ol class="breadcrumb m-0">
                        <li class="breadcrumb-item"><a href="{{ route('dashboard.index') }}">{{ __('app.home') }}</a></li>
                        <li class="breadcrumb-item"><a href="{{ route('employees.index') }}">{{ __('app.employees') }}</a></li>
                        <li class="breadcrumb-item active">{{ __('app.edit') }} {{ __('app.employee') }}</li>
                    </ol>
                </div>
            </div>
        </div>
    </div>

    <div class="row">
        <div class="col-12">
            <div class="card">
                <div class="card-body">
                    <h4 class="card-title mb-4">{{ __('app.edit') }} {{ __('app.employee') }}</h4>

                    <!-- Display Validation Errors -->
                    @if ($errors->any())
                        <div class="alert alert-danger alert-border-left alert-dismissible fade show" role="alert">
                            <i class="mdi mdi-block-helper me-2"></i>
                            <strong>{{ __('app.error') }}!</strong> {{ __('app.please_check') }}.
                            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                            <ul class="mt-2 mb-0">
                                @foreach ($errors->all() as $error)
                                    <li>{{ $error }}</li>
                                @endforeach
                            </ul>
                        </div>
                    @endif

                    <!-- Employee Edit Form -->
                    <form action="{{ route('employees.update', $employee) }}" method="POST" enctype="multipart/form-data" id="employeeForm">
                        @csrf
                        @method('PUT')

                        <!-- Basic Information Section -->
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label for="name" class="form-label">{{ __('app.full_name') }} <span class="text-danger">*</span></label>
                                <input type="text" class="form-control" id="name" name="name" value="{{ old('name', $employee->name) }}" required>
                                @error('name')
                                    <div class="text-danger mt-1">{{ $message }}</div>
                                @enderror
                            </div>

                            <div class="col-md-6 mb-3">
                                <label for="employee_id" class="form-label">{{ __('app.employee_id') }}</label>
                                <input type="text" class="form-control" id="employee_id" name="employee_id" value="{{ old('employee_id', $employee->employee_id) }}" readonly>
                                @error('employee_id')
                                    <div class="text-danger mt-1">{{ $message }}</div>
                                @enderror
                            </div>

                            <div class="col-md-6 mb-3">
                                <label for="email" class="form-label">{{ __('app.email') }}</label>
                                <input type="email" class="form-control" id="email" name="email" value="{{ old('email', $employee->email) }}">
                                @error('email')
                                    <div class="text-danger mt-1">{{ $message }}</div>
                                @enderror
                            </div>

                            <div class="col-md-6 mb-3">
                                <label for="user_id" class="form-label">ربط حساب المستخدم (اختياري)</label>
                                <select class="form-select select2" id="user_id" name="user_id">
                                    <option value="">-- لا يوجد حساب مرتبط --</option>
                                    @php
                                        $users = \App\Models\User::orderBy('name')->get();
                                    @endphp
                                    @foreach($users as $user)
                                        <option value="{{ $user->id }}" {{ old('user_id', $employee->user_id ?? optional($employee->user)->id) == $user->id ? 'selected' : '' }}>
                                            {{ $user->name }} ({{ $user->email }})
                                        </option>
                                    @endforeach
                                </select>
                                <small class="text-muted">اختر حساب مستخدم لربطه بهذا الموظف. يمكّن الموظف من تسجيل الدخول والوصول لطلباته.</small>
                                @error('user_id')
                                    <div class="text-danger mt-1">{{ $message }}</div>
                                @enderror
                            </div>

                            <div class="col-md-6 mb-3">
                                <label for="phone" class="form-label">{{ __('app.phone') }}</label>
                                <input type="text" class="form-control" id="phone" name="phone" value="{{ old('phone', $employee->phone) }}">
                                @error('phone')
                                    <div class="text-danger mt-1">{{ $message }}</div>
                                @enderror
                            </div>

                            <!-- Department Selection with Quick Add -->
                            <div class="col-md-6 mb-3">
                                <label for="department_id" class="form-label">{{ __('app.department') }}</label>
                                <div class="input-group">
                                    <select class="form-select" id="department_id" name="department_id">
                                        <option value="">{{ __('app.select_department') }}</option>
                                        @foreach($departments as $dept)
                                            <option value="{{ $dept->id }}" {{ old('department_id', $employee->department_id) == $dept->id ? 'selected' : '' }}>
                                                {{ $dept->name }}
                                            </option>
                                        @endforeach
                                    </select>
                                </div>
                                @error('department_id')
                                    <div class="text-danger mt-1">{{ $message }}</div>
                                @enderror
                            </div>

                            <div class="col-md-6 mb-3">
                                <label for="position_id" class="form-label">{{ __('app.position') }}</label>
                                <div class="input-group">
                                    <select class="form-select" id="position_id" name="position_id">
                                        <option value="">{{ __('app.select_position') }}</option>
                                        @foreach($positions as $pos)
                                            <option value="{{ $pos->id }}" {{ old('position_id', $employee->position_id) == $pos->id ? 'selected' : '' }}>
                                                {{ $pos->name }} @if($pos->department) ({{ $pos->department->name }}) @endif
                                            </option>
                                        @endforeach
                                    </select>
                                    <button class="btn btn-outline-secondary" type="button" data-bs-toggle="modal" data-bs-target="#addPositionModal">
                                        <i class="fas fa-plus"></i>
                                    </button>
                                </div>
                                @error('position_id')
                                    <div class="text-danger mt-1">{{ $message }}</div>
                                @enderror
                            </div>

                            <div class="col-md-6 mb-3">
                                <label for="hire_date" class="form-label">{{ __('app.hire_date') }}</label>
                                <input type="date" class="form-control" id="hire_date" name="hire_date"
                                       value="{{ old('hire_date', $employee->hire_date ? $employee->hire_date->format('Y-m-d') : '') }}">
                                @error('hire_date')
                                    <div class="text-danger mt-1">{{ $message }}</div>
                                @enderror
                            </div>

                            <!-- الراتب الأساسي والحوافز -->
                            <div class="col-md-6 mb-3">
                                <label for="salary" class="form-label">{{ __('app.basic_salary') }}</label>
                                <input type="number" step="0.01" class="form-control" id="salary" name="salary"
                                       value="{{ old('salary', number_format($employee->salary ?? 0, 2, '.', '')) }}">
                                <small class="text-muted">{{ __('app.example') }}: 1500.00 {{ $currency_name }}</small>
                                @error('salary')
                                    <div class="text-danger mt-1">{{ $message }}</div>
                                @enderror
                            </div>

                            <div class="col-md-6 mb-3">
                                <label for="incentives" class="form-label">{{ __('app.incentives') }} ({{ __('app.optional') }})</label>
                                <input type="number" step="0.01" class="form-control" id="incentives" name="incentives"
                                       value="{{ old('incentives', number_format($employee->incentives ?? 0, 2, '.', '')) }}">
                                <small class="text-muted">{{ __('app.example') }}: {{ __('app.production_incentives') }}, {{ __('app.performance_incentives') }}, {{ __('app.sales_incentives') }} ({{ __('app.currency') }}: {{ $currency_name }})</small>
                                @error('incentives')
                                    <div class="text-danger mt-1">{{ $message }}</div>
                                @enderror
                            </div>

                            <div class="col-md-6 mb-3">
                                <label for="default_paid_off_amount" class="form-label">قيمة اليوم الحضور المدفوع (افتراضي)</label>
                                <input type="number" step="0.01" class="form-control" id="default_paid_off_amount" name="default_paid_off_amount"
                                       value="{{ old('default_paid_off_amount', number_format($employee->default_paid_off_amount ?? 0, 2, '.', '')) }}">
                                <small class="text-muted">قيمة مفترضة ليوم الحضور الذي وقع في عطلة (تستخدم كقيمة افتراضية عند إنشاء الرواتب).</small>
                                @error('default_paid_off_amount')
                                    <div class="text-danger mt-1">{{ $message }}</div>
                                @enderror
                            </div>

                            <div class="col-md-6 mb-3">
                                <label for="overtime_hourly_rate" class="form-label">{{ __('app.overtime_hourly_rate') ?? 'سعر ساعة العمل الإضافي' }}</label>
                                <input type="number" step="0.01" class="form-control" id="overtime_hourly_rate" name="overtime_hourly_rate"
                                       value="{{ old('overtime_hourly_rate', number_format($employee->overtime_hourly_rate ?? 0, 2, '.', '')) }}">
                                <small class="text-muted">{{ __('app.example') }}: 5.00 {{ $currency_name }}</small>
                                @error('overtime_hourly_rate')
                                    <div class="text-danger mt-1">{{ $message }}</div>
                                @enderror
                            </div>

                            <div class="col-md-6 mb-3">
                                <label for="deduction_hourly_rate" class="form-label">{{ __('app.deduction_hourly_rate') ?? 'ساعة خصم النقص' }}</label>
                                <input type="number" step="0.01" class="form-control" id="deduction_hourly_rate" name="deduction_hourly_rate"
                                       value="{{ old('deduction_hourly_rate', number_format($employee->deduction_hourly_rate ?? 0, 2, '.', '')) }}">
                                <small class="text-muted">{{ __('app.example') }}: 5.00 {{ $currency_name }} - المبلغ الذي يخصم من الموظف عن كل ساعة نقص</small>
                                @error('deduction_hourly_rate')
                                    <div class="text-danger mt-1">{{ $message }}</div>
                                @enderror
                            </div>

                            <div class="col-md-6 mb-3">
                                <label for="total_salary_display" class="form-label">{{ __('app.total_salary') }} ({{ __('app.estimated') }})</label>
                                <input type="text" id="total_salary_display" class="form-control" readonly placeholder="{{ __('app.will_be_calculated') }}">
                                <small class="text-muted">{{ __('app.basic_salary') }} + {{ __('app.incentives') }} + {{ __('app.additional_salary_components') }} ({{ __('app.currency') }}: {{ $currency_name }})</small>
                            </div>

                            <!-- ✅ الحقل الجديد: عدد أيام الإجازة الشهرية المجانية -->
                            <div class="col-md-6 mb-3">
                                <label for="monthly_leave_days_allowed" class="form-label">{{ __('app.monthly_leave_days_allowed') }}</label>
                                <input type="number" name="monthly_leave_days_allowed" id="monthly_leave_days_allowed"
                                       class="form-control"
                                       value="{{ old('monthly_leave_days_allowed', $employee->monthly_leave_days_allowed) }}"
                                       min="0" max="30" required>
                                @error('monthly_leave_days_allowed')
                                    <div class="text-danger mt-1">{{ $message }}</div>
                                @enderror
                                <small class="text-muted">{{ __('app.number_of_days') }} {{ __('app.employee_can_take') }} {{ __('app.per_month') }} {{ __('app.without_deduction') }}.</small>
                            </div>

                            <div class="col-md-6 mb-3">
                                <label for="annual_entitlement" class="form-label">{{ __('app.annual_entitlement') ?? 'الرصيد السنوي' }}</label>
                                <input type="number"
                                       name="annual_entitlement"
                                       id="annual_entitlement"
                                       class="form-control"
                                       value="{{ old('annual_entitlement', $employee->annual_entitlement) }}"
                                       min="0">
                                @error('annual_entitlement')
                                    <div class="text-danger mt-1">{{ $message }}</div>
                                @enderror
                                <small class="text-muted">{{ __('app.annual_entitlement_info') }}</small>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <div class="form-check">
                                    <!-- ✅ hidden input لإرسال 0 عند عدم الاختيار -->
                                    <input type="hidden" name="accrue_leaves" value="0">
                                    <input class="form-check-input" type="checkbox" value="1" id="accrue_leaves" name="accrue_leaves" {{ old('accrue_leaves', $employee->accrue_leaves ?? true) ? 'checked' : '' }}>
                                    <label class="form-check-label" for="accrue_leaves">{{ __('app.accrue_leaves_label') }}</label>
                                </div>
                                <small class="text-muted">{{ __('app.accrue_leaves_info') }}</small>
                            </div>
                        </div>

                        <!-- ✅ جدول التراكم الشهري والسنوي -->
                        <div class="row mt-4">
                            <div class="col-12">
                                @php
                                    $leaveMonthlyDetails = $employee->calculateLeaveDeductionDetailsNew(now()->year, now()->month);
                                @endphp
                                
                                <!-- ✅ تشخيص المشاكل -->
                                @if(!$employee->hire_date || $employee->monthly_leave_days_allowed === null)
                                    <div class="alert alert-danger" role="alert">
                                        <i class="fas fa-exclamation-circle"></i>
                                        <strong>بيانات غير مكتملة!</strong>
                                        <p class="mb-2 mt-2">الموظف يحتاج إلى البيانات التالية لعرض جدول التراكم:</p>
                                        <ul class="mb-0">
                                            <li>@if(!$employee->hire_date)<span class="text-danger">❌ تاريخ التعيين (hire_date)</span>@else<span class="text-success">✅ تاريخ التعيين</span>@endif</li>
                                            <li>@if($employee->monthly_leave_days_allowed === null)<span class="text-danger">❌ عدد أيام الإجازة الشهرية المسموحة (monthly_leave_days_allowed)</span>@else<span class="text-success">✅ عدد أيام الإجازة الشهرية: {{ $employee->monthly_leave_days_allowed }}</span>@endif</li>
                                        </ul>
                                    </div>
                                @elseif(!$employee->isAccrualEnabled())
                                    <div class="alert alert-info" role="alert">
                                        <i class="fas fa-info-circle"></i>
                                        <strong>{{ __('employees.show.leave_accrual.accrual_disabled') ?? 'تراكم الإجازات معطل' }}</strong>
                                        <p class="mb-0 mt-2">{{ __('employees.show.leave_accrual.accrual_disabled_message') ?? 'هذا الموظف لا يراكم الإجازات الشهرية والسنوية. يجب تفعيل خاصية التراكم من بيانات الموظف.' }}</p>
                                    </div>
                                @elseif($leaveMonthlyDetails && count($leaveMonthlyDetails) > 0)
                                    <div class="card">
                                        <div class="card-header">
                                            <h5 class="card-title mb-0">{{ __('employees.show.leave_accrual.title') }}</h5>
                                        </div>
                                        <div class="card-body">
                                            <div class="table-responsive">
                                                <table class="table table-bordered table-sm">
                                                    <thead>
                                                        <tr>
                                                            <th colspan="9" class="bg-light text-center"><strong>{{ __('employees.show.leave_accrual.labels.monthly') }}</strong></th>
                                                            <th colspan="6" class="bg-light text-center"><strong>{{ __('employees.show.leave_accrual.labels.annual') }}</strong></th>
                                                        </tr>
                                                        <tr>
                                                            <th>{{ __('employees.show.leave_accrual.columns.year') }}</th>
                                                            <th>{{ __('employees.show.leave_accrual.columns.month') }}</th>
                                                            <!-- Monthly leave -->
                                                            <th>{{ __('employees.show.leave_accrual.columns.granted') }} <small class="d-block text-muted">(شهري)</small></th>
                                                            <th>{{ __('employees.show.leave_accrual.columns.used') }} <small class="d-block text-muted">(شهري)</small></th>
                                                            <th>{{ __('employees.show.leave_accrual.columns.balance_before') }} <small class="d-block text-muted">(شهري)</small></th>
                                                            <th>{{ __('employees.show.leave_accrual.columns.balance_after') }} <small class="d-block text-muted">(شهري)</small></th>
                                                            <th>{{ __('employees.show.leave_accrual.columns.excess') }} <small class="d-block text-muted">(شهري)</small></th>
                                                            <th>{{ __('employees.show.leave_accrual.columns.amount', ['currency' => get_currency_code()]) }} <small class="d-block text-muted">(شهري)</small></th>
                                                            <th>الإجمالي<br><small class="text-muted">{{ get_currency_code() }}</small></th>
                                                            <!-- Annual leave -->
                                                            <th>{{ __('employees.show.leave_accrual.columns.used') }} <small class="d-block text-muted">(سنوي)</small></th>
                                                            <th>{{ __('employees.show.leave_accrual.columns.balance_before') }} <small class="d-block text-muted">(سنوي)</small></th>
                                                            <th>{{ __('employees.show.leave_accrual.columns.balance_after') }} <small class="d-block text-muted">(سنوي)</small></th>
                                                            <th>{{ __('employees.show.leave_accrual.columns.excess') }} <small class="d-block text-muted">(سنوي)</small></th>
                                                            <th>{{ __('employees.show.leave_accrual.columns.amount', ['currency' => get_currency_code()]) }} <small class="d-block text-muted">(سنوي)</small></th>
                                                            <th>{{ __('employees.show.leave_accrual.columns.annual_entitlement') }}</th>
                                                        </tr>
                                                    </thead>
                                                    <tbody>
                                                        @foreach($leaveMonthlyDetails as $detail)
                                                            <tr>
                                                                <td>{{ $detail['year'] }}</td>
                                                                <td>{{ $detail['month_name'] ?? \Carbon\Carbon::create()->month($detail['month'])->translatedFormat('F') }}</td>
                                                                <!-- Monthly Leave Data -->
                                                                <td class="text-center">{{ number_format($detail['granted_monthly'] ?? 0, 2) }}</td>
                                                                <td class="text-center">{{ number_format($detail['used_monthly'] ?? 0, 2) }}</td>
                                                                <td class="text-center">{{ number_format($detail['balance_before_monthly'] ?? 0, 2) }}</td>
                                                                <td class="text-center">{{ number_format($detail['balance_after_monthly'] ?? 0, 2) }}</td>
                                                                <td class="text-center text-warning">{{ number_format($detail['excess_monthly'] ?? 0, 2) }}</td>
                                                                <td class="text-danger fw-bold">{{ format_currency($detail['deduction_amount_monthly'] ?? 0) }}</td>
                                                                <td class="text-danger fw-bold">{{ format_currency($detail['leave_deduction_amount'] ?? 0) }}</td>
                                                                <!-- Annual Leave Data -->
                                                                <td class="text-center">{{ number_format($detail['used_annual'] ?? 0, 2) }}</td>
                                                                <td class="text-center">{{ number_format($detail['balance_before_annual'] ?? 0, 2) }}</td>
                                                                <td class="text-center">{{ number_format($detail['balance_after_annual'] ?? 0, 2) }}</td>
                                                                <td class="text-center text-warning">{{ number_format($detail['excess_annual'] ?? 0, 2) }}</td>
                                                                <td class="text-danger fw-bold">{{ format_currency($detail['deduction_amount_annual'] ?? 0) }}</td>
                                                                <td>{{ number_format($employee->annual_entitlement ?? ($employee->monthly_leave_days_allowed * 12), 2) }}</td>
                                                            </tr>
                                                        @endforeach
                                                    </tbody>
                                                </table>
                                            </div>
                                        </div>
                                    </div>

                                    <!-- ✅ ملخص الخصومات للسنة الحالية -->
                                    @php
                                        $currentYear = now()->year;
                                        $currentYearMonthlyDeductions = $leaveMonthlyDetails
                                            ->filter(fn($d) => $d['year'] == $currentYear)
                                            ->sum('deduction_amount_monthly');
                                        $currentYearAnnualDeductions = $leaveMonthlyDetails
                                            ->filter(fn($d) => $d['year'] == $currentYear)
                                            ->sum('deduction_amount_annual');
                                        $totalCurrentYearDeductions = $currentYearMonthlyDeductions + $currentYearAnnualDeductions;
                                    @endphp
                                    <div class="alert alert-danger mt-3" role="alert">
                                        <h6 class="alert-heading">ملخص الخصومات للسنة الحالية ({{ $currentYear }})</h6>
                                        <hr>
                                        <div class="row">
                                            <div class="col-md-6">
                                                <p class="mb-2"><strong>إجمالي خصومات الإجازات الشهرية:</strong></p>
                                                <p class="fs-5 fw-bold text-danger">{{ format_currency($currentYearMonthlyDeductions) }}</p>
                                            </div>
                                            <div class="col-md-6">
                                                <p class="mb-2"><strong>إجمالي خصومات الإجازات السنوية:</strong></p>
                                                <p class="fs-5 fw-bold text-danger">{{ format_currency($currentYearAnnualDeductions) }}</p>
                                            </div>
                                        </div>
                                        <hr>
                                        <p class="mb-0"><strong>إجمالي الخصومات (شهري + سنوي):</strong></p>
                                        <p class="fs-4 fw-bold text-danger">{{ format_currency($totalCurrentYearDeductions) }}</p>
                                    </div>
                                @else
                                    <div class="alert alert-warning" role="alert">
                                        <i class="fas fa-exclamation-triangle"></i>
                                        <strong>{{ __('employees.show.leave_accrual.no_data') ?? 'لا توجد بيانات' }}</strong>
                                        <p class="mb-0 mt-2">لم يتم العثور على بيانات تراكمية للإجازات. تأكد من أن الموظف لديه فترة عمل كافية وأن التراكم مُفعّل.</p>
                                    </div>
                                @endif
                            </div>
                        </div>

                        <div class="row mt-2">
                            <div class="col-md-6 mb-3">
                                <div class="form-check">
                                    <input class="form-check-input" type="checkbox" value="1" id="deduct_if_underworked" name="deduct_if_underworked" {{ old('deduct_if_underworked', $employee->deduct_if_underworked ?? false) ? 'checked' : '' }}>
                                    <label class="form-check-label" for="deduct_if_underworked">{{ __('app.deduct_if_underworked_label') }}</label>
                                </div>
                                <small class="text-muted">{{ __('app.deduct_if_underworked_info') }}</small>
                            </div>

                            <div class="col-md-6 mb-3">
                                <div class="form-check">
                                    <input class="form-check-input" type="checkbox" value="1" id="overtime_paid" name="overtime_paid" {{ old('overtime_paid', $employee->overtime_paid ?? true) ? 'checked' : '' }}>
                                    <label class="form-check-label" for="overtime_paid">{{ __('app.overtime_paid_label') }}</label>
                                </div>
                                <small class="text-muted">{{ __('app.overtime_paid_info') }}</small>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-12 mb-3">
                                <label class="form-label">{{ __('app.weekly_off_days_label') }}</label>
                                <div class="d-flex gap-2 flex-wrap">
                                    @php $days = __('app.week_days'); @endphp
                                        <div class="mb-2 w-100">
                                            <select id="weekly_off_select" name="weekly_off_days[]" class="form-control select2" multiple="multiple" data-placeholder="{{ __('app.weekly_off_days_label') }}">
                                                @foreach($days as $key=>$label)
                                                    <option value="{{ $key }}" {{ in_array($key, old('weekly_off_days', $employee->weekly_off_days ?? [])) ? 'selected' : '' }}>{{ $label }}</option>
                                                @endforeach
                                            </select>
                                        </div>
                                </div>
                                <small class="text-muted">{{ __('app.weekly_off_days_info') }}</small>
                            </div>
                        </div>

                        <!-- ✅ مكونات الراتب الإضافية (الجديد - مثل create.blade.php) -->
                            <div class="col-md-12 mb-3">
                                <h5>{{ __('app.additional_salary_components') }} ({{ __('app.up_to') }} 7)</h5>
                                <!-- ✅ استخدام id "componentsContainer" ليطابق الكود الأصلي -->
                                <div id="componentsContainer">
                                    @for ($i = 1; $i <= 7; $i++)
                                        <div class="row mb-2 component-row align-items-center">
                                            <div class="col-md-5">
                                                <input type="text" class="form-control component-name" name="component_names[]" placeholder="{{ __('app.component_name') }}" maxlength="255"
                                                       value="{{ old('component_names.' . ($i-1), $employee->{"component_name_$i"}) }}">
                                            </div>
                                            <div class="col-md-5">
                                                <input type="number" class="form-control component-value" name="component_values[]" placeholder="{{ __('app.component_value') }}" step="0.01" min="0"
                                                       value="{{ old('component_values.' . ($i-1), $employee->{"component_$i"}) }}">
                                            </div>
                                            <div class="col-md-2">
                                                <button type="button" class="btn btn-sm btn-outline-danger remove-component">{{ __('app.remove') }}</button>
                                            </div>
                                        </div>
                                    @endfor
                                </div>
                                <button type="button" id="add-component-btn" class="btn btn-info btn-sm mb-3">{{ __('app.add_component') }}</button>
                            </div>
                            <!-- ✅ نهاية مكونات الراتب -->
                            <div class="col-md-6 mb-3">
                                <label for="birth_date" class="form-label">{{ __('app.birth_date') }}</label>
                                <input type="date" class="form-control" id="birth_date" name="birth_date"
                                       value="{{ old('birth_date', $employee->birth_date ? $employee->birth_date->format('Y-m-d') : '') }}"
                                       max="{{ date('Y-m-d') }}">
                                @error('birth_date')
                                    <div class="text-danger mt-1">{{ $message }}</div>
                                @enderror
                            </div>

                            <div class="col-md-6 mb-3">
                                <label for="age_display" class="form-label">{{ __('app.age') }} ({{ __('app.will_be_calculated') }})</label>
                                <input type="text" id="age_display" class="form-control" readonly
                                       value="{{ $employee->age ? $employee->age . ' ' . __('app.years') : '' }}">
                                <input type="hidden" name="age" id="age_hidden" value="{{ $employee->age }}">
                                @error('age')
                                    <div class="text-danger mt-1">{{ $message }}</div>
                                @enderror
                            </div>

                            <div class="col-md-6 mb-3">
                                <label for="qualification" class="form-label">{{ __('app.qualification') }}</label>
                                <input type="text" class="form-control" id="qualification" name="qualification"
                                       value="{{ old('qualification', $employee->qualification) }}">
                                @error('qualification')
                                    <div class="text-danger mt-1">{{ $message }}</div>
                                @enderror
                            </div>

                            <div class="col-md-6 mb-3">
                                <label for="national_id" class="form-label">{{ __('app.national_id') }}</label>
                                <input type="text" class="form-control" id="national_id" name="national_id"
                                       value="{{ old('national_id', $employee->national_id) }}">
                                @error('national_id')
                                    <div class="text-danger mt-1">{{ $message }}</div>
                                @enderror
                            </div>

                            <div class="col-md-6 mb-3">
                                <label for="address" class="form-label">{{ __('app.address') }}</label>
                                <textarea class="form-control" id="address" name="address" rows="2">{{ old('address', $employee->address) }}</textarea>
                                @error('address')
                                    <div class="text-danger mt-1">{{ $message }}</div>
                                @enderror
                            </div>

                            <div class="col-md-6 mb-3">
                                  <label for="status" class="form-label">{{ __('app.status_label') }}</label>
                                <select class="form-select" id="status" name="status">
                                    <option value="active" {{ old('status', $employee->status) == 'active' ? 'selected' : '' }}>{{ __('app.active') }}</option>
                                    <option value="inactive" {{ old('status', $employee->status) == 'inactive' ? 'selected' : '' }}>{{ __('app.inactive') }}</option>
                                    <option value="pending" {{ old('status', $employee->status) == 'pending' ? 'selected' : '' }}>{{ __('app.pending') }}</option>
                                </select>
                                @error('status')
                                    <div class="text-danger mt-1">{{ $message }}</div>
                                @enderror
                            </div>

                            <div class="col-md-6 mb-3">
                                <label for="photo" class="form-label">{{ __('app.photo') }}</label>
                                <input type="file" class="form-control" id="photo" name="photo" accept="image/*">
                                @error('photo')
                                    <div class="text-danger mt-1">{{ $message }}</div>
                                @enderror
                                @if($employee->photo)
                                    <div class="mt-2">
                                        <img src="{{ asset('storage/' . $employee->photo) }}" alt="{{ __('app.current_photo') }}" class="img-thumbnail" style="max-height: 100px;">
                                    </div>
                                @endif
                            </div>

                            <div class="col-md-6 mb-3">
                                <label for="cv_files" class="form-label">{{ __('app.cv_files') }} ({{ __('app.can_upload_multiple') }})</label>
                                <input type="file" class="form-control" id="cv_files" name="cv_files[]" accept=".pdf,.doc,.docx" multiple>
                                @error('cv_files')
                                    <div class="text-danger mt-1">{{ $message }}</div>
                                @enderror
                                @error('cv_files.*')
                                    <div class="text-danger mt-1">{{ $message }}</div>
                                @enderror
                                @if($employee->cv_files && count($employee->cv_files) > 0)
                                    <div class="mt-2">
                                        <p><strong>{{ __('app.current_files') }}:</strong></p>
                                        <ul class="list-unstyled">
                                            @foreach ($employee->cv_files as $file)
                                                <li class="mb-1">
                                                    <a href="{{ asset('storage/' . str_replace('storage/', '', $file)) }}" target="_blank" class="text-decoration-none text-primary">
                                                        <i class="fas fa-file-pdf me-2"></i> {{ basename($file) }}
                                                    </a>
                                                    <form action="{{ route('employees.remove.cv.file', $employee) }}" method="POST" class="d-inline" onsubmit="return confirm('{{ __('app.are_you_sure') }}')">
                                                        @csrf
                                                        <input type="hidden" name="file" value="{{ $file }}">
                                                        <button type="submit" class="btn btn-sm btn-danger ms-2">{{ __('app.remove') }}</button>
                                                    </form>
                                                </li>
                                            @endforeach
                                        </ul>
                                    </div>
                                @endif
                            </div>

                            <!-- Scheduled Work Hours Section -->
                            <hr class="my-4">
                            <h5 class="mb-3">{{ __('app.scheduled_work_hours') }}</h5>
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label for="scheduled_check_in" class="form-label">{{ __('app.scheduled_check_in') }}</label>
                                    <input type="time" class="form-control" id="scheduled_check_in" name="scheduled_check_in"
                                           value="{{ old('scheduled_check_in', \Carbon\Carbon::parse($employee->scheduled_check_in)->format('H:i') ?? '') }}">
                                    @error('scheduled_check_in')
                                        <div class="text-danger mt-1">{{ $message }}</div>
                                    @enderror
                                </div>

                                <div class="col-md-6 mb-3">
                                    <label for="scheduled_check_out" class="form-label">{{ __('app.scheduled_check_out') }}</label>
                                    <input type="time" class="form-control" id="scheduled_check_out" name="scheduled_check_out"
                                           value="{{ old('scheduled_check_out', \Carbon\Carbon::parse($employee->scheduled_check_out)->format('H:i') ?? '') }}">
                                    @error('scheduled_check_out')
                                        <div class="text-danger mt-1">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>

                        </div> <!-- End of Row -->

                        <!-- Submit Button -->
                        <div class="d-flex justify-content-end">
                            <a href="{{ route('employees.index') }}" class="btn btn-secondary me-2">{{ __('app.cancel') }}</a>
                            <button type="submit" class="btn btn-primary">{{ __('app.update') ?? __('app.save') }}</button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Modals for Adding Department and Position -->
@include('employees.partials.add-department-modal')
@include('employees.partials.add-position-modal')
@endsection

@section('scripts')
<script>
    document.addEventListener('DOMContentLoaded', function () {
        const birthDateInput = document.getElementById('birth_date');
        const ageDisplay = document.getElementById('age_display');
        const ageHidden = document.getElementById('age_hidden');
        // ✅ تصدير متغير $currency_name إلى JavaScript
        const currencyName = @json($currency_name);

        // ✅ دالة لحساب إجمالي الراتب (الأساسي + الحوافز + مكونات الراتب)
        function updateTotalSalary() {
            const basic = parseFloat(document.getElementById('salary').value) || 0;
            const incentives = parseFloat(document.getElementById('incentives').value) || 0;

            // حساب مجموع مكونات الراتب
            let componentsTotal = 0;
            // ✅ البحث عن عناصر component-value داخل container الصحيح
            const componentRows = document.querySelectorAll('#componentsContainer .component-value');
            componentRows.forEach(input => {
                const value = parseFloat(input.value) || 0;
                componentsTotal += value;
            });

            const total = basic + incentives + componentsTotal;
            // ✅ استخدام المتغير currencyName المُصدر من PHP
            document.getElementById('total_salary_display').value = total.toFixed(2) + ' ' + currencyName;
        }

        // ✅ دالة لإضافة مكون راتب (حقل نصي للاسم وحقل رقمي للقيمة) - مبنية على الكود الأصلي
        function addComponentRow(componentName = '', componentValue = '') {
            const container = document.getElementById('componentsContainer');
            const componentIndex = container.children.length + 1;
            if (componentIndex > 7) {
                alert('لا يمكن إضافة أكثر من 7 مكونات.');
                return;
            }

            const newRow = document.createElement('div');
            newRow.className = 'row component-row mb-2';

            newRow.innerHTML = `
                <div class="col-md-5">
                    <input type="text" name="component_names[]" class="form-control component-name" placeholder="اسم المكون" maxlength="255" value="${componentName}">
                </div>
                <div class="col-md-5">
                    <input type="number" step="0.01" name="component_values[]" class="form-control component-value" placeholder="القيمة" min="0" value="${componentValue}">
                </div>
                <div class="col-md-2">
                    <button type="button" class="btn btn-danger btn-sm remove-component">حذف</button>
                </div>
            `;
            container.appendChild(newRow);

            // ✅ ربط حدث "input" لحقل القيمة الجديد لتحديث المجموع
            newRow.querySelector('.component-value').addEventListener('input', updateTotalSalary);

            // ✅ إضافة مستمع الحدث للحذف إلى الزر الجديد
            newRow.querySelector('.remove-component').addEventListener('click', function () {
                newRow.remove();
                updateTotalSalary(); // تحديث المجموع بعد الحذف
            });

            // التأكد من أن الحد الأقصى هو 7 مكونات عند الإضافة
            if (container.children.length >= 7) {
                document.getElementById('add-component-btn').disabled = true;
            }
        }

        // ✅ إضافة مكون عند النقر على الزر
        document.getElementById('add-component-btn').addEventListener('click', function () {
            addComponentRow();
        });

        // ✅ تحميل المكونات الحالية من قاعدة البيانات عند تحميل الصفحة
        // استخدام json_encode لتمرير البيانات من PHP إلى JavaScript بشكل آمن
        const componentNames = [];
        const componentValues = [];
        @for ($i = 1; $i <= 7; $i++)
            @if ($employee->{"component_name_$i"} !== null || $employee->{"component_$i"} !== null)
                componentNames.push(@json($employee->{"component_name_$i"}));
                componentValues.push(@json($employee->{"component_$i"}));
            @endif
        @endfor

        // إذا كانت هناك مكونات، قم بإضافتها
        if (componentNames.length > 0) {
            for (let i = 0; i < componentNames.length; i++) {
                addComponentRow(componentNames[i], componentValues[i]);
            }
        } else {
            // إذا لم تكن هناك مكونات، قم بإضافة صف فارغ واحد على الأقل
            addComponentRow();
        }

        // ✅ ربط حدث "input" لحقول القيم التي تم تحميلها مسبقًا (من قاعدة البيانات)
        document.querySelectorAll('#componentsContainer .component-value').forEach(input => {
            input.addEventListener('input', updateTotalSalary);
        });

        // ✅ ربط حدث "click" لأزرار الحذف التي تم تحميلها مسبقًا
        document.querySelectorAll('#componentsContainer .remove-component').forEach(button => {
            button.addEventListener('click', function () {
                this.closest('.component-row').remove();
                updateTotalSalary(); // تحديث المجموع بعد الحذف
                // ✅ إعادة تمكين الزر إذا كان العدد أقل من 7 بعد الحذف
                if (document.getElementById('componentsContainer').children.length < 7) {
                    document.getElementById('add-component-btn').disabled = false;
                }
            });
        });

        // ✅ ربط حدث "input" للحقول الثابتة (الراتب الأساسي والحوافز) لتحديث المجموع
        document.getElementById('salary')?.addEventListener('input', updateTotalSalary);
        document.getElementById('incentives')?.addEventListener('input', updateTotalSalary);

        // ✅ تحديث المجموع عند التحميل (لحساب القيم الافتراضية أو القديمة)
        updateTotalSalary();

        // التأكد من أن الحد الأقصى هو 7 مكونات عند التحميل
        const container = document.getElementById('componentsContainer');
        if (container.children.length >= 7) {
            document.getElementById('add-component-btn').disabled = true;
        }

        // حساب العمر
        if (birthDateInput) {
            birthDateInput.addEventListener('change', function () {
                const birthDateValue = this.value;
                if (birthDateValue) {
                    const birthDate = new Date(birthDateValue);
                    const today = new Date();
                    let age = today.getFullYear() - birthDate.getFullYear();
                    const monthDiff = today.getMonth() - birthDate.getMonth();
                    if (monthDiff < 0 || (monthDiff === 0 && today.getDate() < birthDate.getDate())) {
                        age--;
                    }
                    ageDisplay.value = age + ' سنة';
                    ageHidden.value = age;
                } else {
                    ageDisplay.value = '';
                    ageHidden.value = '';
                }
            });
        }

    });

    function submitAddDepartment() {
        const nameInput = document.getElementById('new_department_name');
        const name = nameInput.value.trim();

        if (!name) {
            alert('يرجى إدخال اسم القسم.');
            return;
        }

        fetch('{{ route("departments.store") }}', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            },
            body: JSON.stringify({ name: name })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                const select = document.getElementById('department_id');
                const option = new Option(data.department.name, data.department.id);
                select.add(option);
                select.value = data.department.id;

                bootstrap.Modal.getInstance(document.getElementById('addDepartmentModal')).hide();
                alert(data.message);

                nameInput.value = '';
            } else {
                alert('حدث خطأ: ' + (data.message || 'فشل في إضافة القسم.'));
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('حدث خطأ أثناء إضافة القسم: ' + error.message);
        });
    }

    function submitAddPosition() {
        const nameInput = document.getElementById('new_position_name');
        const departmentSelect = document.getElementById('new_position_department_id');

        const name = nameInput.value.trim();
        const departmentId = departmentSelect.value;

        if (!name) {
            alert('يرجى إدخال اسم المنصب.');
            return;
        }

        fetch('{{ route("positions.store") }}', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            },
            body: JSON.stringify({ name: name, department_id: departmentId })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                const select = document.getElementById('position_id');
                const option = new Option(data.position.name, data.position.id);
                select.add(option);
                select.value = data.position.id;

                bootstrap.Modal.getInstance(document.getElementById('addPositionModal')).hide();
                alert(data.message);

                nameInput.value = '';
                departmentSelect.value = '';
            } else {
                alert('حدث خطأ: ' + (data.message || 'فشل في إضافة المنصب.'));
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('حدث خطأ أثناء إضافة المنصب: ' + error.message);
        });
    }
</script>
@push('scripts')
<link href="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css" rel="stylesheet" />
<script src="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js"></script>
<script>
    // Initialize Select2 for weekly off multi-select if available
    document.addEventListener('DOMContentLoaded', function () {
        try {
            if (window.jQuery && typeof window.jQuery.fn.select2 === 'function') {
                window.jQuery('#weekly_off_select').select2({ placeholder: '{{ __('app.weekly_off_days_label') }}', width: '100%' });
            }
        } catch (e) {
            // ignore
        }
    });
</script>
@endpush
@endsection