{{-- resources/views/attendance/index.blade.php --}}
@extends('layouts.app')

@section('content')
<div class="container-fluid">
    <!-- Page Title -->
    <div class="row">
        <div class="col-12">
            <div class="page-title-box d-flex align-items-center justify-content-between">
                <h4 class="mb-0">{{ __('attendance.page_title') }}</h4>
                <div class="page-title-right">
                    <ol class="breadcrumb m-0">
                        <li class="breadcrumb-item"><a href="{{ route('dashboard.index') }}">{{ __('attendance.breadcrumb_home') }}</a></li>
                        <li class="breadcrumb-item active">{{ __('attendance.breadcrumb_attendance') }}</li>
                        </div>
                    </div>
                </div>

    <!-- Action Buttons: Quick Check-In/Out, Show No-Attendance, Import -->
    <div class="row mb-3">
        <div class="col-12">
            <div class="d-flex flex-wrap gap-2 align-items-center">
                @can('create attendance')
                    <button type="button" class="btn btn-success btn-sm" data-bs-toggle="modal" data-bs-target="#checkInModal">
                        <i class="fas fa-sign-in-alt"></i> تسجيل حضور سريع
                    </button>
                    <button type="button" class="btn btn-danger btn-sm" data-bs-toggle="modal" data-bs-target="#checkOutModal">
                        <i class="fas fa-sign-out-alt"></i> تسجيل انصراف سريع
                    </button>
                @else
                    <button type="button" class="btn btn-success btn-sm" disabled>
                        <i class="fas fa-sign-in-alt"></i> تسجيل حضور سريع
                    </button>
                    <button type="button" class="btn btn-danger btn-sm" disabled>
                        <i class="fas fa-sign-out-alt"></i> تسجيل انصراف سريع
                    </button>
                @endcan

                <button type="button" class="btn btn-primary btn-sm" data-bs-toggle="modal" data-bs-target="#noAttendanceModal">
                    <i class="fas fa-list"></i> عرض قائمة الأشخاص الغائبين
                </button>

                @can('import', \App\Models\Attendance::class)
                    <a href="{{ route('attendance.import.form') }}" class="btn btn-outline-primary btn-sm">
                        <i class="fas fa-file-import"></i> استيراد حضور
                    </a>
                @endcan
            </div>
        </div>
    </div>

    <!-- Stats Cards -->
    <div class="row">
        <div class="col-md-3">
            <div class="card bg-primary text-white">
                <div class="card-body">
                    <div class="d-flex">
                        <div class="flex-grow-1">
                            <p class="text-truncate font-size-14 mb-2">{{ __('attendance.present_today') }}</p>
                            <h4 class="mb-0">{{ $presentToday ?? 0 }}</h4>
                        </div>
                        <div class="avatar-sm">
                            <span class="avatar-title bg-transparent rounded">
                                <i class="fas fa-calendar-check fa-2x"></i>
                            </span>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="col-md-3">
            <div class="card bg-success text-white">
                <div class="card-body">
                    <div class="d-flex">
                        <div class="flex-grow-1">
                            <p class="text-truncate font-size-14 mb-2">{{ __('attendance.absent_today') }}</p>
                            <h4 class="mb-0">{{ $absentWithoutLeaveCount ?? 0 }}</h4>
                            <small>{{ __('attendance.not_recorded') }}: {{ $notRecordedCount ?? 0 }} &middot; {{ __('attendance.on_approved_leave') }}: {{ $onLeaveCount ?? 0 }}</small>
                                
                        </div>
                        <div class="avatar-sm">
                            <span class="avatar-title bg-transparent rounded">
                                <i class="fas fa-calendar-times fa-2x"></i>
                            </span>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="col-md-3">
            <div class="card bg-warning text-white">
                <div class="card-body">
                    <div class="d-flex">
                        <div class="flex-grow-1">
                            <p class="text-truncate font-size-14 mb-2">{{ __('attendance.late_today') }}</p>
                            <h4 class="mb-0">{{ $lateToday ?? 0 }}</h4>
                        </div>
                        <div class="avatar-sm">
                            <span class="avatar-title bg-transparent rounded">
                                <i class="fas fa-clock fa-2x"></i>
                            </span>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="col-md-3">
            <div class="card bg-info text-white">
                <div class="card-body">
                    <div class="d-flex">
                        <div class="flex-grow-1">
                            <p class="text-truncate font-size-14 mb-2">{{ __('attendance.average_check_in') }}</p>
                            <h4 class="mb-0">{{ $averageCheckInTime ?? '00:00' }}</h4>
                        </div>
                        <div class="avatar-sm">
                            <span class="avatar-title bg-transparent rounded">
                                <i class="fas fa-chart-line fa-2x"></i>
                            </span>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Filters Section -->
    <div class="row mb-4">
        <div class="col-md-3">
            <input type="text" id="searchInput" class="form-control" placeholder="{{ __('attendance.search_placeholder') }}">
        </div>
        <div class="col-md-3">
            <select id="employeeFilter" class="form-select">
                <option value="">{{ __('attendance.all_employees') }}</option>
                @foreach($employees ?? [] as $employee)
                    <option value="{{ $employee->id }}">{{ $employee->name }} ({{ $employee->employee_id }})</option>
                @endforeach
            </select>
        </div>
        <div class="col-md-2">
            <select id="statusFilter" class="form-select">
                <option value="">{{ __('attendance.all_statuses') }}</option>
                <option value="present">{{ __('attendance.status_present') }}</option>
                <option value="absent">{{ __('attendance.status_absent') }}</option>
                <option value="late">{{ __('attendance.status_late') }}</option>
                <option value="early_leave">{{ __('attendance.status_early_leave') }}</option>
            </select>
        </div>
        <div class="col-md-2">
            <input type="date" id="dateFilter" class="form-control" placeholder="{{ __('attendance.date') }}">
        </div>
        <div class="col-md-2">
            <select id="monthFilter" class="form-select">
                <option value="">{{ __('attendance.all_months') }}</option>
                @for ($i = 1; $i <= 12; $i++)
                    <option value="{{ $i }}">{{ \Carbon\Carbon::createFromDate(null, $i, 1)->translatedFormat('F') }}</option>
                @endfor
            </select>
        </div>
        <div class="col-md-2">
            <select id="yearFilter" class="form-select">
                <option value="">{{ __('attendance.all_years') }}</option>
                @php
                    $currentYear = date('Y');
                    $startYear = $currentYear - 5;
                    $endYear = $currentYear + 1;
                @endphp
                @for ($y = $startYear; $y <= $endYear; $y++)
                    <option value="{{ $y }}">{{ $y }}</option>
                @endfor
            </select>
        </div>
        <div class="col-md-2 d-flex align-items-end">
            <button id="applyFilters" class="btn btn-primary w-100 me-1">
                <i class="fas fa-filter"></i> {{ __('attendance.apply_filters') }}
            </button>
            <button id="resetFilters" class="btn btn-outline-secondary w-100">
                <i class="fas fa-sync-alt"></i> {{ __('attendance.reset_filters') }}
            </button>
        </div>
    </div>

    <!-- Attendance Table -->
    <div class="row">
        <div class="col-12">
            <div class="card">
                <div class="card-body">
                    <h4 class="card-title mb-4">{{ __('attendance.attendance_log') }}</h4>
                    <div class="table-responsive">
                        <table class="table table-centered table-nowrap table-hover">
                            <thead class="thead-light">
                                <tr>
                                    <th>#</th>
                                    <th>{{ __('attendance.employee') }}</th>
                                    <th>{{ __('attendance.date') }}</th>
                                    <th>{{ __('attendance.scheduled_check_in') }}</th>
                                    <th>{{ __('attendance.scheduled_check_out') }}</th>
                                    <th>{{ __('attendance.actual_check_in') }}</th>
                                    <th>{{ __('attendance.actual_check_out') }}</th>
                                    <th>{{ __('attendance.work_duration') }}</th>
                                    <th>{{ __('attendance.status') }}</th>
                                    <th>{{ __('attendance.notes') }}</th>
                                    <th>{{ __('attendance.actions') }}</th>
                                </tr>
                            </thead>
                            <tbody>
                                @forelse($attendances as $attendance)
                                <tr>
                                    <td>{{ $attendance->id }}</td>
                                    <td>
                                        {{ $attendance->employee->name }}
                                        <br>
                                        <small class="text-muted">{{ $attendance->employee->department?->name ?? '-' }}</small>
                                    </td>
                                    <td>{{ $attendance->date->format('Y-m-d') }}</td>
                                    <td>{{ $attendance->scheduled_check_in ?? __('attendance.not_scheduled') }}</td>
                                    <td>{{ $attendance->scheduled_check_out ?? __('attendance.not_scheduled') }}</td>
                                    <td>{{ $attendance->check_in ?? __('attendance.not_checked_in') }}</td>
                                    <td>{{ $attendance->check_out ?? __('attendance.not_checked_out') }}</td>
                                    <td>{{ $attendance->work_duration ?? '00:00:00' }}</td>
                                    <td>
                                        @php
                                            $attDate = $attendance->date instanceof \Carbon\Carbon ? $attendance->date->toDateString() : (string)$attendance->date;
                                            $isHolidayRow = \App\Models\Holiday::whereDate('date', $attDate)->exists();
                                            $dayNameMap = [1=>'monday',2=>'tuesday',3=>'wednesday',4=>'thursday',5=>'friday',6=>'saturday',7=>'sunday'];
                                            $dayName = \Carbon\Carbon::parse($attDate)->dayOfWeekIso;
                                            $isWeeklyOffRow = in_array($dayNameMap[$dayName], (array)($attendance->employee->weekly_off_days ?? []));
                                            $displayStatus = $attendance->status;
                                            if ($attendance->status === 'present' && ($isHolidayRow || $isWeeklyOffRow)) {
                                                $displayStatus = 'present_on_holiday';
                                            }
                                        @endphp
                                        @if($displayStatus === 'present_on_holiday')
                                            <div>حاضر (يوم عطلة)</div>
                                        @else
                                            {{ __('attendance.status_' . ($attendance->status ?? 'absent')) }}
                                        @endif

                                        {{-- شارة توضح أن هذا الحضور مدفوع ليوم عطلة (إن وُجد) --}}
                                        @if(!empty($attendance->paid_for_off))
                                            <div class="mt-1"><span class="badge bg-success text-white" title="{{ __('attendance.paid_off_badge_tooltip') ?? 'حضور في يوم إجازة — مدفوع' }}">{{ __('attendance.paid_off_badge') ?? 'أجر حضور' }}</span></div>
                                        @endif
                                    </td>
                                    <td>
                                        @php
                                            $leaveKey = $attendance->employee_id . '|' . ($attendance->date instanceof \Carbon\Carbon ? $attendance->date->toDateString() : (string) $attendance->date);
                                            // prefer explicit relation if present
                                            $leaveObj = $attendance->leave ?? null;
                                            $leaveInfo = null;
                                            if ($leaveObj) {
                                                $leaveInfo = ['id' => $leaveObj->id, 'reason' => $leaveObj->reason ?? null, 'is_unpaid' => $leaveObj->is_unpaid_deduction_applied ?? false, 'leave_type' => $leaveObj->leave_type ?? null];
                                            } elseif (!empty($leavesMap[$leaveKey])) {
                                                $leaveInfo = $leavesMap[$leaveKey];
                                            }
                                        @endphp

                                        @if($leaveInfo)
                                            <div>
                                                @php
                                                    $badgeText = $leaveInfo['is_unpaid'] ? 'خصم مادي' : 'خصم من الرصيد';
                                                    $title = $leaveInfo['reason'] ? $leaveInfo['reason'] : ($leaveInfo['leave_type'] ?? '');
                                                @endphp
                                                <button
                                                    type="button"
                                                    class="badge bg-warning text-dark openLeaveModal"
                                                    data-leave-id="{{ $leaveInfo['id'] }}"
                                                    data-leave-reason="{{ e($leaveInfo['reason'] ?? '') }}"
                                                    data-leave-type="{{ e($leaveInfo['leave_type'] ?? '') }}"
                                                    data-leave-unpaid="{{ $leaveInfo['is_unpaid'] ? '1' : '0' }}"
                                                    title="{{ $title }}"
                                                >{{ $badgeText }}</button>
                                            </div>
                                        @endif

                                        @if($attendance->notes === __('attendance.quick_check_in') || $attendance->notes === 'تسجيل حضور سريع')
                                            {{ __('attendance.quick_check_in') }}
                                        @elseif($attendance->status === 'present' && strpos($attendance->notes ?? '', 'غياب بدون إذن') === false)
                                            {{-- لا تظهر ملاحظات الخصم للحاضرين الفعليين --}}
                                        @else
                                            {{ $attendance->notes ?? '-' }}
                                        @endif
                                    </td>
                                    <td>
                                        <a href="{{ route('attendance.show', $attendance) }}" class="btn btn-info btn-sm"><i class="fas fa-eye"></i></a>
                                        <a href="{{ route('attendance.edit', $attendance) }}" class="btn btn-warning btn-sm"><i class="fas fa-edit"></i></a>
                                        <form action="{{ route('attendance.destroy', $attendance) }}" method="POST" class="d-inline">
                                            @csrf @method('DELETE')
                                            <button type="submit" class="btn btn-danger btn-sm"><i class="fas fa-trash"></i></button>
                                        </form>
                                    </td>
                                </tr>
                                @empty
                                <tr>
                                    <td colspan="11" class="text-center">{{ __('attendance.no_records') }}</td>
                                </tr>
                                @endforelse
                            </tbody>
                        </table>
                        {{ $attendances->links() }}
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Check-In Modal -->
<div class="modal fade" id="checkInModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <form id="quickCheckInForm">
                @csrf
                <div class="modal-header">
                    <h5 class="modal-title">{{ __('attendance.quick_check_in') }}</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="mb-3">
                        <label for="employee_id_in" class="form-label">{{ __('attendance.employee') }}</label>
                        <select class="form-control" name="employee_id" id="employee_id_in" required>
                            <option value="">{{ __('attendance.select_employee') }}</option>
                            @foreach($employees as $emp)
                                <option value="{{ $emp->id }}"
                                    data-checkin="{{ $emp->scheduled_check_in ?? '09:00' }}"
                                    data-checkout="{{ $emp->scheduled_check_out ?? '17:00' }}">
                                    {{ $emp->name }}
                                </option>
                            @endforeach
                        </select>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">{{ __('attendance.scheduled_check_in') }}</label>
                        <input type="time" class="form-control" id="scheduled_check_in_in" name="scheduled_check_in" value="09:00">
                    </div>
                    <div class="mb-3">
                        <label class="form-label">{{ __('attendance.scheduled_check_out') }}</label>
                        <input type="time" class="form-control" id="scheduled_check_out_in" name="scheduled_check_out" value="17:00">
                    </div>
                    <div class="mb-3">
                        <label class="form-label">{{ __('attendance.actual_check_in') }}</label>
                        <input type="time" class="form-control" name="check_in" value="{{ now()->format('H:i') }}" required>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">{{ __('attendance.date') }}</label>
                        <input type="date" class="form-control" name="date" value="{{ now()->format('Y-m-d') }}" required>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">{{ __('attendance.cancel') }}</button>
                    <button type="submit" class="btn btn-success">{{ __('attendance.submit_check_in') }}</button>
                </div>
            </form>
        </div>
    </div>
</div>

    <!-- Modal: Employees without attendance -->
    <div class="modal fade" id="noAttendanceModal" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">{{ __('attendance.no_attendance_modal_title') }}</h5>
                    <div class="ms-3">
                        <label class="form-label small mb-1">{{ __('attendance.choose_date') }}</label>
                        <input type="date" id="noAttendanceDate" class="form-control form-control-sm" value="{{ now()->format('Y-m-d') }}">
                    </div>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="إغلاق"></button>
                </div>
                <div class="modal-body">
                    <div class="table-responsive">
                        <table class="table table-striped">
                            <thead>
                                <tr>
                                    <th>#</th>
                                    <th>الاسم</th>
                                    <th>الحالة</th>
                                </tr>
                            </thead>
                            <tbody>
                                {{-- dynamic rows will be injected here by JS for the selected date --}}
                                <tr>
                                    <td colspan="3" class="text-center">جارٍ التحميل...</td>
                                </tr>
                            </tbody>
                        </table>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">إغلاق</button>
                </div>
            </div>
        </div>
    </div>

<!-- Check-Out Modal -->
<div class="modal fade" id="checkOutModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <form id="quickCheckOutForm">
                @csrf
                <div class="modal-header">
                    <h5 class="modal-title">{{ __('attendance.quick_check_out') }}</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="mb-3">
                        <label for="employee_id_out" class="form-label">{{ __('attendance.employee') }}</label>
                        <select class="form-control" name="employee_id" id="employee_id_out" required>
                            <option value="">{{ __('attendance.select_employee') }}</option>
                            @foreach($employees as $emp)
                                <option value="{{ $emp->id }}"
                                    data-checkin="{{ $emp->scheduled_check_in ?? '09:00' }}"
                                    data-checkout="{{ $emp->scheduled_check_out ?? '17:00' }}">
                                    {{ $emp->name }}
                                </option>
                            @endforeach
                        </select>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">{{ __('attendance.scheduled_check_in') }}</label>
                        <input type="time" class="form-control" id="scheduled_check_in_out" name="scheduled_check_in" value="09:00">
                    </div>
                    <div class="mb-3">
                        <label class="form-label">{{ __('attendance.scheduled_check_out') }}</label>
                        <input type="time" class="form-control" id="scheduled_check_out_out" name="scheduled_check_out" value="17:00">
                    </div>
                    <div class="mb-3">
                        <label class="form-label">{{ __('attendance.actual_check_out') }}</label>
                        <input type="time" class="form-control" name="check_out" value="{{ now()->format('H:i') }}" required>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">{{ __('attendance.date') }}</label>
                        <input type="date" class="form-control" name="date" value="{{ now()->format('Y-m-d') }}" max="{{ now()->format('Y-m-d') }}" required>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">{{ __('attendance.cancel') }}</button>
                    <button type="submit" class="btn btn-danger">{{ __('attendance.submit_check_out') }}</button>
                </div>
            </form>
        </div>
    </div>
</div>

@endsection

<!-- Leave Detail Modal (opened from attendance badge) -->
<div class="modal fade" id="leaveDetailModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">{{ __('attendance.leave_summary') ?? 'ملخص الإجازة' }}</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="إغلاق"></button>
            </div>
            <div class="modal-body">
                <p><strong>{{ __('attendance.leave_id') ?? 'معرف الإجازة' }}:</strong> <span id="leaveDetailId">-</span></p>
                <p><strong>{{ __('attendance.leave_type') ?? 'نوع الإجازة' }}:</strong> <span id="leaveDetailType">-</span></p>
                <p><strong>{{ __('attendance.leave_reason') ?? 'السبب' }}:</strong></p>
                <div id="leaveDetailReason" class="small text-muted">-</div>
            </div>
            <div class="modal-footer">
                <a href="#" id="leaveDetailGo" class="btn btn-primary">{{ __('attendance.go_to_details') ?? 'اذهب للتفاصيل' }}</a>
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">{{ __('attendance.close') ?? 'إغلاق' }}</button>
            </div>
        </div>
    </div>
</div>

@push('scripts')
<script>
document.addEventListener('DOMContentLoaded', function() {
    // --- وظيفة لربط اختيار الموظف بتحديث الأوقات المجدولة ---
    function bindEmployeeSelect(selectId, checkInId, checkOutId, modalId) {
        const select = document.getElementById(selectId);
        const checkInField = document.getElementById(checkInId);
        const checkOutField = document.getElementById(checkOutId);
        const modal = document.getElementById(modalId);

        if (!select) return;

        function updateFields() {
            const opt = select.options[select.selectedIndex];
            if (opt) {
                checkInField.value = opt.getAttribute('data-checkin') || '09:00';
                checkOutField.value = opt.getAttribute('data-checkout') || '17:00';
            }
        }

        // عند تغيير الموظف
        select.addEventListener('change', updateFields);

        // عند فتح الـ modal
        if (modal) {
            modal.addEventListener('show.bs.modal', function () {
                updateFields();
            });
        }
    }

    bindEmployeeSelect('employee_id_in', 'scheduled_check_in_in', 'scheduled_check_out_in', 'checkInModal');
    bindEmployeeSelect('employee_id_out', 'scheduled_check_in_out', 'scheduled_check_out_out', 'checkOutModal');

    // --- وظيفة لمعالجة نموذج تسجيل الحضور ---
    const quickCheckInForm = document.getElementById('quickCheckInForm');
    if (quickCheckInForm) {
        quickCheckInForm.addEventListener('submit', function(e) {
            e.preventDefault(); // منع الإرسال الافتراضي

            const formData = new FormData(quickCheckInForm);

            fetch('{{ route("attendance.quick-check-in") }}', {
                method: 'POST',
                headers: {
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
                    'Accept': 'application/json'
                },
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    // إغلاق الـ Modal
                    const modalInstance = bootstrap.Modal.getInstance(document.getElementById('checkInModal'));
                    modalInstance.hide();
                    quickCheckInForm.reset(); // مسح النموذج

                    // عرض رسالة نجاح (يمكن استخدام SweetAlert أو Bootstrap Alert)
                    alert(data.message || 'تم تسجيل الحضور بنجاح!');

                    // تحديث الصفحة أو جدول الحضور (اختياري)
                    location.reload();
                } else {
                    alert(data.message || 'حدث خطأ أثناء تسجيل الحضور.');
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('حدث خطأ غير متوقع أثناء تسجيل الحضور.');
            });
        });
    }

    // --- وظيفة لمعالجة نموذج تسجيل الإنصراف ---
    const quickCheckOutForm = document.getElementById('quickCheckOutForm');
    if (quickCheckOutForm) {
        quickCheckOutForm.addEventListener('submit', function(e) {
            e.preventDefault(); // منع الإرسال الافتراضي

            const formData = new FormData(quickCheckOutForm);

            fetch('{{ route("attendance.quick-check-out") }}', {
                method: 'POST',
                headers: {
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
                    'Accept': 'application/json'
                },
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    // إغلاق الـ Modal
                    const modalInstance = bootstrap.Modal.getInstance(document.getElementById('checkOutModal'));
                    modalInstance.hide();
                    quickCheckOutForm.reset(); // مسح النموذج

                    // عرض رسالة نجاح (يمكن استخدام SweetAlert أو Bootstrap Alert)
                    alert(data.message || 'تم تسجيل الإنصراف بنجاح!');

                    // تحديث الصفحة أو جدول الحضور (اختياري)
                    location.reload();
                } else {
                    alert(data.message || 'حدث خطأ أثناء تسجيل الإنصراف.');
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('حدث خطأ غير متوقع أثناء تسجيل الإنصراف.');
            });
        });
    }

    // --- وظيفة لفلترة الجدول باستخدام GET parameters ---
    const searchInput = document.getElementById('searchInput');
    const employeeFilter = document.getElementById('employeeFilter');
    const statusFilter = document.getElementById('statusFilter');
    const dateFilter = document.getElementById('dateFilter');
    const monthFilter = document.getElementById('monthFilter');
    const yearFilter = document.getElementById('yearFilter');
    const applyFiltersBtn = document.getElementById('applyFilters');
    const resetFiltersBtn = document.getElementById('resetFilters');

    function applyFilters() {
        // بناء URL مع المعلمات
        let url = new URL(window.location.href);
        url.search = ''; // مسح المعلمات الحالية

        if (searchInput.value) url.searchParams.append('search', searchInput.value);
        if (employeeFilter.value) url.searchParams.append('employee_id', employeeFilter.value);
        if (statusFilter.value) url.searchParams.append('status', statusFilter.value);
        if (dateFilter.value) url.searchParams.append('date', dateFilter.value);
        if (monthFilter.value) url.searchParams.append('month', monthFilter.value);
        if (yearFilter.value) url.searchParams.append('year', yearFilter.value);

        window.location.href = url.toString();
    }

    function resetFilters() {
        searchInput.value = '';
        employeeFilter.value = '';
        statusFilter.value = '';
        dateFilter.value = '';
        monthFilter.value = '';
        yearFilter.value = '';
        // إعادة تحميل الصفحة بدون معلمات الفلاتر
        window.location.href = window.location.pathname + '?page=1'; // الحفاظ على الترقيم
    }

    applyFiltersBtn?.addEventListener('click', applyFilters);
    resetFiltersBtn?.addEventListener('click', resetFilters);

    // تعيين القيم من URL إلى الحقول عند التحميل (اختياري)
    const urlParams = new URLSearchParams(window.location.search);
    searchInput.value = urlParams.get('search') || '';
    employeeFilter.value = urlParams.get('employee_id') || '';
    statusFilter.value = urlParams.get('status') || '';
    dateFilter.value = urlParams.get('date') || '';
    monthFilter.value = urlParams.get('month') || '';
    yearFilter.value = urlParams.get('year') || '';
    // ---- Function to fetch and render employees without attendance for a given date ----
    async function fetchNoAttendance(date) {
        const tbody = document.querySelector('#noAttendanceModal table tbody');
        tbody.innerHTML = '<tr><td colspan="3" class="text-center">جارٍ التحميل...</td></tr>';

        try {
            const url = new URL('{{ route("attendance.no-attendance") }}', window.location.origin);
            url.searchParams.set('date', date);
            const res = await fetch(url.toString(), { headers: { 'Accept': 'application/json' } });
            let json = null;
            try {
                json = await res.json();
            } catch (e) {
                // non-JSON response
            }

            if (!res.ok) {
                const msg = (json && json.message) ? json.message : ('خطأ في الخادم (' + res.status + ')');
                tbody.innerHTML = '<tr><td colspan="3" class="text-center">' + msg + '</td></tr>';
                return;
            }

            if (!json || !json.list || json.list.length === 0) {
                tbody.innerHTML = '<tr><td colspan="3" class="text-center">لا يوجد موظفون بدون تسجيل لهذا التاريخ.</td></tr>';
                return;
            }

                const rows = json.list.map(person => {
                // Priority: approved leave -> holiday -> weekly off -> absent
                if (person.on_leave) {
                    let reasonHtml = person.leave_reason ? `<div class="small text-muted mt-1">${person.leave_reason}</div>` : '';
                    let leaveLink = person.leave_id ? `<div class="mt-1"><a href="/leaves/${person.leave_id}">{{ addslashes(__('attendance.view_leave_details')) }}</a></div>` : '';
                    return `
                        <tr>
                            <td>${person.id}</td>
                            <td><a href="/employees/${person.id}/edit">${person.name}</a><br><small class="text-muted">${person.department || '-'}</small></td>
                            <td>
                                <div><span class="badge bg-warning text-dark">{{ addslashes(__('attendance.approved_leave_badge')) }}</span></div>
                                ${reasonHtml}
                                ${leaveLink}
                            </td>
                        </tr>`;
                }

                if (person.is_holiday) {
                    let holidayHtml = person.holiday_name ? `<div class="small text-muted mt-1">${person.holiday_name}</div>` : '';
                    return `
                        <tr>
                            <td>${person.id}</td>
                            <td><a href="/employees/${person.id}/edit">${person.name}</a><br><small class="text-muted">${person.department || '-'}</small></td>
                            <td>
                                <div><span class="badge bg-info text-white">{{ addslashes(__('attendance.official_holiday_badge')) }}</span></div>
                                ${holidayHtml}
                            </td>
                        </tr>`;
                }

                if (person.is_weekly_off) {
                    return `
                        <tr>
                            <td>${person.id}</td>
                            <td><a href="/employees/${person.id}/edit">${person.name}</a><br><small class="text-muted">${person.department || '-'}</small></td>
                            <td>
                                <div><span class="badge bg-secondary text-white">{{ addslashes(__('attendance.weekly_off_badge')) }}</span></div>
                            </td>
                        </tr>`;
                }

                // default: absent without leave
                return `
                    <tr>
                        <td>${person.id}</td>
                        <td><a href="/employees/${person.id}/edit">${person.name}</a><br><small class="text-muted">${person.department || '-'}</small></td>
                        <td>
                                <div class="d-flex align-items-center">
                                <span class="badge bg-danger">{{ addslashes(__('attendance.absent_without_leave_badge')) }}</span>
                                <button class="btn btn-sm btn-outline-primary ms-2 sendReminderBtn" data-employee-id="${person.id}">{{ addslashes(__('attendance.send_reminder')) }}</button>
                                <div class="ms-2">
                                    <select id="deduct_kind_${person.id}" class="form-select form-select-sm">
                                        <option value="monthly">{{ addslashes(__('attendance.deduct_monthly_option')) }}</option>
                                        <option value="annual">{{ addslashes(__('attendance.deduct_annual_option')) }}</option>
                                    </select>
                                </div>
                                <button class="btn btn-sm btn-outline-danger ms-2 markLeaveBtn" data-employee-id="${person.id}">{{ addslashes(__('attendance.grant_unauthorized_leave')) }}</button>
                            </div>
                        </td>
                    </tr>`;
            }).join('');

            tbody.innerHTML = rows;

            // attach handlers for dynamic buttons
            tbody.querySelectorAll('.sendReminderBtn').forEach(function(btn) {
                btn.addEventListener('click', function() {
                    const employeeId = this.getAttribute('data-employee-id');
                    if (!employeeId) return;
                    if (!confirm('{{ addslashes(__('app.confirm_send_attendance_reminder')) }}')) return;
                    fetch('{{ route("attendance.notify-missing") }}', {
                        method: 'POST',
                        headers: {
                            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
                            'Accept': 'application/json',
                            'Content-Type': 'application/json'
                        },
                        body: JSON.stringify({ employee_id: employeeId, date: date })
                    }).then(r => r.json()).then(j => { alert(j.message || (j.success ? 'تم إرسال التذكير.' : 'تعذر إرسال التذكير.')); })
                    .catch(err => { console.error(err); alert('حدث خطأ أثناء محاولة إرسال التذكير.'); });
                });
            });

            tbody.querySelectorAll('.markLeaveBtn').forEach(function(btn) {
                btn.addEventListener('click', function() {
                    const employeeId = this.getAttribute('data-employee-id');
                    if (!employeeId) return;
                    const select = document.getElementById('deduct_kind_' + employeeId);
                    const kind = select ? select.value : 'monthly';
                    if (!confirm('{{ addslashes(__('app.confirm_grant_unauthorized_leave')) }}')) return;

                    fetch('/attendance/' + employeeId + '/mark-unauthorized-leave', {
                        method: 'POST',
                        headers: {
                            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
                            'Accept': 'application/json',
                            'Content-Type': 'application/json'
                        },
                        body: JSON.stringify({ kind: kind, date: date })
                    })
                    .then(res => res.json())
                    .then(json => {
                        if (json.success) {
                            alert(json.message || 'تمت العملية بنجاح');
                            // refresh the list for the same date
                            fetchNoAttendance(date);
                        } else {
                            alert(json.message || 'فشل تنفيذ العملية');
                        }
                    })
                    .catch(err => {
                        console.error(err);
                        alert('حدث خطأ أثناء تنفيذ الطلب');
                    });
                });
            });

        } catch (err) {
            console.error(err);
            document.querySelector('#noAttendanceModal table tbody').innerHTML = '<tr><td colspan="3" class="text-center">حدث خطأ أثناء جلب البيانات.</td></tr>';
        }
    }

    // when modal opens, fetch list for selected date
    const noAttendanceModalEl = document.getElementById('noAttendanceModal');
    const noAttendanceDateInput = document.getElementById('noAttendanceDate');
    if (noAttendanceModalEl) {
        noAttendanceModalEl.addEventListener('show.bs.modal', function () {
            const d = noAttendanceDateInput ? noAttendanceDateInput.value : new Date().toISOString().split('T')[0];
            fetchNoAttendance(d);
        });
    }

    // when date changes, refresh list
    noAttendanceDateInput?.addEventListener('change', function() {
        fetchNoAttendance(this.value);
    });

    // Open leave detail modal when a badge/button with class `.openLeaveModal` is clicked.
    // Use event delegation so dynamically-added buttons (via AJAX) also work.
    document.body.addEventListener('click', function(ev) {
        const el = ev.target.closest ? ev.target.closest('.openLeaveModal') : null;
        if (!el) return;

        ev.preventDefault();
        const id = el.getAttribute('data-leave-id');
        const reason = el.getAttribute('data-leave-reason') || '-';
        const type = el.getAttribute('data-leave-type') || '-';
        const go = document.getElementById('leaveDetailGo');
        const idEl = document.getElementById('leaveDetailId');
        const reasonEl = document.getElementById('leaveDetailReason');
        const typeEl = document.getElementById('leaveDetailType');

        if (idEl) idEl.textContent = id || '-';
        if (reasonEl) reasonEl.textContent = reason || '-';
        if (typeEl) typeEl.textContent = type || '-';
        if (go) go.setAttribute('href', '/leaves/' + id);

        const modalEl = document.getElementById('leaveDetailModal');
        if (modalEl) {
            const modal = new bootstrap.Modal(modalEl);
            modal.show();
        }
    });
});
</script>
@endpush