<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\User;
use App\Models\Employee; // إذا كنت ترغب في ربط المستخدم بموظف
use App\Models\Role;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str; // لدعم توليد النصوص العشوائية

class UserSeeder extends Seeder
{
    public function run(): void
    {
        $perRole = (int) env('USER_SEED_PER_ROLE', 10); // default to 10 per role for safer seeding

        // جلب الأدوار (تأكد من أن RoleSeeder شُغل قبل هذا)
        $adminRole = Role::where('name', 'admin')->first();
        $hrRole = Role::where('name', 'hr')->first();
        $employeeRole = Role::where('name', 'employee')->first();

        if (!$adminRole || !$hrRole || !$employeeRole) {
            $this->command->error('بعض الأدوار المطلوبة (admin, hr, employee) غير موجودة. تأكد من تشغيل RoleSeeder أولاً.');
            return;
        }

        $usersToCreate = [];

        // --- إنشاء $perRole مستخدم Super Admin ---
        for ($i = 1; $i <= $perRole; $i++) {
            $usersToCreate[] = [
                'name' => "Super Admin {$i}",
                'email' => "admin{$i}@company.com",
                'password' => Hash::make('password'),
                'employee_id' => null, // أو ربط بموظف لاحقًا
                'status' => 'active',
                'permissions' => [],
                'role' => $adminRole->name,
            ];
        }

        // --- إنشاء $perRole مستخدم HR ---
        for ($i = 1; $i <= $perRole; $i++) {
            $usersToCreate[] = [
                'name' => "HR User {$i}",
                'email' => "hr{$i}@company.com",
                'password' => Hash::make('password'),
                'employee_id' => null,
                'status' => 'active',
                'permissions' => [
                    'view_employees',
                    'create_employees',
                    'edit_employees',
                    'view_leaves',
                    'create_leaves',
                    'edit_leaves',
                    'approve_leaves',
                    'reject_leaves',
                    'manage_leaves',
                    'view_attendance',
                    'create_attendance',
                    'edit_attendance',
                    'view_loans',
                    'create_loans',
                    'edit_loans',
                    'approve_loans',
                    'view_deductions',
                    'create_deductions',
                    'edit_deductions',
                    'view_reports',
                    'view_payrolls',
                ],
                'role' => $hrRole->name,
            ];
        }

        // --- إنشاء $perRole مستخدم Employee ---
        for ($i = 1; $i <= $perRole; $i++) {
            $usersToCreate[] = [
                'name' => "Employee {$i}",
                'email' => "employee{$i}@company.com",
                'password' => Hash::make('password'),
                'employee_id' => null,
                'status' => 'active',
                'permissions' => [
                    'view_own_profile',
                    'request_leave',
                    'view_own_payroll',
                    'view_own_attendance',
                ],
                'role' => $employeeRole->name,
            ];
        }

        DB::transaction(function () use ($usersToCreate) {
            foreach ($usersToCreate as $userData) {
                $user = User::firstOrCreate(
                    ['email' => $userData['email']], // شرط البحث
                    [
                        'name' => $userData['name'],
                        'password' => $userData['password'],
                        'employee_id' => $userData['employee_id'],
                        'status' => $userData['status'],
                        'permissions' => $userData['permissions'],
                    ]
                );

                // ربط المستخدم بالدور
                $user->roles()->sync([$userData['role']->id]);
            }
        });

        $total = $perRole * 3;
        $this->command->info("✅ تم إنشاء/تأكيد {$total} مستخدمين ({$perRole} لكل دور) بنجاح.");
    }
}