<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;

class RolesAndPermissionsSeeder extends Seeder
{
    public function run()
    {
        // Configuration: strict trimming only when ROLE_SEEDER_APPLY_TRIM env is true.
        // By default the seeder will perform a dry-run and only report removals.
        $applyTrim = env('ROLE_SEEDER_APPLY_TRIM', false);
        $limit = (int) env('ROLE_SEEDER_LIMIT', 10);

        // Helper: pivot tables that may exist in this project
        $pivotTables = [];
        if (\Schema::hasTable('user_role')) $pivotTables[] = 'user_role';
        if (\Schema::hasTable('user_roles')) $pivotTables[] = 'user_roles';

        // --- Merge legacy role name 'super-admin' into canonical 'super_admin' if present ---
        $legacy = DB::table('roles')->where('name', 'super-admin')->first();
        $canonical = DB::table('roles')->where('name', 'super_admin')->first();
        if ($legacy && ! $canonical) {
            // Rename legacy role
            DB::table('roles')->where('id', $legacy->id)->update(['name' => 'super_admin']);
            $this->command?->info("Renamed role 'super-admin' (id={$legacy->id}) to 'super_admin'.");
        } elseif ($legacy && $canonical) {
            // Move pivots from legacy to canonical, avoiding duplicates
            foreach (['role_permission'] as $rp) {
                if (DB::getSchemaBuilder()->hasTable($rp)) {
                    $rows = DB::table($rp)->where('role_id', $legacy->id)->get();
                    foreach ($rows as $row) {
                        $exists = DB::table($rp)->where('role_id', $canonical->id)->where('permission_id', $row->permission_id)->exists();
                        if (! $exists) {
                            DB::table($rp)->insert(['role_id' => $canonical->id, 'permission_id' => $row->permission_id]);
                        }
                    }
                    DB::table($rp)->where('role_id', $legacy->id)->delete();
                }
            }

            foreach ($pivotTables as $pivot) {
                $rows = DB::table($pivot)->where('role_id', $legacy->id)->get();
                foreach ($rows as $row) {
                    $exists = DB::table($pivot)->where('role_id', $canonical->id)->where('user_id', $row->user_id)->exists();
                    if (! $exists) {
                        DB::table($pivot)->where('user_id', $row->user_id)->where('role_id', $legacy->id)->update(['role_id' => $canonical->id]);
                    } else {
                        // duplicate exists, remove legacy link
                        DB::table($pivot)->where('user_id', $row->user_id)->where('role_id', $legacy->id)->delete();
                    }
                }
            }

            DB::table('roles')->where('id', $legacy->id)->delete();
            $this->command?->info("Merged 'super-admin' into existing 'super_admin' (id={$canonical->id}).");
        }
        // Roles
        $roles = ['super_admin', 'admin', 'hr', 'employee'];
        foreach ($roles as $role) {
            DB::table('roles')->updateOrInsert(['name' => $role], ['name' => $role]);
        }

        // Permissions
        $permissions = [
            'manage_users',
            'manage_roles',
            'manage_permissions',
            'view_payroll',
            'create_payroll',
            'edit_payroll',
            'delete_payroll',
            'view_employees',
            'edit_employees',
            'delete_employees',
            'view_attendance',
            'edit_attendance',
            'view_leaves',
            'approve_leaves',
            'request_leaves',
        ];
        foreach ($permissions as $permission) {
            DB::table('permissions')->updateOrInsert(['name' => $permission], ['name' => $permission]);
        }

        // Get role IDs
        $superAdminRoleId = DB::table('roles')->where('name', 'super_admin')->value('id');
        $adminRoleId = DB::table('roles')->where('name', 'admin')->value('id');
        $hrRoleId = DB::table('roles')->where('name', 'hr')->value('id');
        $employeeRoleId = DB::table('roles')->where('name', 'employee')->value('id');

        // Assign all permissions to super_admin and admin
        foreach ($permissions as $permission) {
            $permissionId = DB::table('permissions')->where('name', $permission)->value('id');
            DB::table('role_permission')->updateOrInsert(['role_id' => $superAdminRoleId, 'permission_id' => $permissionId], []);
            DB::table('role_permission')->updateOrInsert(['role_id' => $adminRoleId, 'permission_id' => $permissionId], []);
        }

        // HR: logical permissions
        $hrPermissions = ['view_payroll', 'view_employees', 'view_attendance', 'view_leaves', 'approve_leaves'];
        foreach ($hrPermissions as $permission) {
            $permissionId = DB::table('permissions')->where('name', $permission)->value('id');
            DB::table('role_permission')->updateOrInsert(['role_id' => $hrRoleId, 'permission_id' => $permissionId], []);
        }

        // Employee: limited permissions
        $employeePermissions = ['view_payroll', 'view_employees', 'view_attendance', 'request_leaves'];
        foreach ($employeePermissions as $permission) {
            $permissionId = DB::table('permissions')->where('name', $permission)->value('id');
            DB::table('role_permission')->updateOrInsert(['role_id' => $employeeRoleId, 'permission_id' => $permissionId], []);
        }

        // Ensure exactly $limit users for each role (create missing, optionally trim excess)
        foreach ($roles as $role) {
            $roleId = DB::table('roles')->where('name', $role)->value('id');

            // Ensure standard users exist and are attached
            for ($i = 1; $i <= $limit; $i++) {
                $email = $role . $i . '@example.com';
                $existingUser = DB::table('users')->where('email', $email)->first();

                if (! $existingUser) {
                    $userId = DB::table('users')->insertGetId([
                        'name' => ucfirst($role) . ' ' . $i,
                        'email' => $email,
                        'password' => Hash::make('password'),
                        'remember_token' => Str::random(10),
                    ]);
                    $this->command?->info("Created user {$email} (id={$userId})");
                } else {
                    $userId = $existingUser->id;
                }

                // Attach in all existing pivot tables
                foreach ($pivotTables as $pivot) {
                    DB::table($pivot)->updateOrInsert(['user_id' => $userId, 'role_id' => $roleId], []);
                }
            }

            // Collect current unique assigned users across pivot tables
            $assigned = [];
            foreach ($pivotTables as $pivot) {
                $hasId = \Schema::hasColumn($pivot, 'id');
                $query = DB::table($pivot)->where('role_id', $roleId)->join('users', $pivot . '.user_id', '=', 'users.id')->select($pivot . '.user_id', 'users.email');
                if ($hasId) {
                    $query->addSelect($pivot . '.id as pivot_id');
                }
                $rows = $query->get();
                foreach ($rows as $r) {
                    $assigned[$r->user_id] = ['user_id' => $r->user_id, 'email' => $r->email, 'pivot_id' => $r->pivot_id ?? null];
                }
            }

            $assignedList = array_values($assigned);
            $totalAssigned = count($assignedList);

            if ($totalAssigned <= $limit) {
                $this->command?->info("Role {$role} has {$totalAssigned} assigned (<= {$limit}) — OK");
                continue;
            }

            // Determine which to keep: prefer pattern matches
            $patternMatches = [];
            $others = [];
            foreach ($assignedList as $a) {
                if (preg_match('/^' . preg_quote($role, '/') . '(\d+)@example\.com$/', $a['email'], $m)) {
                    $patternMatches[] = ['user_id' => $a['user_id'], 'num' => (int) $m[1], 'pivot_id' => $a['pivot_id']];
                } else {
                    $others[] = ['user_id' => $a['user_id'], 'pivot_id' => $a['pivot_id']];
                }
            }

            usort($patternMatches, function ($x, $y) { return $x['num'] <=> $y['num']; });
            usort($others, function ($x, $y) { return ($x['pivot_id'] ?? $x['user_id']) <=> ($y['pivot_id'] ?? $y['user_id']); });

            $keep = [];
            foreach ($patternMatches as $p) {
                if (count($keep) >= $limit) break;
                $keep[] = $p['user_id'];
            }
            foreach ($others as $o) {
                if (count($keep) >= $limit) break;
                if (! in_array($o['user_id'], $keep, true)) $keep[] = $o['user_id'];
            }

            $allUserIds = array_map(fn($x) => $x['user_id'], $assignedList);
            $toRemove = array_values(array_diff($allUserIds, $keep));

            if (empty($toRemove)) {
                $this->command?->info("Nothing to remove for role {$role} after selection.");
                continue;
            }

            if (! $applyTrim) {
                $this->command?->info("DRY-RUN: Would remove " . count($toRemove) . " pivot(s) for role {$role}: " . implode(', ', $toRemove));
            } else {
                // Remove from all pivot tables
                foreach ($pivotTables as $pivot) {
                    DB::table($pivot)->where('role_id', $roleId)->whereIn('user_id', $toRemove)->delete();
                }
                $this->command?->info("Removed " . count($toRemove) . " pivot(s) for role {$role}.");
            }
        }
    }
}
