<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Schema::table('payrolls', function (Blueprint $table) {

            // === 0. الأعمدة الأساسية المفقودة/التي تسبب مشاكل الترتيب ===
            // الخطأ يقول 'bonuses' غير موجود. يجب إضافته أولاً.
            if (!Schema::hasColumn('payrolls', 'bonuses')) {
                // سنضعها بعد 'deductions' أو 'incentives' لترتيب منطقي
                $table->decimal('bonuses', 10, 2)->default(0.00)->after('deductions');
            }
            
            // يجب التأكد أيضاً من وجود 'deductions' (خصومات يدوية)
            if (!Schema::hasColumn('payrolls', 'deductions')) {
                $table->decimal('deductions', 10, 2)->default(0.00)->after('incentives');
            }


            // === 1. الأعمدة المحسوبة (total_salary و net_salary) ===
            // المشكلة كانت في محاولة الإضافة "after `bonuses`".
            // نستخدم الآن حقل مضمون الوجود أو نضعها في نهاية الجدول.
            
            if (!Schema::hasColumn('payrolls', 'total_salary')) {
                 // نستخدم 'bonuses' كمرجع الآن بعد أن تأكدنا من وجوده، أو نستخدم 'deductions'
                 $table->decimal('total_salary', 10, 2)->default(0.00)->after('bonuses');
            }
            if (!Schema::hasColumn('payrolls', 'net_salary')) {
                 $table->decimal('net_salary', 10, 2)->default(0.00)->after('total_salary');
            }
            if (!Schema::hasColumn('payrolls', 'notes')) {
                $table->text('notes')->nullable()->after('net_salary');
            }

            // === 2. أعمدة الخصومات وتتبع الإجازات (التي كانت تسبب مشاكل التكرار سابقاً) ===
            // الأعمدة التي تسببت في خطأ "Column already exists" (في المشكلة الأولى)
            // نستخدم شرط "if (!Schema::hasColumn())" للتأكد من عدم تكرار الإضافة.
            if (!Schema::hasColumn('payrolls', 'active_deductions')) {
                // نضعها بعد 'total_salary' أو 'net_salary' لتفادي مشاكل الترتيب
                $table->decimal('active_deductions', 10, 2)->default(0.00)->after('net_salary');
            }
            if (!Schema::hasColumn('payrolls', 'loan_installments')) {
                $table->decimal('loan_installments', 10, 2)->default(0.00)->after('active_deductions');
            }
            if (!Schema::hasColumn('payrolls', 'leave_deduction')) {
                $table->decimal('leave_deduction', 10, 2)->default(0.00)->after('loan_installments');
            }
            
            // 3. حقول الإجازات الجديدة والمحدثة (يجب أن تعمل دون مشاكل الآن)
            if (!Schema::hasColumn('payrolls', 'leave_balance_before_deduction')) {
                $table->decimal('leave_balance_before_deduction', 10, 2)->default(0.00)->after('leave_deduction')->comment('رصيد الإجازات التراكمي في نهاية الشهر السابق');
            }
            if (!Schema::hasColumn('payrolls', 'leave_balance_after_deduction')) {
                $table->decimal('leave_balance_after_deduction', 10, 2)->default(0.00)->after('leave_balance_before_deduction')->comment('رصيد الإجازات المتبقي بعد خصم هذا الشهر');
            }
            if (!Schema::hasColumn('payrolls', 'leave_deduction_amount')) {
                $table->decimal('leave_deduction_amount', 10, 2)->default(0.00)->after('leave_balance_after_deduction')->comment('المبلغ المالي المخصوم فعليًا مقابل الإجازات الزائدة');
            }
            if (!Schema::hasColumn('payrolls', 'leave_granted_this_month')) {
                $table->decimal('leave_granted_this_month', 10, 2)->default(0.00)->after('leave_deduction_amount')->comment('عدد أيام الإجازة الممنوحة للموظف لهذا الشهر');
            }
            if (!Schema::hasColumn('payrolls', 'used_leave_days')) {
                $table->decimal('used_leave_days', 10, 2)->default(0.00)->after('leave_granted_this_month')->comment('عدد أيام الإجازة المستخدمة فعليًا هذا الشهر');
            }
            if (!Schema::hasColumn('payrolls', 'excess_days')) {
                $table->decimal('excess_days', 10, 2)->default(0.00)->after('used_leave_days')->comment('عدد الأيام الزائدة المستحقة للخصم');
            }

        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::table('payrolls', function (Blueprint $table) {
             
            // عند التراجع، نحذف كل الأعمدة التي أضفناها في هذا الترحيل، مع فحص وجودها لتجنب الأخطاء.
            if (Schema::hasColumn('payrolls', 'excess_days')) {
                $table->dropColumn('excess_days');
            }
            if (Schema::hasColumn('payrolls', 'used_leave_days')) {
                $table->dropColumn('used_leave_days');
            }
            if (Schema::hasColumn('payrolls', 'leave_granted_this_month')) {
                $table->dropColumn('leave_granted_this_month');
            }
            if (Schema::hasColumn('payrolls', 'leave_deduction_amount')) {
                $table->dropColumn('leave_deduction_amount');
            }
            if (Schema::hasColumn('payrolls', 'leave_balance_after_deduction')) {
                $table->dropColumn('leave_balance_after_deduction');
            }
            if (Schema::hasColumn('payrolls', 'leave_balance_before_deduction')) {
                $table->dropColumn('leave_balance_before_deduction');
            }
            if (Schema::hasColumn('payrolls', 'leave_deduction')) {
                $table->dropColumn('leave_deduction');
            }
            if (Schema::hasColumn('payrolls', 'loan_installments')) {
                $table->dropColumn('loan_installments');
            }
            if (Schema::hasColumn('payrolls', 'active_deductions')) {
                $table->dropColumn('active_deductions');
            }
            if (Schema::hasColumn('payrolls', 'notes')) {
                $table->dropColumn('notes');
            }
            // إذا كنا متأكدين أن bonuses/total_salary/net_salary تمت إضافتها فقط في هذا الترحيل:
            // if (Schema::hasColumn('payrolls', 'net_salary')) { $table->dropColumn('net_salary'); }
            // if (Schema::hasColumn('payrolls', 'total_salary')) { $table->dropColumn('total_salary'); }
            // if (Schema::hasColumn('payrolls', 'bonuses')) { $table->dropColumn('bonuses'); }
            // if (Schema::hasColumn('payrolls', 'deductions')) { $table->dropColumn('deductions'); }

        });
    }
};