<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * إضافة العلاقات الأجنبية لجدول الموظفين.
     */
    public function up(): void
    {
        // التحقق من وجود الجداول المرجعية
        $requiredTables = ['departments', 'positions', 'users', 'roles'];
        foreach ($requiredTables as $table) {
            if (!Schema::hasTable($table)) {
                echo "[تحذير] جدول {$table} غير موجود. تأكد من تشغيل الهجرات بالترتيب.\n";
                return;
            }
        }

        Schema::table('employees', function (Blueprint $table) {
            // ------------------------------
            // حذف الحقول القديمة (نصية)
            // ------------------------------
            if (Schema::hasColumn('employees', 'department')) {
                $table->dropColumn('department');
            }
            if (Schema::hasColumn('employees', 'position')) {
                $table->dropColumn('position');
            }

            // ------------------------------
            // إضافة الأعمدة الجديدة (إن لم تكن موجودة)
            // ------------------------------

            if (!Schema::hasColumn('employees', 'department_id')) {
                $table->foreignId('department_id')->nullable()->after('name');
            }

            if (!Schema::hasColumn('employees', 'position_id')) {
                $table->foreignId('position_id')->nullable()->after('department_id');
            }

            if (!Schema::hasColumn('employees', 'role_id')) {
                $table->foreignId('role_id')->nullable()
                      ->constrained('roles')
                      ->nullOnDelete()
                      ->after('position_id');
            }

            if (!Schema::hasColumn('employees', 'user_id')) {
                $table->foreignId('user_id')->nullable()->after('role_id');
            }

            // ------------------------------
            // إضافة العلاقات الخارجية (FK)
            // ------------------------------

            // الطريقة الآمنة: أضف فقط إن لم تكن موجودة
            try {
                $table->foreign('department_id')->references('id')->on('departments')->nullOnDelete();
            } catch (\Exception $e) {
                // تجاهل الخطأ إذا كانت العلاقة موجودة مسبقًا
            }

            try {
                $table->foreign('position_id')->references('id')->on('positions')->nullOnDelete();
            } catch (\Exception $e) {
                // تجاهل الخطأ إذا كانت العلاقة موجودة
            }

            try {
                $table->foreign('user_id')->references('id')->on('users')->nullOnDelete();
            } catch (\Exception $e) {
                // تجاهل الخطأ
            }
        });
    }

    /**
     * التراجع عن إضافة العلاقات.
     */
    public function down(): void
    {
        Schema::table('employees', function (Blueprint $table) {
            // قائمة العلاقات التي قد تكون موجودة
            $foreignKeys = [
                'department_id',
                'position_id',
                'user_id'
            ];

            foreach ($foreignKeys as $column) {
                try {
                    $table->dropForeign([$column]);
                } catch (\Exception $e) {
                    // تجاهل الخطأ إن لم تكن العلاقة موجودة
                    echo "[ملاحظة] علاقة {$column} غير موجودة أو تم حذفها مسبقًا.\n";
                }
            }

            // إزالة الأعمدة
            $columnsToRemove = ['department_id', 'position_id', 'user_id'];
            foreach ($columnsToRemove as $column) {
                if (Schema::hasColumn('employees', $column)) {
                    $table->dropColumn($column);
                }
            }

            // إعادة الحقول القديمة (نصية)
            if (!Schema::hasColumn('employees', 'department')) {
                $table->string('department')->nullable();
            }
            if (!Schema::hasColumn('employees', 'position')) {
                $table->string('position')->nullable();
            }
        });
    }
};