<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up()
    {
        Schema::create('loans', function (Blueprint $table) {
            $table->id();

            // ربط القرض بالموظف
            $table->foreignId('employee_id')
                  ->constrained('employees')
                  ->cascadeOnDelete();

            $table->string('loan_number')->unique();
            $table->string('title');
            $table->text('description')->nullable();
            $table->decimal('amount', 10, 2);
            $table->decimal('interest_rate', 5, 2)->default(0); // نسبة الفائدة %
            $table->decimal('total_amount', 10, 2); // المبلغ الإجمالي مع الفائدة
            $table->integer('installments_count'); // عدد الأقساط
            $table->decimal('installment_amount', 10, 2); // قيمة القسط الشهري
            $table->date('start_date');
            $table->date('end_date');

            // حالة القرض
            $table->enum('status', ['pending', 'approved', 'active', 'completed', 'cancelled'])
                  ->default('pending');

            // الموافقة على القرض بواسطة مستخدم (مرتبط بنظام الأدوار الداخلي)
            $table->foreignId('approved_by')
                  ->nullable()
                  ->constrained('users')
                  ->nullOnDelete();

            $table->date('approved_at')->nullable();
            $table->text('approval_notes')->nullable();

            $table->timestamps();

            // فهرس لتسريع البحث حسب الموظف وحالة القرض
            $table->index(['employee_id', 'status']);
        });

        Schema::create('loan_installments', function (Blueprint $table) {
            $table->id();

            // ربط القسط بالقرض
            $table->foreignId('loan_id')
                  ->constrained('loans')
                  ->cascadeOnDelete();

            $table->integer('installment_number');
            $table->decimal('amount', 10, 2);
            $table->date('due_date');
            $table->date('paid_date')->nullable();
            $table->enum('status', ['pending', 'paid', 'overdue', 'cancelled'])->default('pending');
            $table->text('notes')->nullable();
            $table->timestamps();

            // فهرس لتسريع البحث حسب القرض وحالة القسط
            $table->unique(['loan_id', 'installment_number'], 'loan_installment_unique');
        });
    }

    public function down()
    {
        Schema::dropIfExists('loan_installments');
        Schema::dropIfExists('loans');
    }
};
