<?php

namespace App\Services;

use App\Models\Employee;
use App\Models\Leave;
use App\Models\LeaveBalanceChange;
use App\Models\Holiday;
use Carbon\Carbon;

class LeaveBalanceService
{
    /**
     * Calculate number of deducted days for a leave period.
     * Excludes weekly off days (from employee.weekly_off_days) and holidays marked as paid.
     *
     * @param Leave $leave
     * @return int
     */
    public function calculateDeductedDays(Leave $leave): int
    {
        $employee = $leave->employee;
        $start = Carbon::parse($leave->start_date)->startOfDay();
        $end = Carbon::parse($leave->end_date)->startOfDay();

        $weeklyOff = $employee->weekly_off_days ?? [];
        // normalize to dayOfWeek numbers (0=Sunday..6=Saturday) if stored as names, try to handle both
        $weekendNumbers = [];
        foreach ($weeklyOff as $d) {
            if (is_int($d)) $weekendNumbers[] = $d;
            else {
                $map = [
                    'sunday' => 0, 'monday' => 1, 'tuesday' => 2, 'wednesday' => 3,
                    'thursday' => 4, 'friday' => 5, 'saturday' => 6
                ];
                $key = strtolower($d);
                if (isset($map[$key])) $weekendNumbers[] = $map[$key];
            }
        }

        $total = 0;
        $current = $start->copy();
        while ($current->lessThanOrEqualTo($end)) {
            // skip weekly off
            if (in_array($current->dayOfWeek, $weekendNumbers)) {
                $current->addDay();
                continue;
            }

            // skip holidays that are marked is_paid = true (do not deduct)
            $isHoliday = Holiday::whereDate('date', $current->toDateString())->where('is_paid', true)->exists();
            if ($isHoliday) {
                $current->addDay();
                continue;
            }

            $total++;
            $current->addDay();
        }

        return $total;
    }

    /**
     * Get available balance for kind ('monthly'|'annual') as of date.
     * Uses employee.entitlement + sum of leave_balance_changes effective <= asOfDate - used leaves.
     *
     * @param Employee $employee
     * @param string $kind
     * @param Carbon|null $asOfDate
     * @return int
     */
    public function availableBalance(Employee $employee, string $kind, Carbon $asOfDate = null): int
    {
        $asOfDate = $asOfDate ?? Carbon::now();

        $entitlement = $kind === 'monthly' ? ($employee->monthly_leave_days_allowed ?? 0) : ($employee->annual_entitlement ?? 0);

        $changes = LeaveBalanceChange::where('employee_id', $employee->id)
            ->where('kind', $kind)
            ->whereDate('effective_from', '<=', $asOfDate->toDateString())
            ->sum('days_change');

        // used: approved leaves of types that deduct from this kind, up to asOfDate
        $used = Leave::where('employee_id', $employee->id)
            ->where('status', 'approved')
            ->whereNotNull('approved_at')
            ->where(function ($q) use ($kind) {
                $q->whereHas('leaveTypeModel', function ($qq) use ($kind) {
                    if ($kind === 'monthly') {
                        $qq->whereIn('code', ['monthly', 'annual']);
                    } else {
                        $qq->where('code', 'annual');
                    }
                });
            })
            ->whereDate('approved_at', '<=', $asOfDate->toDateString())
            ->sum('deducted_days');

        return max(0, (int) ($entitlement + $changes - $used));
    }

    /**
     * Apply approval: set approved_at etc, compute deducted days, create LeaveBalanceChange if needed.
     *
     * @param Leave $leave
     * @param int|null $approverId
     * @return Leave
     */
    public function applyApproval(Leave $leave, ?int $approverId = null): Leave
    {
        $leave->approved_at = now();
        // only set approved_by if the user exists (avoid FK errors in tests)
        $approvedBy = $approverId ?? auth()->id();
        $leave->approved_by = \App\Models\User::find($approvedBy)?->id;
        $leave->status = 'approved';

        // compute deducted days
        // If deductible_days is already set (for mixed single-record leaves), prefer it
        // otherwise compute from calendar excluding weekly-off/holidays.
        if (!is_null($leave->deductible_days)) {
            $deducted = (int) $leave->deductible_days;
        } else {
            $deducted = $this->calculateDeductedDays($leave);
        }
        $leave->deducted_days = $deducted;

        // handle leave type behavior
        $type = $leave->leaveTypeModel ?? null;
        if ($type && $type->balance_kind) {
            // create a leave balance change (negative)
            // only set created_by if the user exists (avoid FK errors in tests where user id may not exist)
            $createdBy = null;
            if ($leave->approved_by) {
                $createdBy = \App\Models\User::find($leave->approved_by)?->id;
            }

            $change = LeaveBalanceChange::create([
                'employee_id' => $leave->employee_id,
                'kind' => $type->balance_kind,
                'days_change' => -1 * $deducted,
                'effective_from' => $leave->approved_at->toDateString(),
                'note' => 'Applied for leave #' . $leave->id,
                'created_by' => $createdBy,
            ]);
            $leave->applied_balance_change_id = $change->id;

            // balance snapshot
            $service = $this;
            $before = $service->availableBalance($leave->employee, $type->balance_kind, Carbon::parse($leave->approved_at)->subDay());
            $after = $service->availableBalance($leave->employee, $type->balance_kind, Carbon::parse($leave->approved_at));
            $leave->balance_snapshot = [
                'before' => $before,
                'after' => $after,
            ];
        }

        // unpaid leave: mark for payroll deduction (affects_salary)
        if ($type && $type->affects_salary) {
            $leave->is_unpaid_deduction_applied = false; // payroll will pick it up
        }

        // If this leave has an internal unpaid portion (mixed single-record),
        // ensure payroll will pick it up by not marking unpaid deduction as applied.
        if (isset($leave->unpaid_deducted_days) && $leave->unpaid_deducted_days > 0) {
            $leave->is_unpaid_deduction_applied = false;
        }

        $leave->save();

        return $leave;
    }

    /**
     * Adjust an already-approved leave: revert previous balance change (if any)
     * and apply a new balance change according to updated leave dates/values.
     * This will set the leave status to 'modified' to indicate a post-approval edit.
     *
     * @param Leave $leave
     * @param array $newData
     * @param int|null $actorId
     * @return Leave
     */
    public function adjustApproval(Leave $leave, array $newData, ?int $actorId = null): Leave
    {
        // start transaction at caller if needed; we'll manage here for safety
        \DB::beginTransaction();
        try {
            // revert previous balance change if exists
            if ($leave->applied_balance_change_id) {
                $old = LeaveBalanceChange::find($leave->applied_balance_change_id);
                if ($old) {
                    LeaveBalanceChange::create([
                        'employee_id' => $old->employee_id,
                        'kind' => $old->kind,
                        'days_change' => -1 * $old->days_change, // reverse
                        'effective_from' => Carbon::now()->toDateString(),
                        'note' => 'Reverted change for leave #' . $leave->id,
                        'created_by' => $actorId,
                    ]);
                }
            }

            // apply updates to the leave model
            $leave->fill($newData);

            // ensure days_count/deductible calculation happens
            $leave->save();

            // recompute deducted days
            if (!is_null($leave->deductible_days)) {
                $deducted = (int) $leave->deductible_days;
            } else {
                $deducted = $this->calculateDeductedDays($leave);
            }
            $leave->deducted_days = $deducted;

            // create new balance change if leave type uses balance
            $type = $leave->leaveTypeModel ?? null;
            if ($type && $type->balance_kind) {
                $createdBy = $actorId ? \App\Models\User::find($actorId)?->id : null;

                $change = LeaveBalanceChange::create([
                    'employee_id' => $leave->employee_id,
                    'kind' => $type->balance_kind,
                    'days_change' => -1 * $deducted,
                    'effective_from' => Carbon::now()->toDateString(),
                    'note' => 'Adjusted for leave #' . $leave->id,
                    'created_by' => $createdBy,
                ]);

                $leave->applied_balance_change_id = $change->id;

                // balance snapshot
                $before = $this->availableBalance($leave->employee, $type->balance_kind, Carbon::parse($change->effective_from)->subDay());
                $after = $this->availableBalance($leave->employee, $type->balance_kind, Carbon::parse($change->effective_from));
                $leave->balance_snapshot = ['before' => $before, 'after' => $after];
            }

            // mark that approval was modified, but keep status as 'approved'
            $leave->approval_modified = true;
            $leave->approval_modified_at = Carbon::now();
            $leave->approval_modified_by = $actorId ? \App\Models\User::find($actorId)?->id : null;
            $leave->save();

            \DB::commit();
            return $leave;
        } catch (\Throwable $e) {
            \DB::rollBack();
            throw $e;
        }
    }
}
