<?php

namespace App\Policies;

// تجنب التضارب في الأسماء باستخدام alias
use App\Models\User as AuthenticatableUser; // المستخدم الذي نتحقق منه (النموذج المرتبط بقاعدة البيانات)
use App\Models\User; // المستخدم الحالي (المصادق عليه)
use Illuminate\Auth\Access\HandlesAuthorization;

class UserPolicy
{
    use HandlesAuthorization;

    /**
     * تحديد ما إذا كان المستخدم الحالي يمكنه عرض قائمة المستخدمين.
     *
     * @param  \App\Models\User  $user
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function viewAny(User $user)
    {
        // إذا كان المستخدم لديه دور 'admin' أو 'super-admin'، سمح له بالعرض
        if ($user->hasRole('admin') || $user->hasRole('super-admin')) {
            return true;
        }
        // وإلا، تحقق من الصلاحية المحددة
        return $user->can('view_users');
    }

    /**
     * تحديد ما إذا كان المستخدم الحالي يمكنه عرض مستخدم محدد.
     *
     * @param  \App\Models\User  $user
     * @param  \App\Models\User  $model
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function view(User $user, AuthenticatableUser $model)
    {
        // إذا كان المستخدم لديه دور 'admin' أو 'super-admin'، سمح له بالعرض
        if ($user->hasRole('admin') || $user->hasRole('super-admin')) {
            return true;
        }
        // وإلا، تحقق من الصلاحية المحددة
        return $user->can('view_users');
    }

    /**
     * تحديد ما إذا كان المستخدم الحالي يمكنه إنشاء مستخدم.
     *
     * @param  \App\Models\User  $user
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function create(User $user)
    {
        // إذا كان المستخدم لديه دور 'admin' أو 'super-admin'، سمح له بالإنشاء
        if ($user->hasRole('admin') || $user->hasRole('super-admin')) {
            return true;
        }
        // وإلا، تحقق من الصلاحية المحددة
        return $user->can('manage_users');
    }

    /**
     * تحديد ما إذا كان المستخدم الحالي يمكنه تحديث مستخدم.
     *
     * @param  \App\Models\User  $user
     * @param  \App\Models\User  $model
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function update(User $user, AuthenticatableUser $model)
    {
        // لا يمكن للمستخدم تعديل نفسه (لمنع تعطيل الحساب مثلاً)
        // أو السماح للمستخدم بتعديل ملفه الشخصي فقط
        if ($user->id === $model->id) {
            // يمكنك تعديل هذا ليسمح بتعديل الملف الشخصي فقط
            // return true;
            // لكن الافتراضي هنا منع التعديل على النفس عبر لوحة الإدارة
            return false; // أو $user->can('manage_users') && $user->id !== $model->id;
        }
        // إذا كان المستخدم لديه دور 'admin' أو 'super-admin'، سمح له بالتعديل
        if ($user->hasRole('admin') || $user->hasRole('super-admin')) {
            return true;
        }
        // وإلا، تحقق من الصلاحية المحددة
        return $user->can('manage_users');
    }

    /**
     * تحديد ما إذا كان المستخدم الحالي يمكنه حذف مستخدم.
     *
     * @param  \App\Models\User  $user
     * @param  \App\Models\User  $model
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function delete(User $user, AuthenticatableUser $model)
    {
        // لا يمكن للمستخدم حذف نفسه
        if ($user->id === $model->id) {
            return false;
        }
        // إذا كان المستخدم لديه دور 'admin' أو 'super-admin'، سمح له بالحذف
        if ($user->hasRole('admin') || $user->hasRole('super-admin')) {
            return true;
        }
        // وإلا، تحقق من الصلاحية المحددة
        return $user->can('manage_users');
    }

    // تم حذف الدالة `toggleStatus` لأنها لم تعد مُعرفة في UserController

    /**
     * تحديد ما إذا كان المستخدم الحالي يمكنه استعادة مستخدم محذوف.
     *
     * @param  \App\Models\User  $user
     * @param  \App\Models\User  $model
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function restore(User $user, AuthenticatableUser $model)
    {
        // إذا كان المستخدم لديه دور 'admin' أو 'super-admin'، سمح له بالاستعادة
        if ($user->hasRole('admin') || $user->hasRole('super-admin')) {
            return true;
        }
        // وإلا، تحقق من الصلاحية المحددة
        return $user->can('manage_users');
    }

    /**
     * تحديد ما إذا كان المستخدم الحالي يمكنه حذف المستخدم نهائياً.
     *
     * @param  \App\Models\User  $user
     * @param  \App\Models\User  $model
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function forceDelete(User $user, AuthenticatableUser $model)
    {
        // عادةً لا يُسمح بالحذف النهائي إلا للسوبر أدمن
        // ومنع حذف النفس
        if ($user->id === $model->id) {
            return false;
        }
        return $user->hasRole('super-admin'); // أو false إذا لم ترد السماح به مطلقاً
    }
}