<?php

namespace App\Policies;

use App\Models\Payroll;
use App\Models\User;
use Illuminate\Auth\Access\HandlesAuthorization;
use Illuminate\Auth\Access\Response;

class PayrollPolicy
{
    use HandlesAuthorization;

    /**
     * Helper method to check if user has admin/super-admin roles.
     *
     * @param  \App\Models\User  $user
     * @return boolean
     */
    protected function isAdminOrSuper(User $user)
    {
        return $user->hasRole(['admin', 'super-admin']);
    }

    /**
     * Determine whether the user can view any models.
     *
     * @param  \App\Models\User  $user
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function viewAny(User $user)
    {
        // Allow admins/super-admins, or users with specific permission
        // OR allow employees to view their own payrolls
        return $this->isAdminOrSuper($user) || 
               $user->hasRole('hr') || // HR can view all
               $user->can('view_payrolls') || 
               $user->can('manage_payroll') ||
               $user->hasRole('employee'); // Employee can potentially access their own list
        // Fallback to manage_payroll for backward compatibility
    }

    /**
     * Determine whether the user can view the model.
     *
     * @param  \App\Models\User  $user
     * @param  \App\Models\Payroll  $payroll
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function view(User $user, Payroll $payroll)
    {
        // Allow admins/super-admins, HR, or users with specific permission
        if ($this->isAdminOrSuper($user) || $user->hasRole('hr') || $user->can('view_payrolls') || $user->can('manage_payroll')) {
            return true;
        }

        // ✅ السماح للموظف العادي برؤية *راتبه الخاص فقط*
        // التحقق مما إذا كان المستخدم مرتبطًا بسجل موظف
        // وأيضًا التحقق مما إذا كان هذا السجل هو نفسه $payroll->employee_id
        return $user->employee && $user->employee->id === $payroll->employee_id;
    }

    /**
     * Determine whether the user can create models.
     *
     * @param  \App\Models\User  $user
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function create(User $user)
    {
        // Only admins/super-admins or users with specific permission can create
        // Employees typically do not create their own payrolls
        return $this->isAdminOrSuper($user) || $user->hasRole('hr') || $user->can('create_payrolls') || $user->can('manage_payroll');
        // Fallback to manage_payroll for backward compatibility
    }

    /**
     * Determine whether the user can update the model.
     *
     * @param  \App\Models\User  $user
     * @param  \App\Models\Payroll  $payroll
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function update(User $user, Payroll $payroll)
    {
        // Allow admins/super-admins, HR, or users with specific permission
        // The status check should ideally be done in the controller logic
        return $this->isAdminOrSuper($user) || $user->hasRole('hr') || $user->can('edit_payrolls') || $user->can('manage_payroll');
        // Fallback to manage_payroll for backward compatibility
        // Note: Status check ('pending') is handled in the controller, not here.
    }

    /**
     * Determine whether the user can delete the model.
     *
     * @param  \App\Models\User  $user
     * @param  \App\Models\Payroll  $payroll
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function delete(User $user, Payroll $payroll)
    {
        // Allow admins/super-admins, HR, or users with specific permission
        // The status check should ideally be done in the controller logic
        return $this->isAdminOrSuper($user) || $user->hasRole('hr') || $user->can('delete_payrolls') || $user->can('manage_payroll');
        // Fallback to manage_payroll for backward compatibility
        // Note: Status check ('paid') is handled in the controller, not here.
    }

    /**
     * Determine whether the user can approve the payroll.
     *
     * @param  \App\Models\User  $user
     * @param  \App\Models\Payroll  $payroll
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function approve(User $user, Payroll $payroll)
    {
        // Typically, only admins/super-admins, HR, or users with specific permission can approve
        return $this->isAdminOrSuper($user) || $user->hasRole('hr') || $user->can('approve_payrolls') || $user->can('manage_payroll');
        // Fallback to manage_payroll for backward compatibility
    }

    /**
     * Determine whether the user can pay the payroll.
     *
     * @param  \App\Models\User  $user
     * @param  \App\Models\Payroll  $payroll
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function pay(User $user, Payroll $payroll)
    {
        // Typically, only admins/super-admins, HR, or users with specific permission can pay
        return $this->isAdminOrSuper($user) || $user->hasRole('hr') || $user->can('pay_payrolls') || $user->can('manage_payroll');
        // Fallback to manage_payroll for backward compatibility
    }

    /**
     * Determine whether the user can generate monthly payroll.
     *
     * @param  \App\Models\User  $user
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function generateMonthly(User $user)
    {
        // Typically, only admins/super-admins, HR, or users with specific permission can generate
        return $this->isAdminOrSuper($user) || $user->hasRole('hr') || $user->can('generate_monthly_payroll') || $user->can('manage_payroll');
        // Fallback to manage_payroll for backward compatibility
    }

    /**
     * Determine whether the user can restore the model.
     *
     * @param  \App\Models\User  $user
     * @param  \App\Models\Payroll  $payroll
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function restore(User $user, Payroll $payroll)
    {
        // Allow admins/super-admins, HR, or users with specific permission
        return $this->isAdminOrSuper($user) || $user->hasRole('hr') || $user->can('restore_payrolls') || $user->can('manage_payroll');
        // Fallback to manage_payroll for backward compatibility
    }

    /**
     * Determine whether the user can permanently delete the model.
     *
     * @param  \App\Models\User  $user
     * @param  \App\Models\Payroll  $payroll
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function forceDelete(User $user, Payroll $payroll)
    {
        // Usually restricted to super-admins only
        return $user->hasRole('super-admin');
        // Alternatively, check for a very specific permission:
        // return $user->hasRole('super-admin') || $user->can('force_delete_payrolls');
    }
}