<?php

namespace App\Policies;

use App\Models\Loan;
use App\Models\User;
use Illuminate\Auth\Access\HandlesAuthorization;

class LoanPolicy
{
    use HandlesAuthorization;

    /**
     * تحقق من أن المستخدم هو Admin أو Super-admin أو لديه صلاحية إدارة القروض.
     */
    protected function hasLoanAccess(User $user): bool
    {
        return $user->hasRole(['admin', 'super-admin']) || $user->can('manage_loans');
    }

    /**
     * عرض كل القروض.
     */
    public function viewAny(User $user): bool
    {
        // ✅ السماح للموظف برؤية قائمة قروضه الخاصة فقط
        return $this->hasLoanAccess($user) || $user->hasRole('employee');
    }

    /**
     * عرض قرض معين.
     */
    public function view(User $user, Loan $loan): bool
    {
        // Admin/Super-admin أو أي شخص لديه صلاحية إدارة القروض يمكنه مشاهدة أي قرض
        if ($this->hasLoanAccess($user)) {
            return true;
        }

        // ✅ التحقق مما إذا كان المستخدم مرتبطًا ب(employee) و
        // أن (employee) المرتبط بـ (loan) هو نفسه (employee) المرتبط بالمستخدم
        // هذا يفترض أن Loan::employee_id يشير إلى Employee::id
        return $user->employee && $user->employee->id === $loan->employee_id;
    }

    /**
     * إنشاء طلب قرض.
     */
    public function create(User $user): bool
    {
        // Admin/Super-admin أو أي شخص لديه صلاحية إدارة القروض يمكنه إنشاء أي طلب
        // الموظف العادي يمكنه إنشاء طلب لنفسه
        return $this->hasLoanAccess($user) || $user->employee !== null;
    }

    /**
     * تعديل طلب قرض.
     */
    public function update(User $user, Loan $loan): bool
    {
        // Admin/Super-admin أو أي شخص لديه صلاحية إدارة القروض يمكنه تعديل أي طلب
        if ($this->hasLoanAccess($user)) {
            return true;
        }

        // الموظف يمكنه تعديل طلبه فقط إذا كان pending
        return $user->employee && $user->employee->id === $loan->employee_id && $loan->status === 'pending';
    }

    /**
     * حذف طلب قرض.
     */
    public function delete(User $user, Loan $loan): bool
    {
        // Admin/Super-admin أو أي شخص لديه صلاحية إدارة القروض يمكنه حذف أي طلب
        if ($this->hasLoanAccess($user)) {
            return true;
        }

        // الموظف يمكنه حذف طلبه فقط إذا كان pending
        return $user->employee && $user->employee->id === $loan->employee_id && $loan->status === 'pending';
    }

    /**
     * الموافقة على القرض.
     */
    public function approve(User $user, Loan $loan): bool
    {
        return $this->hasLoanAccess($user);
    }

    /**
     * رفض القرض.
     */
    public function reject(User $user, Loan $loan): bool
    {
        return $this->hasLoanAccess($user);
    }

    /**
     * إلغاء طلب قرض.
     */
    public function cancel(User $user, Loan $loan): bool
    {
        // Admin/Super-admin أو أي شخص لديه صلاحية إدارة القروض يمكنه الإلغاء لأي طلب
        if ($this->hasLoanAccess($user)) {
            return true;
        }

        // الموظف يمكنه إلغاء طلبه فقط إذا كان pending
        return $user->employee && $user->employee->id === $loan->employee_id && $loan->status === 'pending';
    }

    /**
     * استعادة قرض محذوف.
     */
    public function restore(User $user, Loan $loan): bool
    {
        return $this->hasLoanAccess($user);
    }

    /**
     * الحذف النهائي (عادةً للسوبر أدمن فقط).
     */
    public function forceDelete(User $user, Loan $loan): bool
    {
        return $user->hasRole('super-admin');
    }
}