<?php

namespace App\Policies;

use App\Models\Leave;
use App\Models\User;
use Illuminate\Auth\Access\HandlesAuthorization;

class LeavePolicy
{
    use HandlesAuthorization;

    /**
     * تحقق من أن المستخدم هو Admin أو Super-admin أو لديه صلاحية إدارة الإجازات.
     */
    protected function hasLeaveAccess(User $user): bool
    {
        return $user->hasRole(['admin', 'super-admin']) || $user->can('manage_leaves');
    }

    /**
     * عرض كل الإجازات.
     */
    public function viewAny(User $user): bool
    {
        // ✅ السماح للموظف برؤية قائمة إجازاته الخاصة فقط
        return $this->hasLeaveAccess($user) || $user->hasRole('employee');
    }

    /**
     * عرض إجازة معينة.
     */
    public function view(User $user, Leave $leave): bool
    {
        // Admin/Super-admin أو أي شخص لديه صلاحية إدارة الإجازات يمكنه مشاهدة أي إجازة
        if ($this->hasLeaveAccess($user)) {
            return true;
        }

        // ✅ التحقق مما إذا كان المستخدم مرتبطًا ب(employee) و
        // أن (employee) المرتبط بـ (leave) هو نفسه (employee) المرتبط بالمستخدم
        // هذا يفترض أن Leave::employee_id يشير إلى Employee::id
        return $user->employee && $user->employee->id === $leave->employee_id;
    }

    /**
     * إنشاء طلب إجازة.
     */
    public function create(User $user): bool
    {
        // Admin/Super-admin أو أي شخص لديه صلاحية إدارة الإجازات يمكنه إنشاء أي طلب
        // الموظف العادي يمكنه إنشاء طلب لنفسه
        return $this->hasLeaveAccess($user) || $user->employee !== null;
    }

    /**
     * تعديل طلب إجازة.
     * يمكن للأدمن/الموارد البشرية/المالية تعديل أي إجازة (حتى المعتمدة).
     * الموظف يمكنه تعديل طلبه فقط إذا كان pending.
     */
    public function update(User $user, Leave $leave): bool
    {
        // Admin/Super-admin أو أي شخص لديه صلاحية إدارة الإجازات يمكنه تعديل أي طلب (حتى المعتمد)
        if ($this->hasLeaveAccess($user)) {
            return true;
        }

        // الموظف يمكنه تعديل طلبه فقط إذا كان pending
        return $user->employee && $user->employee->id === $leave->employee_id && $leave->status === 'pending';
    }

    /**
     * حذف طلب إجازة.
     */
    public function delete(User $user, Leave $leave): bool
    {
        // Admin/Super-admin أو أي شخص لديه صلاحية إدارة الإجازات يمكنه حذف أي طلب
        if ($this->hasLeaveAccess($user)) {
            return true;
        }

        // الموظف يمكنه حذف طلبه فقط إذا كان pending
        return $user->employee && $user->employee->id === $leave->employee_id && $leave->status === 'pending';
    }

    /**
     * الموافقة على الإجازة.
     */
    public function approve(User $user, Leave $leave): bool
    {
        return $this->hasLeaveAccess($user);
    }

    /**
     * رفض الإجازة.
     */
    public function reject(User $user, Leave $leave): bool
    {
        return $this->hasLeaveAccess($user);
    }

    /**
     * إلغاء طلب إجازة.
     */
    public function cancel(User $user, Leave $leave): bool
    {
        // Admin/Super-admin أو أي شخص لديه صلاحية إدارة الإجازات يمكنه الإلغاء لأي طلب
        if ($this->hasLeaveAccess($user)) {
            return true;
        }

        // الموظف يمكنه إلغاء طلبه فقط إذا كان pending
        return $user->employee && $user->employee->id === $leave->employee_id && $leave->status === 'pending';
    }

    /**
     * استعادة إجازة محذوفة.
     */
    public function restore(User $user, Leave $leave): bool
    {
        return $this->hasLeaveAccess($user);
    }

    /**
     * الحذف النهائي (عادةً للسوبر أدمن فقط).
     */
    public function forceDelete(User $user, Leave $leave): bool
    {
        return $user->hasRole('super-admin');
    }
}