<?php

namespace App\Policies;

use App\Models\Employee; // النموذج الذي يُمثل سجل الموظف
use App\Models\User;     // النموذج الذي يُمثل المستخدم المُصادق عليه
use Illuminate\Auth\Access\HandlesAuthorization;

class EmployeePolicy
{
    use HandlesAuthorization;

    /**
     * تحديد ما إذا كان المستخدم مسموحًا له بعرض *قائمة* الموظفين.
     * عادةً لا يُسمح للموظف العادي برؤية قائمة جميع الموظفين.
     * يُستخدم هذا في EmployeeController@index أو أي مكان يعرض جميع الموظفين.
     * ✅ مع التحقق من أن الموظفين ينتمون لنفس tenant_id.
     */
    public function viewAny(User $user)
    {
        // يُسمح للادمن والسوبر ادمن والـ HR فقط
        // قبول كلا من الأشكال: 'super-admin' و 'super_admin'
        if ($user->hasRole(['admin', 'super-admin', 'super_admin', 'hr'])) {
            // في نظام Multi-Tenancy، هذا الاستعلام في Controller يجب أن يتم تصفية النتائج
            // باستخدام Global Scope أو باستخدام tenant_id = $user->tenant_id
            return true;
        }

        // للموظف العادي، لا يُسمح بعرض قائمة الموظفين
        // هذا هو السبب في ظهور 403 عندما يحاول "حسن" الدخول إلى /employees
        return false;
    }

    /**
     * تحديد ما إذا كان المستخدم مسموحًا له بعرض *سجل* موظف معين (مثل في صفحة show).
     * هذا هو المكان الذي نمنع فيه "حسن" من رؤية "علي".
     * ✅ مع التحقق من أن المستخدم والموظف من نفس tenant_id.
     */
    public function view(User $user, Employee $employee)
    {
        // يُسمح للادمن والسوبر ادمن والـ HR برؤية أي موظف (من نفس شركتهم فقط)
        if ($user->hasRole(['admin', 'super-admin', 'hr'])) {
            // ✅ التحقق من أن المستخدم والموظف من نفس tenant
            return $user->tenant_id === $employee->tenant_id;
        }

        // يُسمح للموظف العادي برؤية *سجله الخاص فقط*
        // التحقق مما إذا كان المستخدم مرتبطًا بسجل موظف
        // وأيضًا التحقق مما إذا كان هذا السجل هو نفسه $employee المُمرر
        // ✅ التحقق من أن tenant_id مطابق
        if ($user->hasRole('employee') && $user->employee && $user->employee->id === $employee->id && $user->tenant_id === $employee->tenant_id) {
            return true;
        }

        // في جميع الحالات الأخرى، لا يُسمح بالعرض
        return false;
    }

    /**
     * تحديد ما إذا كان المستخدم مسموحًا له بإنشاء نموذج موظف جديد.
     * (عادةً مقتصر على الأدوار الإدارية)
     * ✅ مع التحقق من أن الموظف الجديد سيرتبط بـ tenant_id الصحيح.
     */
    public function create(User $user)
    {
        // يجب أن يتم ربط الموظف الجديد تلقائيًا بـ tenant_id من المستخدم أو الجلسة
        return $user->hasRole(['admin', 'super-admin', 'hr']);
    }

    /**
     * تحديد ما إذا كان المستخدم مسموحًا له بتحديث *سجل* موظف معين.
     * (عادةً مقتصر على الأدوار الإدارية، أو للموظف لتحديث سجله فقط)
     * ✅ مع التحقق من أن المستخدم والموظف من نفس tenant_id.
     */
    public function update(User $user, Employee $employee)
    {
        // يُسمح للادمن والسوبر ادمن والـ HR بتحديث أي موظف (من نفس شركتهم فقط)
        if ($user->hasRole(['admin', 'super-admin', 'hr'])) {
            // ✅ التحقق من أن المستخدم والموظف من نفس tenant
            return $user->tenant_id === $employee->tenant_id;
        }

        // يُسمح للموظف العادي بتحديث *سجله الخاص فقط*
        if ($user->hasRole('employee') && $user->employee && $user->employee->id === $employee->id) {
            // ✅ التحقق من أن tenant_id مطابق
            return $user->tenant_id === $employee->tenant_id;
        }

        return false;
    }

    /**
     * تحديد ما إذا كان المستخدم مسموحًا له بحذف *سجل* موظف معين.
     * (عادةً مقتصر على الأدوار الإدارية)
     * ✅ مع التحقق من أن المستخدم والموظف من نفس tenant_id.
     */
    public function delete(User $user, Employee $employee)
    {
        // ✅ التحقق من الدور ومن أن tenant_id مطابق
        return ($user->hasRole(['admin', 'super-admin']) && $user->tenant_id === $employee->tenant_id);
    }

    /**
     * تحديد ما إذا كان المستخدم مسموحًا له باستعادة *سجل* موظف محذوف.
     * (عادةً مقتصر على الأدوار الإدارية)
     * ✅ مع التحقق من أن المستخدم والموظف من نفس tenant_id.
     */
    public function restore(User $user, Employee $employee)
    {
        // ✅ التحقق من الدور ومن أن tenant_id مطابق
        return ($user->hasRole(['admin', 'super-admin']) && $user->tenant_id === $employee->tenant_id);
    }

    /**
     * تحديد ما إذا كان المستخدم مسموحًا له بحذف *سجل* موظف نهائياً.
     * (عادةً مقتصر على السوبر أدمن)
     * ✅ مع التحقق من أن المستخدم والموظف من نفس tenant_id.
     */
    public function forceDelete(User $user, Employee $employee)
    {
        // ✅ التحقق من الدور ومن أن tenant_id مطابق
        return ($user->hasRole('super-admin') && $user->tenant_id === $employee->tenant_id);
    }
}