<?php

namespace App\Policies;

use App\Models\Deduction;
use App\Models\User;
use Illuminate\Auth\Access\HandlesAuthorization;

class DeductionPolicy
{
    use HandlesAuthorization;

    public function viewAny(User $user)
    {
        // إذا كان المستخدم لديه دور 'admin' أو 'super-admin'، سمح له بالعرض
        if ($user->hasRole('admin') || $user->hasRole('super-admin')) {
            return true;
        }

        // ✅ السماح للموظف برؤية قائمة استقطاعاته الخاصة فقط
        return $user->hasRole('employee');
    }

    public function view(User $user, Deduction $deduction)
    {
        // إذا كان المستخدم لديه دور 'admin' أو 'super-admin'، سمح له بالعرض
        if ($user->hasRole('admin') || $user->hasRole('super-admin')) {
            return true;
        }

        // ✅ التحقق مما إذا كان المستخدم مرتبطًا ب(employee) و
        // أن (employee) المرتبط بـ (deduction) هو نفسه (employee) المرتبط بالمستخدم
        return $user->employee && $user->employee->id === $deduction->employee_id;
    }

    public function create(User $user)
    {
        // إذا كان المستخدم لديه دور 'admin' أو 'super-admin'، سمح له بالإنشاء
        if ($user->hasRole('admin') || $user->hasRole('super-admin')) {
            return true;
        }

        // ✅ السماح للموظف بإنشاء استقطاع لنفسه فقط
        return $user->employee !== null;
    }

    public function update(User $user, Deduction $deduction)
    {
        // إذا كان المستخدم لديه دور 'admin' أو 'super-admin'، سمح له بالتعديل
        if ($user->hasRole('admin') || $user->hasRole('super-admin')) {
            return true;
        }

        // ✅ التحقق مما إذا كان المستخدم مرتبطًا ب(employee) و
        // أن (employee) المرتبط بـ (deduction) هو نفسه (employee) المرتبط بالمستخدم
        return $user->employee && $user->employee->id === $deduction->employee_id;
    }

    public function delete(User $user, Deduction $deduction)
    {
        // إذا كان المستخدم لديه دور 'admin' أو 'super-admin'، سمح له بالحذف
        if ($user->hasRole('admin') || $user->hasRole('super-admin')) {
            return true;
        }

        // ✅ التحقق مما إذا كان المستخدم مرتبطًا ب(employee) و
        // أن (employee) المرتبط بـ (deduction) هو نفسه (employee) المرتبط بالمستخدم
        return $user->employee && $user->employee->id === $deduction->employee_id;
    }

    public function apply(User $user, Deduction $deduction)
    {
        // إذا كان المستخدم لديه دور 'admin' أو 'super-admin'، سمح له بالتطبيق
        if ($user->hasRole('admin') || $user->hasRole('super-admin')) {
            return true;
        }

        // ✅ التحقق مما إذا كان المستخدم مرتبطًا ب(employee) و
        // أن (employee) المرتبط بـ (deduction) هو نفسه (employee) المرتبط بالمستخدم
        return $user->employee && $user->employee->id === $deduction->employee_id;
    }

    public function cancel(User $user, Deduction $deduction)
    {
        // إذا كان المستخدم لديه دور 'admin' أو 'super-admin'، سمح له بالإلغاء
        if ($user->hasRole('admin') || $user->hasRole('super-admin')) {
            return true;
        }

        // ✅ التحقق مما إذا كان المستخدم مرتبطًا ب(employee) و
        // أن (employee) المرتبط بـ (deduction) هو نفسه (employee) المرتبط بالمستخدم
        return $user->employee && $user->employee->id === $deduction->employee_id;
    }

    public function restore(User $user, Deduction $deduction)
    {
        // إذا كان المستخدم لديه دور 'admin' أو 'super-admin'، سمح له بالاستعادة
        if ($user->hasRole('admin') || $user->hasRole('super-admin')) {
            return true;
        }

        // ✅ التحقق مما إذا كان المستخدم مرتبطًا ب(employee) و
        // أن (employee) المرتبط بـ (deduction) هو نفسه (employee) المرتبط بالمستخدم
        return $user->employee && $user->employee->id === $deduction->employee_id;
    }

    public function forceDelete(User $user, Deduction $deduction)
    {
        // عادةً لا يُسمح بالحذف النهائي إلا للسوبر أدمن
        return $user->hasRole('super-admin'); // أو false إذا لم ترد السماح به مطلقاً
    }
}