<?php

namespace App\Policies;

use App\Models\User;
use App\Models\Attendance;
use Illuminate\Auth\Access\HandlesAuthorization;

class AttendancePolicy
{
    use HandlesAuthorization;

    /**
     * تحديد ما إذا كان المستخدم مسموحًا له بعرض *قائمة* سجلات الحضور.
     * يُستخدم هذا في AttendanceController@index.
     */
    public function viewAny(User $user)
    {
        // يُسمح للادمن والسوبر ادمن والـ HR
        if ($user->hasRole(['admin', 'super-admin', 'super_admin', 'hr'])) {
            return true;
        }

        // يُسمح للموظف العادي برؤية *قائمة* الحضور (لكن المراقب يجب أن يُصفّي النتائج لاحقًا)
        if ($user->hasRole('employee')) {
            return true;
        }

        // التحقق من صلاحيات مخصصة (إن وُجدت)
        // return $user->can('view_attendance');
        // للتبسيط، نعتمد على الأدوار فقط حسب السياق.

        return false;
    }

    /**
     * تحديد ما إذا كان المستخدم مسموحًا له بعرض *سجل حضور معين* (مثل في صفحة show).
     * هذا هو المكان الذي نمنع فيه "حسن" من رؤية حضور "علي".
     */
    public function view(User $user, Attendance $attendance)
    {
        // يُسمح للادمن والسوبر ادمن والـ HR برؤية أي سجل حضور
        if ($user->hasRole(['admin', 'super-admin', 'super_admin', 'hr'])) {
            return true;
        }

        // يُسمح للموظف العادي برؤية *سجل حضوره الخاص فقط*
        // التحقق مما إذا كان المستخدم مرتبطًا بسجل موظف
        // وأيضًا التحقق مما إذا كان هذا السجل هو نفسه $attendance->employee_id
        if ($user->hasRole('employee') && $user->employee && $user->employee->id === $attendance->employee_id) {
            return true;
        }

        // في جميع الحالات الأخرى، لا يُسمح بالعرض
        return false;
    }

    /**
     * تحديد ما إذا كان المستخدم مسموحًا له بإنشاء سجل حضور.
     * (عادةً مقتصر على الأدوار الإدارية)
     */
    public function create(User $user)
    {
        return $user->hasRole(['admin', 'super-admin', 'super_admin', 'hr']);
    }

    /**
     * تحديد ما إذا كان المستخدم مسموحًا له بتحديث *سجل حضور معين*.
     * (عادةً مقتصر على الأدوار الإدارية)
     */
    public function update(User $user, Attendance $attendance)
    {
        // يُسمح للادمن والسوبر ادمن والـ HR بتحديث أي سجل
        if ($user->hasRole(['admin', 'super-admin', 'super_admin', 'hr'])) {
            return true;
        }

        // للموظف العادي، لا يُسمح بالتحديث
        // (أو يمكن السماح له بتحديث سجله فقط في حالات محددة، لكن عادةً لا)
        return false;
    }

    /**
     * تحديد ما إذا كان المستخدم مسموحًا له بحذف *سجل حضور معين*.
     * (عادةً مقتصر على الأدوار الإدارية)
     */
    public function delete(User $user, Attendance $attendance)
    {
        // يُسمح للادمن والسوبر ادمن والـ HR بحذف أي سجل
        if ($user->hasRole(['admin', 'super-admin', 'super_admin', 'hr'])) {
            return true;
        }

        // للموظف العادي، لا يُسمح بالحذف
        return false;
    }

    /**
     * تحديد ما إذا كان المستخدم مسموحًا له بالموافقة على حضور.
     * (عادةً مقتصر على الأدوار الإدارية)
     */
    public function approve(User $user, Attendance $attendance)
    {
        return $user->hasRole(['admin', 'super-admin', 'super_admin', 'hr']);
    }

    /**
     * تحديد ما إذا كان المستخدم مسموحًا له بالرفض على حضور.
     * (عادةً مقتصر على الأدوار الإدارية)
     */
    public function reject(User $user, Attendance $attendance)
    {
        return $user->hasRole(['admin', 'super-admin', 'super_admin', 'hr']);
    }

    /**
     * تحديد ما إذا كان المستخدم مسموحًا له باستعادة سجل حضور محذوف.
     * (عادةً مقتصر على الأدوار الإدارية)
     */
    public function restore(User $user, Attendance $attendance)
    {
        return $user->hasRole(['admin', 'super-admin', 'super_admin', 'hr']);
    }

    /**
     * تحديد ما إذا كان المستخدم مسموحًا له بحذف سجل حضور نهائياً.
     * (عادةً مقتصر على السوبر أدمن)
     */
    public function forceDelete(User $user, Attendance $attendance)
    {
        return $user->hasRole(['super-admin', 'super_admin']);
    }

    /**
     * تحديد ما إذا كان المستخدم مسموحًا له باستيراد ملفات الحضور (CSV).
     * هذا يستعمله AttendanceImportController::showForm و import()
     */
    public function import(User $user)
    {
        return $user->hasRole(['admin', 'super-admin', 'super_admin', 'hr']);
    }
}