<?php

namespace App\Policies;

use App\Models\Asset;
use App\Models\User;
use Illuminate\Auth\Access\HandlesAuthorization;

class AssetPolicy
{
    use HandlesAuthorization;

    /**
     * Determine whether the user can view any models.
     *
     * @param  \App\Models\User  $user
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function viewAny(User $user)
    {
        // يمكن للمستخدمين الذين لديهم صلاحية عرض الأصول أو إدارة الأصول رؤية القائمة
        return $user->can('view_assets') || $user->can('manage_assets');
    }

    /**
     * Determine whether the user can view the model.
     *
     * @param  \App\Models\User  $user
     * @param  \App\Models\Asset  $asset
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function view(User $user, Asset $asset)
    {
        // يمكن للمستخدمين الذين لديهم صلاحية عرض الأصول أو إدارة الأصول رؤية تفاصيل الأصل
        // أو إذا كان الأصل مخصص لهذا المستخدم
        return $user->can('view_assets') || 
               $user->can('manage_assets') || 
               $user->employee_id == $asset->employee_id;
    }

    /**
     * Determine whether the user can create models.
     *
     * @param  \App\Models\User  $user
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function create(User $user)
    {
        // فقط المستخدمون الذين لديهم صلاحية إدارة الأصول يمكنهم إنشاء أصول جديدة
        return $user->can('manage_assets');
    }

    /**
     * Determine whether the user can update the model.
     *
     * @param  \App\Models\User  $user
     * @param  \App\Models\Asset  $asset
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function update(User $user, Asset $asset)
    {
        // فقط المستخدمون الذين لديهم صلاحية إدارة الأصول يمكنهم تحديث الأصول
        return $user->can('manage_assets');
    }

    /**
     * Determine whether the user can delete the model.
     *
     * @param  \App\Models\User  $user
     * @param  \App\Models\Asset  $asset
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function delete(User $user, Asset $asset)
    {
        // فقط المستخدمون الذين لديهم صلاحية إدارة الأصول يمكنهم حذف الأصول
        return $user->can('manage_assets');
    }

    /**
     * Determine whether the user can assign the asset to an employee.
     *
     * @param  \App\Models\User  $user
     * @param  \App\Models\Asset  $asset
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function assign(User $user, Asset $asset)
    {
        // فقط المستخدمون الذين لديهم صلاحية تعيين الأصول يمكنهم تعيينها
        // أو من لديهم صلاحية إدارة الأصول بشكل عام
        return $user->can('assign_assets') || $user->can('manage_assets');
    }

    /**
     * Determine whether the user can unassign the asset from an employee.
     *
     * @param  \App\Models\User  $user
     * @param  \App\Models\Asset  $asset
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function unassign(User $user, Asset $asset)
    {
        // فقط المستخدمون الذين لديهم صلاحية إلغاء تعيين الأصول يمكنهم إلغاء تعيينها
        // أو من لديهم صلاحية إدارة الأصول بشكل عام
        return $user->can('assign_assets') || $user->can('manage_assets');
    }

    /**
     * Determine whether the user can mark the asset as under maintenance.
     *
     * @param  \App\Models\User  $user
     * @param  \App\Models\Asset  $asset
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function maintain(User $user, Asset $asset)
    {
        // فقط المستخدمون الذين لديهم صلاحية صيانة الأصول يمكنهم وضعها في الصيانة
        // أو من لديهم صلاحية إدارة الأصول بشكل عام
        return $user->can('maintain_assets') || $user->can('manage_assets');
    }

    /**
     * Determine whether the user can retire the asset.
     *
     * @param  \App\Models\User  $user
     * @param  \App\Models\Asset  $asset
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function retire(User $user, Asset $asset)
    {
        // فقط المستخدمون الذين لديهم صلاحية إيقاف الأصول يمكنهم إيقافها
        // أو من لديهم صلاحية إدارة الأصول بشكل عام
        return $user->can('retire_assets') || $user->can('manage_assets');
    }

    /**
     * Determine whether the user can restore the model.
     *
     * @param  \App\Models\User  $user
     * @param  \App\Models\Asset  $asset
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function restore(User $user, Asset $asset)
    {
        // فقط المستخدمون الذين لديهم صلاحية إدارة الأصول يمكنهم استعادة الأصول المحذوفة
        return $user->can('manage_assets');
    }

    /**
     * Determine whether the user can permanently delete the model.
     *
     * @param  \App\Models\User  $user
     * @param  \App\Models\Asset  $asset
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function forceDelete(User $user, Asset $asset)
    {
        // عادةً لا يُسمح بالحذف النهائي إلا للسوبر أدمن
        return $user->hasRole('super-admin'); // أو false إذا لم ترد السماح به مطلقاً
    }
}