<?php

namespace App\Notifications;

use App\Models\Payroll;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class PayrollPaid extends Notification
{
    use Queueable;

    protected $payroll;

    public function __construct(Payroll $payroll)
    {
        $this->payroll = $payroll;
    }

    public function via($notifiable)
    {
        return ['database', 'mail'];
    }

    public function toMail($notifiable)
    {
        return (new MailMessage)
                    ->subject('تم دفع الراتب - ' . $this->payroll->month->format('F Y'))
                    ->line('مرحباً ' . $this->payroll->employee->name)
                    ->line('تم دفع راتبك لشهر ' . $this->payroll->month->format('F Y'))
                    ->line('تاريخ الدفع: ' . $this->payroll->payment_date->format('Y-m-d'))
                    ->line('الراتب الصافي: ' . number_format($this->payroll->total_salary, 2))
                    ->line('تفاصيل الخصومات:')
                    ->line('- المسحوبات النقدية: ' . number_format($this->payroll->cash_deductions, 2))
                    ->line('- القروض: ' . number_format($this->payroll->loan_deductions, 2))
                    ->line('- المسحوبات العينية: ' . number_format($this->payroll->in_kind_deductions, 2))
                    ->action('عرض التفاصيل', url('/payroll/' . $this->payroll->id))
                    ->line('شكراً لاستخدامك نظام إدارة الموارد البشرية!');
    }

    public function toArray($notifiable)
    {
        return [
            'type' => 'payroll_paid',
            'payroll_id' => $this->payroll->id,
            'employee_name' => $this->payroll->employee->name,
            'month' => $this->payroll->month->format('F Y'),
            'total_salary' => $this->payroll->total_salary,
            'payment_date' => $this->payroll->payment_date->format('Y-m-d'),
            'title_key' => 'notifications.payroll_paid_title',
            'title_params' => [],
            'message_key' => 'notifications.payroll_paid_message',
            'message_params' => ['month' => $this->payroll->month->format('F Y')]
        ];
    }
}
