<?php

namespace App\Notifications;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Messages\DatabaseMessage;
use Illuminate\Notifications\Notification;
use App\Models\Payroll;

class PayrollCreatedNotification extends Notification implements ShouldQueue
{
    use Queueable;

    /**
     * The payroll record.
     */
    public $payroll;

    /**
     * Create a new notification instance.
     */
    public function __construct(Payroll $payroll)
    {
        $this->payroll = $payroll;
    }

    /**
     * Get the notification's delivery channels.
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        $monthDate = \Carbon\Carbon::createFromDate($this->payroll->year, $this->payroll->month, 1)->locale('ar')->format('F Y');
        $currency = get_currency_code();
        
        return (new MailMessage)
            ->subject('📋 تم إنشاء راتب جديد - ' . $this->payroll->employee->name)
            ->greeting('مرحباً ' . $notifiable->name . ',')
            ->line('تم إنشاء راتب جديد للموظف: **' . $this->payroll->employee->name . '**')
            ->line('الشهر: **' . $monthDate . '**')
            ->line('المبلغ الإجمالي: **' . number_format($this->payroll->total_salary, 2) . ' ' . $currency . '**')
            ->line('المبلغ الصافي: **' . number_format($this->payroll->net_salary, 2) . ' ' . $currency . '**')
            ->line('الحالة: **' . ($this->payroll->status === 'pending' ? 'قيد الانتظار' : 'مدفوع') . '**')
            ->action('عرض تفاصيل الراتب', route('payroll.show', $this->payroll->id))
            ->line('يرجى مراجعة الراتب والموافقة عليه إن لزم الأمر.');
    }

    /**
     * Get the database representation of the notification.
     */
    public function toDatabase(object $notifiable): DatabaseMessage
    {
        $monthDate = \Carbon\Carbon::createFromDate($this->payroll->year, $this->payroll->month, 1)->locale('ar')->format('F Y');
        $currency = get_currency_code();
        
        return new DatabaseMessage([
            'title' => 'تم إنشاء راتب جديد',
            'message' => 'تم إنشاء راتب جديد للموظف ' . $this->payroll->employee->name . ' للشهر ' . $monthDate . ' بمبلغ ' . number_format($this->payroll->net_salary, 2) . ' ' . $currency,
            'payroll_id' => $this->payroll->id,
            'employee_id' => $this->payroll->employee_id,
            'employee_name' => $this->payroll->employee->name,
            'amount' => $this->payroll->net_salary,
            'month' => $this->payroll->month,
            'year' => $this->payroll->year,
            'type' => 'payroll_created',
            'status' => 'pending',
        ]);
    }
}
