<?php

namespace App\Notifications;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Messages\DatabaseMessage;
use Illuminate\Notifications\Notification;
use App\Models\Loan;

class LoanRejectedNotification extends Notification implements ShouldQueue
{
    use Queueable;

    /**
     * The loan request.
     */
    public $loan;

    /**
     * The rejection reason.
     */
    public $rejectionReason;

    /**
     * Create a new notification instance.
     */
    public function __construct(Loan $loan, $rejectionReason = null)
    {
        $this->loan = $loan;
        $this->rejectionReason = $rejectionReason;
    }

    /**
     * Get the notification's delivery channels.
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        return (new MailMessage)
            ->subject('❌ تم رفض طلب القرض الخاص بك')
            ->greeting('مرحباً ' . $this->loan->employee->name . ',')
            ->line('للأسف، تم رفض طلب القرض الخاص بك.')
            ->line('المبلغ المطلوب: **' . number_format($this->loan->amount, 2) . ' ' . get_currency_code() . '**')
            ->line('عدد الأقساط: **' . $this->loan->number_of_installments . '**')
            ->when($this->rejectionReason, function ($message) {
                $message->line('السبب: **' . $this->rejectionReason . '**');
            })
            ->action('عرض طلب القرض', route('loans.show', $this->loan->id))
            ->line('يمكنك تقديم طلب جديد في وقت لاحق.');
    }

    /**
     * Get the database representation of the notification.
     */
    public function toDatabase(object $notifiable): DatabaseMessage
    {
        return new DatabaseMessage([
            'title' => 'تم رفض طلب القرض',
            'message' => 'تم رفض طلب القرض بمبلغ ' . number_format($this->loan->amount, 2) . ' ' . get_currency_code(),
            'loan_id' => $this->loan->id,
            'employee_id' => $this->loan->employee_id,
            'rejection_reason' => $this->rejectionReason,
            'type' => 'loan_rejected',
            'status' => 'rejected',
        ]);
    }
}
