<?php

namespace App\Notifications;

use App\Models\Loan;
use App\Models\LoanInstallment;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class LoanInstallmentPaidNotification extends Notification
{
    use Queueable;

    protected $installment;
    protected $loan;

    public function __construct(LoanInstallment $installment, Loan $loan = null)
    {
        $this->installment = $installment;
        $this->loan = $loan ?? $installment->loan;
    }

    public function via($notifiable)
    {
        return ['database', 'mail'];
    }

    public function toMail($notifiable)
    {
        $paidDate = $this->installment->paid_date ? $this->installment->paid_date->format('Y-m-d') : now()->format('Y-m-d');

        return (new MailMessage)
                    ->subject('تم دفع قسط القرض - ' . $paidDate)
                    ->line('مرحباً ' . optional($this->loan->employee)->name)
                    ->line('تم دفع قسط القرض رقم ' . $this->loan->loan_number)
                    ->line('رقم القسط: ' . $this->installment->installment_number)
                    ->line('المبلغ المدفوع: ' . number_format($this->installment->amount, 2))
                    ->line('طريقة الدفع: ' . ($this->installment->payment_method ?? '-'))
                    ->line('تاريخ الدفع: ' . $paidDate)
                    ->action('عرض تفاصيل القرض', url('/loans/' . $this->loan->id))
                    ->line('شكراً لاستخدامك نظامنا.');
    }

    public function toArray($notifiable)
    {
        return [
            'type' => 'loan_installment_paid',
            'loan_id' => $this->loan->id,
            'installment_id' => $this->installment->id,
            'installment_number' => $this->installment->installment_number,
            'amount' => $this->installment->amount,
            'paid_date' => $this->installment->paid_date ? $this->installment->paid_date->format('Y-m-d') : now()->format('Y-m-d'),
            'payment_method' => $this->installment->payment_method ?? null,
            'title_key' => 'notifications.loan_installment_paid_title',
            'title_params' => [],
            'message_key' => 'notifications.loan_installment_paid_message',
            'message_params' => [
                'installment_number' => $this->installment->installment_number,
                'loan_number' => $this->loan->loan_number,
                'paid_date' => $this->installment->paid_date ? $this->installment->paid_date->format('Y-m-d') : now()->format('Y-m-d'),
            ],
        ];
    }
}
