<?php

namespace App\Notifications;

use App\Models\LoanInstallment;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class LoanInstallmentDue extends Notification
{
    use Queueable;

    protected $installment;

    public function __construct(LoanInstallment $installment)
    {
        $this->installment = $installment;
    }

    public function via($notifiable)
    {
        return ['database', 'mail'];
    }

    public function toMail($notifiable)
    {
        return (new MailMessage)
                    ->subject('قسط قرض مستحق الدفع - ' . $this->installment->due_date->format('Y-m-d'))
                    ->line('مرحباً ' . $this->installment->loan->employee->name)
                    ->line('هذا إشعار بأن قسط قرضك رقم ' . $this->installment->loan->loan_number . ' مستحق الدفع')
                    ->line('رقم القسط: ' . $this->installment->installment_number)
                    ->line('المبلغ المستحق: ' . number_format($this->installment->amount, 2))
                    ->line('تاريخ الاستحقاق: ' . $this->installment->due_date->format('Y-m-d'))
                    ->action('عرض تفاصيل القرض', url('/loans/' . $this->installment->loan->id))
                    ->line('يرجى التأكد من دفع القسط في الوقت المحدد لتجنب الغرامات!');
    }

    public function toArray($notifiable)
    {
        return [
            'type' => 'loan_installment_due',
            'loan_id' => $this->installment->loan->id,
            'installment_id' => $this->installment->id,
            'installment_number' => $this->installment->installment_number,
            'amount' => $this->installment->amount,
            'due_date' => $this->installment->due_date->format('Y-m-d'),
            'title_key' => 'notifications.loan_installment_due_title',
            'title_params' => [],
            'message_key' => 'notifications.loan_installment_due_message',
            'message_params' => [
                'installment_number' => $this->installment->installment_number,
                'loan_number' => $this->installment->loan->loan_number,
                'due_date' => $this->installment->due_date->format('Y-m-d'),
            ],
        ];
    }
}
