<?php

namespace App\Notifications;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;
use App\Models\Leave; // ✅ استيراد نموذج الإجازة

class LeaveRequestSubmittedNotification extends Notification
{
    use Queueable;

    protected $leave; // ✅ خاصية لتخزين كائن الإجازة

    /**
     * إنشاء نسخة جديدة من الإشعار.
     *
     * @param Leave $leave كائن الإجازة المرتبطة
     * @return void
     */
    public function __construct(Leave $leave)
    {
        $this->leave = $leave; // ✅ تعيين كائن الإجازة
    }

    /**
     * تحديد القنوات التي سيتم إرسال الإشعار عبرها.
     * (مترجم: تحديد طرق توصيل الإشعار: قاعدة بيانات، بريد، رسالة نصية...)
     *
     * @param  mixed  $notifiable الكيان الذي سيتم إعلامه (عادةً نموذج المستخدم User)
     * @return array
     */
    public function via($notifiable)
    {
        // ✅ إرسال الإشعار عبر قاعدة البيانات والبريد الإلكتروني
        return ['database', 'mail'];
        // يمكنك تعديل هذا لتشمل 'broadcast' أو 'nexmo' (SMS) إذا لزم
    }

    /**
     * تمثيل الإشعار كرسالة بريد إلكتروني.
     * (مترجم: كيفية عرض الإشعار كرسالة بريد إلكتروني)
     *
     * @param  mixed  $notifiable الكيان الذي سيتم إعلامه
     * @return \Illuminate\Notifications\Messages\MailMessage
     */
    public function toMail($notifiable)
    {
        // ✅ بناء رسالة البريد الإلكتروني
        return (new MailMessage)
                    ->subject('طلب إجازة جديد - ' . $this->leave->employee->name) // موضوع البريد
                    ->line('مرحباً ' . $this->leave->employee->name) // سطر نصي
                    ->line('لقد قمت بتقديم طلب إجازة جديد.') // سطر نصي
                    ->line('نوع الإجازة: ' . __('leaves.types.' . $this->leave->leave_type)) // سطر نصي مع ترجمة نوع الإجازة
                    ->line('الفترة: من ' . $this->leave->start_date->format('Y-m-d') . ' إلى ' . $this->leave->end_date->format('Y-m-d')) // سطر نصي مع تنسيق التواريخ
                    ->line('عدد الأيام: ' . $this->leave->days_count) // سطر نصي لعدد الأيام
                    ->line('السبب: ' . $this->leave->reason) // سطر نصي للسبب
                    ->action('عرض تفاصيل الطلب', url('/leaves/' . $this->leave->id)) // زر برابط لعرض التفاصيل
                    ->line('شكراً لاستخدامك نظام إدارة الموارد البشرية!'); // سطر نصي ختامي
    }

    /**
     * تمثيل الإشعار كبيانات مصفوفة لتخزينها في قاعدة البيانات.
     * (مترجم: كيفية تخزين الإشعار في قاعدة بيانات `notifications`)
     *
     * @param  mixed  $notifiable الكيان الذي سيتم إعلامه
     * @return array
     */
    public function toDatabase($notifiable)
    {
        // Store keys and params so UI can translate based on current locale
        return [
            'title_key' => 'notifications.leave_request_submitted_title',
            'title_params' => [],
            'message_key' => 'notifications.leave_request_submitted_message',
            'message_params' => [],
            'leave_id' => $this->leave->id,
            'employee_name' => $this->leave->employee->name,
            'leave_type' => $this->leave->leave_type,
            'start_date' => $this->leave->start_date->format('Y-m-d'),
            'end_date' => $this->leave->end_date->format('Y-m-d'),
            'days_count' => $this->leave->days_count,
            'reason' => $this->leave->reason,
            'url' => url('/leaves/' . $this->leave->id),
            'type' => 'leave',
            'sender' => 'system',
        ];
    }
}