<?php

namespace App\Notifications;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Messages\DatabaseMessage;
use Illuminate\Notifications\Notification;
use App\Models\Leave;

class LeaveRejectedNotification extends Notification implements ShouldQueue
{
    use Queueable;

    /**
     * The leave request.
     */
    public $leave;

    /**
     * Create a new notification instance.
     */
    public function __construct(Leave $leave)
    {
        $this->leave = $leave;
    }

    /**
     * Get the notification's delivery channels.
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        $startDate = $this->leave->start_date ? \Carbon\Carbon::parse($this->leave->start_date)->locale('ar')->format('d F Y') : 'N/A';
        $endDate = $this->leave->end_date ? \Carbon\Carbon::parse($this->leave->end_date)->locale('ar')->format('d F Y') : 'N/A';
        
        return (new MailMessage)
            ->subject('❌ تم رفض طلب إجازتك')
            ->greeting('مرحباً ' . $this->leave->employee->name . ',')
            ->line('تم رفض طلب إجازتك للأسف.')
            ->line('نوع الإجازة: **' . ($this->leave->leaveType->name ?? 'N/A') . '**')
            ->line('من: **' . $startDate . '**')
            ->line('إلى: **' . $endDate . '**')
            ->when($this->leave->rejection_reason, function ($message) {
                $message->line('السبب: **' . $this->leave->rejection_reason . '**');
            })
            ->action('عرض طلب الإجازة', route('leaves.show', $this->leave->id))
            ->line('يمكنك تقديم طلب جديد في وقت لاحق.');
    }

    /**
     * Get the database representation of the notification.
     */
    public function toDatabase(object $notifiable): DatabaseMessage
    {
        return new DatabaseMessage([
            'title' => 'تم رفض طلب الإجازة',
            'message' => 'تم رفض طلب إجازتك من ' . $this->leave->start_date . ' إلى ' . $this->leave->end_date,
            'leave_id' => $this->leave->id,
            'employee_id' => $this->leave->employee_id,
            'rejection_reason' => $this->leave->rejection_reason,
            'type' => 'leave_rejected',
            'status' => 'rejected',
        ]);
    }
}
