<?php

namespace App\Notifications;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Messages\DatabaseMessage;
use Illuminate\Notifications\Notification;
use App\Models\Leave;

class LeaveApprovedNotification extends Notification implements ShouldQueue
{
    use Queueable;

    /**
     * The leave request.
     */
    public $leave;

    /**
     * Create a new notification instance.
     */
    public function __construct(Leave $leave)
    {
        $this->leave = $leave;
    }

    /**
     * Get the notification's delivery channels.
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        $startDate = $this->leave->start_date ? \Carbon\Carbon::parse($this->leave->start_date)->locale('ar')->format('d F Y') : 'N/A';
        $endDate = $this->leave->end_date ? \Carbon\Carbon::parse($this->leave->end_date)->locale('ar')->format('d F Y') : 'N/A';
        
        return (new MailMessage)
            ->subject('✅ تمت الموافقة على طلب إجازتك')
            ->greeting('مرحباً ' . $this->leave->employee->name . ',')
            ->line('تم الموافقة على طلب إجازتك بنجاح!')
            ->line('نوع الإجازة: **' . ($this->leave->leaveType->name ?? 'N/A') . '**')
            ->line('من: **' . $startDate . '**')
            ->line('إلى: **' . $endDate . '**')
            ->line('عدد الأيام: **' . $this->leave->number_of_days . ' يوم**')
            ->action('عرض طلب الإجازة', route('leaves.show', $this->leave->id))
            ->line('شكراً لك.');
    }

    /**
     * Get the database representation of the notification.
     */
    public function toDatabase(object $notifiable): DatabaseMessage
    {
        return new DatabaseMessage([
            'title' => 'تمت الموافقة على طلب الإجازة',
            'message' => 'تم الموافقة على طلب إجازتك من ' . $this->leave->start_date . ' إلى ' . $this->leave->end_date,
            'leave_id' => $this->leave->id,
            'employee_id' => $this->leave->employee_id,
            'type' => 'leave_approved',
            'status' => 'approved',
        ]);
    }
}
