<?php

namespace App\Notifications;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Messages\DatabaseMessage;
use Illuminate\Notifications\Notification;
use App\Models\Employee;
use Carbon\Carbon;

class AttendanceReminder extends Notification implements ShouldQueue
{
    use Queueable;

    /**
     * The employee who missed attendance.
     */
    public $employee;

    /**
     * The date of the missed attendance.
     */
    public $date;

    /**
     * The user who sent the reminder.
     */
    public $sentBy;

    /**
     * Create a new notification instance.
     */
    public function __construct(Employee $employee, $date, $sentBy = null)
    {
        $this->employee = $employee;
        $this->date = $date;
        $this->sentBy = $sentBy;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        $dateFormatted = Carbon::parse($this->date)->locale('ar')->format('d F Y');
        
        return (new MailMessage)
            ->subject('تذكير: تسجيل الحضور')
            ->greeting('مرحباً ' . $this->employee->name . ',')
            ->line('لم يتم تسجيل حضورك في التاريخ: **' . $dateFormatted . '**')
            ->line('يُرجى التأكد من تسجيل حضورك في الوقت المحدد.')
            ->action('عرض التقويم', route('employees.calendar', $this->employee->id) . '?month=' . Carbon::parse($this->date)->month . '&year=' . Carbon::parse($this->date)->year)
            ->line('شكراً لك.');
    }

    /**
     * Get the database representation of the notification.
     */
    public function toDatabase(object $notifiable): DatabaseMessage
    {
        $dateFormatted = Carbon::parse($this->date)->locale('ar')->format('d F Y');
        
        return new DatabaseMessage([
            'title' => 'تذكير تسجيل الحضور',
            'message' => 'لم يتم تسجيل حضورك في: ' . $dateFormatted,
            'employee_id' => $this->employee->id,
            'employee_name' => $this->employee->name,
            'date' => $this->date,
            'sent_by' => $this->sentBy,
            'type' => 'attendance_reminder',
        ]);
    }
}
