<?php

namespace App\Models;

// --- Traits ---
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens;

// --- Models for Relationships ---
use App\Models\Employee;
use App\Models\Role; // Import Role Model
use Illuminate\Notifications\DatabaseNotification as Notification;

class User extends Authenticatable
{
    use HasApiTokens, HasFactory, Notifiable;

    /**
     * الحقول المسموح تعبئتها
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'name',
        'email',
        'password',
        'employee_id',
        'company_id',
        'role',
        'status',
        'last_activity',
        'permissions',
        'tenant_id',
        // حقول التحكم في المستخدم
        'is_active',
        'suspended_at',
        'suspension_reason',
        'suspended_by',
        'last_login_at',
        'last_login_ip',
        'login_attempts',
        'locked_until',
        'access_permissions',
        'can_login',
        'deactivation_reason',
        'deactivated_by',
        'deactivated_at',
    ];

    /**
     * الحقول المخفية عند التحويل إلى JSON
     *
     * @var array<int, string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * التحويل التلقائي للحقول
     *
     * @var array<string, string>
     */
    protected $casts = [
        'email_verified_at' => 'datetime',
        'password'          => 'hashed',
        'last_activity'     => 'datetime',
        'permissions'       => 'array', // <-- تم الإضافة: تحويل JSON إلى مصفوفة عند الاسترجاع
        'access_permissions' => 'array',
        'suspended_at' => 'datetime',
        'last_login_at' => 'datetime',
        'locked_until' => 'datetime',
        'deactivated_at' => 'datetime',
        'is_active' => 'boolean',
        'can_login' => 'boolean',
        // 'tenant_id' => 'integer', // <-- يمكن إضافته إذا كنت ترغب في تحويله تلقائيًا (اختياري)
    ];

    /**
     * Boot the model (يتم تنفيذه عند تحميل النموذج).
     * لربط المستخدم تلقائيًا بـ tenant_id من الجلسة أو المستخدم الحالي.
     */
    protected static function boot()
    {
        parent::boot();

        // قبل إنشاء المستخدم، إذا لم يكن محددًا tenant_id، استخدم الـ tenant_id من الجلسة أو المستخدم المصادق عليه
        static::creating(function ($user) {
            if (is_null($user->tenant_id)) {
                // محاولة الحصول على tenant_id من المستخدم المصادق عليه أولاً (مثلاً عند إنشاء موظف جديد من لوحة الإدارة)
                if (auth()->check() && auth()->user()->tenant_id) {
                    $user->tenant_id = auth()->user()->tenant_id;
                } elseif (session()->has('current_tenant_id')) {
                    // بعد ذلك، محاولة الحصول من الجلسة (مثلاً عند تسجيل مستخدم جديد من صفحة التسجيل في subdomain)
                    $user->tenant_id = session('current_tenant_id');
                }
                // إذا لم يُعثر على tenant_id في أي من الحالتين، سيبقى null (يمكنك التعامل مع هذا في Validation أو Controller)
            }
        });
    }

    // ===================== العلاقات =====================

    /**
     * العلاقة مع الشركة
     */
    public function company()
    {
        return $this->belongsTo(Company::class);
    }

    /**
     * العلاقة مع الموظف
     */
    public function employee()
    {
        return $this->belongsTo(Employee::class, 'employee_id');
    }

    /**
     * العلاقة مع الإشعارات
     */
    public function notifications()
    {
        return $this->morphMany(Notification::class, 'notifiable')->latest();
    }

    /**
     * العلاقة مع الأدوار (Many-to-Many)
     * ملاحظة: إذا كنت ترغب في عزل الأدوار حسب tenant، يجب إضافة tenant_id إلى جدول user_roles.
     * هذا يتطلب تعديل في العلاقة أدناه أو في جدول pivot.
     * للبساطة، نفترض أن الأدوار مشتركة أو معرفة مسبقًا لكل tenant.
     */
    public function roles()
    {
        return $this->belongsToMany(Role::class, 'user_role');
    }

    /**
     * Direct user permissions via pivot table `user_permission`.
     */
    public function permissionsRel()
    {
        return $this->belongsToMany(Permission::class, 'user_permission');
    }

    /**
     * Explicit denied permissions for a user.
     */
    public function deniedPermissions()
    {
        return $this->belongsToMany(Permission::class, 'user_permission_denied');
    }

    /**
     * العلاقة مع سجل النشاط (Placeholder)
     * ملاحظة: يجب إضافة tenant_id إلى جدول activities.
     */
    public function activities()
    {
        // مثال على العلاقة بعد إضافة tenant_id إلى جدول activities
        // return $this->hasMany(Activity::class);
        return null;
    }

    // ===================== فحص الأدوار =====================

    /**
     * التحقق من امتلاك المستخدم لدور معين
     * ملاحظة: التحقق سيشمل فقط الأدوار المرتبطة بـ tenant_id الخاص بالمستخدم.
     * يدعم كلا الصيغتين: 'super-admin' و 'super_admin'
     */
    public function hasRole($roleName): bool
    {
        // تطبيع اسم الدور (تحويل الشرطة إلى شرطة سفلية للمقارنة)
        $normalizedUserRole = str_replace('-', '_', $this->role);
        $normalizedCheckRole = str_replace('-', '_', $roleName);
        
        // التحقق من الدور مباشرة من عمود role
        return $normalizedUserRole === $normalizedCheckRole;
    }

    /**
     * التحقق من أي دور ضمن أدوار معينة
     */
    public function hasAnyRole($roles): bool
    {
        if (is_string($roles)) {
            $roles = [$roles];
        }

        foreach ($roles as $role) {
            if ($this->hasRole($role)) {
                return true;
            }
        }

        return false;
    }

    // ===================== فحص الصلاحيات =====================

    /**
     * التحقق من امتلاك المستخدم لصلاحية معينة
     * أولاً: يبحث في صلاحيات المستخدم الفردية
     * ثانياً (إذا لم توجد صلاحيات فردية): يبحث في صلاحيات أدوار المستخدم
     */
    public function hasPermission($permission): bool
    {
        // 0. If explicitly denied, return false
        if ($this->deniedPermissions()->where('name', $permission)->exists()) {
            return false;
        }

        // 1. تحقق من صلاحيات المستخدم الفردية من pivot table
        if ($this->permissionsRel()->where('name', $permission)->exists()) {
            return true;
        }

        // 2. تحقق من صلاحيات الأدوار关联 عبر pivot role_permission
        $roles = $this->roles()->with('permissions')->get();
        foreach ($roles as $role) {
            // 1) If the Role model has a JSON/casted `permissions` attribute (legacy),
            //    it will be returned as an array and will shadow the relation with the same name.
            //    Handle that case first to avoid calling collection methods on an array.
            if (is_array($role->permissions) && in_array($permission, $role->permissions)) {
                return true;
            }

            // 2) Otherwise try to use the relation. If the relation was eager-loaded,
            //    use the loaded relation collection to avoid extra queries.
            if ($role->relationLoaded('permissions')) {
                $rolePerms = $role->getRelation('permissions');
                if ($rolePerms instanceof \Illuminate\Support\Collection && $rolePerms->pluck('name')->contains($permission)) {
                    return true;
                }
            } else {
                // Relation not loaded — query it directly and check.
                if ($role->permissions()->pluck('name')->contains($permission)) {
                    return true;
                }
            }
        }

        // 3. إذا لا شيء، تحقق الحقل القديم JSON إن وُجد (fallback)
        if (is_array($this->permissions) && in_array($permission, $this->permissions)) {
            return true;
        }

        return false;
    }

    /**
     * متوافق مع @can و Gate
     */
    public function can($permissionName, $arguments = [])
    {
        // إذا كانت الصلاحية '*' فهذا يعني "جميع الصلاحيات"
        if ($permissionName === '*') {
            return true;
        }

        return $this->hasPermission($permissionName);
    }

    // ===================== وظائف مساعدة =====================

    /**
     * الاسم الكامل
     */
    public function getFullNameAttribute()
    {
        return $this->name;
    }

    
    public function isInactive()
    {
        return $this->status === 'inactive';
    }

    public function isPending()
    {
        return $this->status === 'pending';
    }

    /**
     * تحديث النشاط الأخير
     */
    public function updateLastActivity()
    {
        $this->update(['last_activity' => now()]);
    }

    /**
     * الصورة الرمزية
     */
    public function getAvatarAttribute()
    {
        if ($this->employee && $this->employee->photo) {
            return asset('storage/' . $this->employee->photo);
        }

        return 'https://ui-avatars.com/api/?name=' . urlencode($this->name) . '&background=4e73df&color=ffffff';
    }

    /**
     * الأدوار
     */
    public function getRoleNames()
    {
        return $this->roles->pluck('name')->toArray();
    }

    /**
     * جميع الصلاحيات من جميع الأدوار (ومن الصلاحيات الفردية)
     */
    public function getAllPermissions()
    {
        $permissions = [];
        // أضف الصلاحيات الفردية أولاً
        if (is_array($this->permissions)) {
            $permissions = array_merge($permissions, $this->permissions);
        }
        // أضف صلاحيات الأدوار
        foreach ($this->roles as $role) {
            if (is_array($role->permissions)) {
                $permissions = array_merge($permissions, $role->permissions);
            }
        }
        return array_unique($permissions);
    }

    /**
     * التحقق من الأدمن (Super Admin)
     * ملاحظة: يمكنك تعديل هذا ليعمل على مستوى tenant أو system-wide.
     */
    public function isAdmin()
    {
        return $this->hasRole('admin') || $this->hasRole('super-admin'); // أو أي اسم آخر للدور الرئيسي
    }

    /**
     * التحقق من المستخدم الحالي
     */
    public function isCurrent()
    {
        return $this->id === auth()->id();
    }

    // ===================== الإشعارات =====================

    public function unreadNotifications()
    {
        // ملاحظة: يجب التأكد من أن جدول notifications يحتوي على tenant_id
        // وربما استخدام Global Scope هنا أيضًا
        return $this->notifications()->whereNull('read_at');
    }

    public function getUnreadNotificationsCountAttribute()
    {
        return $this->unreadNotifications()->count();
    }

    public function latestNotifications($limit = 5)
    {
        // ملاحظة: يجب التأكد من أن جدول notifications يحتوي على tenant_id
        // وربما استخدام Global Scope هنا أيضًا
        return $this->notifications()->latest()->limit($limit);
    }

    // ===================== تسجيل الدخول =====================

    public function getLastLoginAttribute()
    {
        if ($this->activities()) {
            return $this->activities()
                        ->where('description', 'login')
                        ->latest()
                        ->first()?->created_at;
        }
        return null;
    }

    public function isLoggedInRecently()
    {
        if (!$this->last_activity) {
            return false;
        }

        return $this->last_activity->diffInMinutes(now()) < 30;
    }

    public function getConnectionStatusAttribute()
    {
        if ($this->isLoggedInRecently()) {
            return 'online';
        } elseif ($this->last_activity) {
            return 'away';
        } else {
            return 'offline';
        }
    }

    // ===================== أخرى =====================

    public function isEmailVerified()
    {
        return !is_null($this->email_verified_at);
    }

    public function isSuperUser()
    {
        // دعم كل من 'super_admin' و 'super-admin'
        return $this->hasAnyRole(['super_admin', 'super-admin']); // أو أي اسم آخر لدور "السوبر أدمن"
    }

    public function scopeSearch($query, $term)
    {
        return $query->where(function ($q) use ($term) {
            $q->where('name', 'like', "%{$term}%")
              ->orWhere('email', 'like', "%{$term}%");
        });
    }

    // دوال مساعدة للتحكم في المستخدم
    public function isActive()
    {
        return $this->status === 'active' && $this->is_active && $this->can_login && !$this->isSuspended() && !$this->isLocked();
    }

    public function isSuspended()
    {
        return !is_null($this->suspended_at);
    }

    public function isLocked()
    {
        return !is_null($this->locked_until) && now()->lessThan($this->locked_until);
    }

    public function canLogin()
    {
        return $this->isActive() && $this->can_login;
    }

    public function suspend($reason, $suspendedBy)
    {
        $this->suspended_at = now();
        $this->suspension_reason = $reason;
        $this->suspended_by = $suspendedBy;
        $this->can_login = false;
        $this->save();
    }

    public function unsuspend()
    {
        $this->suspended_at = null;
        $this->suspension_reason = null;
        $this->suspended_by = null;
        $this->can_login = true;
        $this->save();
    }

    public function deactivate($reason, $deactivatedBy)
    {
        $this->is_active = false;
        $this->can_login = false;
        $this->deactivation_reason = $reason;
        $this->deactivated_by = $deactivatedBy;
        $this->deactivated_at = now();
        $this->save();
    }

    public function activate()
    {
        $this->is_active = true;
        $this->can_login = true;
        $this->deactivation_reason = null;
        $this->deactivated_by = null;
        $this->deactivated_at = null;
        $this->save();
    }

    public function recordLogin($ip = null)
    {
        $this->last_login_at = now();
        $this->last_login_ip = $ip;
        $this->login_attempts = 0;
        $this->locked_until = null;
        $this->save();

        // تسجيل دخول للشركة
        if ($this->company) {
            $this->company->recordLogin();
        }
    }

    public function recordFailedLogin()
    {
        $this->login_attempts++;
        
        // قفل الحساب بعد 5 محاولات فاشلة
        if ($this->login_attempts >= 5) {
            $this->locked_until = now()->addMinutes(30);
        }
        
        $this->save();
    }

    public function getStatusColor()
    {
        if ($this->isLocked()) return 'danger';
        if ($this->isSuspended()) return 'warning';
        if (!$this->is_active) return 'secondary';
        if (!$this->can_login) return 'warning';
        return 'success';
    }

    public function getStatusText()
    {
        if ($this->isLocked()) return 'مقفول';
        if ($this->isSuspended()) return 'موقوف';
        if (!$this->is_active) return 'غير نشط';
        if (!$this->can_login) return 'لا يمكن الدخول';
        return 'نشط';
    }
}