<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class TrainingProgram extends Model
{
    use HasFactory, SoftDeletes;

    protected $table = 'training_programs';

    protected $fillable = [
        'title',
        'description',
        'category',
        'trainer_id',
        'location',
        'duration',
        'duration_unit',
        'capacity',
        'start_date',
        'end_date',
        'status',
        'completion_percentage'
    ];

    protected $casts = [
        'start_date' => 'date',
        'end_date' => 'date',
        'duration' => 'integer',
        'capacity' => 'integer',
        'completion_percentage' => 'decimal:2'
    ];

    protected $dates = [
        'start_date',
        'end_date',
        'created_at',
        'updated_at',
        'deleted_at'
    ];

    // العلاقة مع المدرب (مستخدم)
    public function trainer()
    {
        return $this->belongsTo(User::class, 'trainer_id');
    }

    // العلاقة مع التسجيلات
    public function enrollments()
    {
        return $this->hasMany(TrainingEnrollment::class, 'program_id');
    }

    // العلاقة مع الموظفين من خلال التسجيلات
    public function employees()
    {
        return $this->belongsToMany(Employee::class, 'training_enrollments', 'program_id', 'employee_id');
    }

    // العلاقة مع سجل التقدم
    public function progressHistory()
    {
        return $this->hasMany(TrainingProgress::class, 'program_id');
    }

    // حساب نسبة الاكتمال
    public function getCompletionPercentageAttribute()
    {
        if ($this->capacity > 0) {
            $enrolledCount = $this->enrollments()->count();
            return ($enrolledCount / $this->capacity) * 100;
        }
        return 0;
    }

    // الحصول على اسم الفئة
    public function getCategoryNameAttribute()
    {
        $categories = [
            'technical' => 'تقني',
            'management' => 'إداري',
            'soft_skills' => 'مهارات لينة',
            'safety' => 'سلامة',
            'compliance' => 'امتثال'
        ];

        return $categories[$this->category] ?? $this->category;
    }

    // الحصول على حالة العرض
    public function getStatusLabelAttribute()
    {
        $statuses = [
            'upcoming' => 'قادم',
            'active' => 'جاري',
            'completed' => 'مكتمل',
            'cancelled' => 'ملغى'
        ];

        return $statuses[$this->status] ?? $this->status;
    }

    // نطاق للبرامج النشطة
    public function scopeActive($query)
    {
        return $query->whereIn('status', ['upcoming', 'active']);
    }

    // نطاق للبرامج المكتملة
    public function scopeCompleted($query)
    {
        return $query->where('status', 'completed');
    }

    // نطاق للبرامج حسب الفئة
    public function scopeByCategory($query, $category)
    {
        return $query->where('category', $category);
    }
}