<?php

namespace App\Models;

// استخدام Trait TenantScoped (إذا كنت ترغب في عزل البيانات داخل هذا النموذج نفسه، وهو نادر)
// use App\Models\Traits\TenantScoped;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

/**
 * نموذج Tenant.
 *
 * يمثل شركة أو "مستأجر" في النظام متعدد المستأجرين (Multi-Tenant SaaS).
 * كل مستأجر (Tenant) يحتوي على مجموعة خاصة به من الموظفين، المستخدمين، القروض، الإجازات... إلخ.
 */
class Tenant extends Model
{
    use HasFactory;

    /**
     * الحقول القابلة للتعبئة.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'name', // اسم الشركة
        'subdomain', // subdomain الفريد للوصول إلى معلومات الشركة (مثلاً: shark_company.myapp.com)
        'status', // الحالة: active, inactive, suspended...
        'package_id', // معرف الحزمة (package) التي تستخدمها الشركة (إذا كان النظام يدعم الحزم)
        'subscription_ends_at', // تاريخ انتهاء الاشتراك
    ];

    /**
     * التحويلات التلقائية للحقول.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'subscription_ends_at' => 'datetime', // تحويل الحقل إلى كائن Carbon
    ];

    // إذا كنت ترغب في تطبيق TenantScoped على هذا النموذج نفسه (نادر الاستخدام، لأنه غالبًا يُستخدم على النماذج مثل Employee, User...)
    // قم بإزالة التعليق من السطرين التاليين بعد التأكد من تعريف Trait.
    // use TenantScoped;

    // ====================== العلاقات ======================

    /**
     * العلاقة مع الحزمة (Package).
     * كل مستأجر مرتبط بحزمة واحدة فقط.
     */
    public function package()
    {
        return $this->belongsTo(Package::class);
    }

    /**
     * العلاقة مع الاشتراكات (TenantSubscription).
     * يمكن أن يكون للمستأجر عدة اشتراكات (مثلاً اشتراك قديم وحالي).
     */
    public function subscriptions()
    {
        return $this->hasMany(TenantSubscription::class);
    }

    /**
     * العلاقة مع المستخدمين.
     * يفترض أن جدول users يحتوي على حقل tenant_id.
     */
    public function users()
    {
        return $this->hasMany(User::class);
    }

    /**
     * العلاقة مع الموظفين.
     * يفترض أن جدول employees يحتوي على حقل tenant_id.
     */
    public function employees()
    {
        return $this->hasMany(Employee::class);
    }

    /**
     * العلاقة مع الإجازات.
     * يفترض أن جدول leaves يحتوي على حقل tenant_id (أو يُطبق عبر العلاقة مع Employee).
     */
    public function leaves()
    {
        return $this->hasMany(Leave::class);
    }

    /**
     * العلاقة مع القروض.
     * يفترض أن جدول loans يحتوي على حقل tenant_id (أو يُطبق عبر العلاقة مع Employee).
     */
    public function loans()
    {
        return $this->hasMany(Loan::class);
    }

    /**
     * العلاقة مع الاستقطاعات.
     * يفترض أن جدول deductions يحتوي على حقل tenant_id (أو يُطبق عبر العلاقة مع Employee).
     */
    public function deductions()
    {
        return $this->hasMany(Deduction::class);
    }

    /**
     * العلاقة مع سجلات الحضور.
     * يفترض أن جدول attendances يحتوي على حقل tenant_id (أو يُطبق عبر العلاقة مع Employee).
     */
    public function attendances()
    {
        return $this->hasMany(Attendance::class);
    }

    /**
     * العلاقة مع سجلات الرواتب.
     * يفترض أن جدول payrolls يحتوي على حقل tenant_id (أو يُطبق عبر العلاقة مع Employee).
     */
    public function payrolls()
    {
        return $this->hasMany(Payroll::class);
    }

    /**
     * العلاقة مع البرامج التدريبية.
     * يفترض أن جدول training_programs يحتوي على حقل tenant_id.
     */
    public function trainingPrograms()
    {
        return $this->hasMany(TrainingProgram::class);
    }

    /**
     * العلاقة مع الإشعارات.
     * يفترض أن جدول notifications يحتوي على حقل tenant_id.
     */
    public function notifications()
    {
        return $this->hasMany(Notification::class);
    }

    /**
     * العلاقة مع سجلات الأنشطة.
     * يفترض أن جدول activities يحتوي على حقل tenant_id.
     */
    public function activities()
    {
        return $this->hasMany(Activity::class);
    }

    /**
     * العلاقة مع الأصول.
     * يفترض أن جدول assets يحتوي على حقل tenant_id.
     */
    public function assets()
    {
        return $this->hasMany(Asset::class);
    }

    /**
     * العلاقة مع الأقسام.
     * يفترض أن جدول departments يحتوي على حقل tenant_id.
     */
    public function departments()
    {
        return $this->hasMany(Department::class);
    }

    /**
     * العلاقة مع المسميات الوظيفية.
     * يفترض أن جدول positions يحتوي على حقل tenant_id.
     */
    public function positions()
    {
        return $this->hasMany(Position::class);
    }

    /**
     * العلاقة مع الإعدادات (Tenant Settings).
     * إذا كنت تخطط لجدول منفصل للإعدادات حسب المستأجر.
     */
    public function settings()
    {
        return $this->hasMany(TenantSetting::class);
    }

    // ====================== وظائف مساعدة ======================

    /**
     * التحقق مما إذا كان المستأجر نشطًا.
     *
     * @return bool
     */
    public function isActive(): bool
    {
        return $this->status === 'active';
    }

    /**
     * التحقق مما إذا كان المستأجر معطلًا.
     *
     * @return bool
     */
    public function isInactive(): bool
    {
        return $this->status === 'inactive';
    }

    /**
     * التحقق مما إذا كان المستأجر معلقًا.
     *
     * @return bool
     */
    public function isSuspended(): bool
    {
        return $this->status === 'suspended';
    }

    /**
     * التحقق مما إذا كان الاشتراك لا يزال ساريًا.
     *
     * @return bool
     */
    public function isSubscriptionActive(): bool
    {
        return $this->isActive() && (!$this->subscription_ends_at || $this->subscription_ends_at->isFuture());
    }

    /**
     * التحقق مما إذا كان الاشتراك منتهيًا.
     *
     * @return bool
     */
    public function isSubscriptionExpired(): bool
    {
        return $this->subscription_ends_at && $this->subscription_ends_at->isPast();
    }
}