<?php
// app/Models/Setting.php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Cache;

class Setting extends Model
{
    use HasFactory;

    protected $fillable = ['key', 'value', 'description'];

    /**
     * دالة مساعدة لجلب قيمة الإعداد من قاعدة البيانات مع دعم الذاكرة المؤقتة.
     *
     * @param string $key      مفتاح الإعداد (مثل: 'default_currency')
     * @param mixed  $default  القيمة الافتراضية إذا لم تُوجد
     * @return mixed
     */
    public static function get(string $key, $default = null)
    {
        return Cache::rememberForever('setting_' . $key, function () use ($key, $default) {
            $setting = self::where('key', $key)->first();
            return $setting ? $setting->value : $default;
        });
    }

    /**
     * دالة مساعدة لتحديث أو إنشاء قيمة الإعداد، مع مسح الذاكرة المؤقتة فورًا.
     *
     * @param string $key   مفتاح الإعداد
     * @param mixed  $value القيمة الجديدة
     * @return \App\Models\Setting
     */
    public static function set(string $key, $value): Setting
    {
        $setting = self::updateOrCreate(
            ['key' => $key],
            ['value' => $value]
        );

        // ✅ مسح الذاكرة المؤقتة مباشرة بعد التحديث
        Cache::forget('setting_' . $key);

        return $setting;
    }

    /**
     * (اختياري) دالة لمسح كل إعدادات الذاكرة المؤقتة دفعة واحدة.
     * مفيدة عند الصيانة أو إعادة التهيئة.
     */
    public static function clearAllCache(): void
    {
        // لا يمكن مسح مفاتيح نمطية دفعة واحدة في Laravel Cache بدون prefix مخصص،
        // لكن يمكن مسح كل الإعدادات يدويًا أو استخدام Tagged Cache.
        // طريقة بسيطة: جلب كل المفاتيح ومسحها
        $allKeys = self::pluck('key');
        foreach ($allKeys as $key) {
            Cache::forget('setting_' . $key);
        }
    }
}