<?php

namespace App\Models;

use App\Scopes\CompanyScope;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;

/**
 * نموذج الراتب
 *
 * يُستخدم لتخزين وحساب تفاصيل راتب موظف لشهر معين.
 * يتضمن: الراتب الأساسي، الحوافز، المكونات، الخصومات، والراتب الصافي.
 *
 * @package App\Models
 */
class Payroll extends Model
{
    use HasFactory;

    protected static function boot()
    {
        parent::boot();
        // تطبيق CompanyScope
        static::addGlobalScope(new CompanyScope());
    }

    /**
     * الحقول القابلة للتعبئة (Mass Assignment).
     *
     * @var array<int, string>
     */
    protected $fillable = [
        // معلومات أساسية
        'employee_id',
        'company_id',
        'month',
        'year',
        'status', // pending, paid, cancelled
        'paid_at',
        'notes',

        // مكونات الراتب
        'basic_salary',
        'incentives', // الحوافز (تم استبدال 'allowances')
        'bonuses',    // المكافآت
        'deductions', // خصومات إضافية (يدوية)

        // الخصومات المحسوبة
        'active_deductions',     // إجمالي الاستقطاعات النشطة (من جدول deductions)
        'loan_installments',     // إجمالي أقساط القروض (من جدول loans)
        'leave_deduction',       // مبلغ خصم الإجازات (من حسابات الموظف)

        // مكونات الراتب الإضافية (تُحسب من جدول payroll_salary_components)
        // 'payroll_salary_components_value' (تم إضافتها في Accessor)

        // الحقول المحسوبة (يتم حسابها وتحديثها باستخدام الدوال)
        'total_salary', // إجمالي الراتب (قبل الخصومات)
        'net_salary',   // الراتب الصافي (بعد الخصومات)
        
        // تفاصيل الإجازات (مُخزنة مع الراتب)
        'leave_balance_before_deduction', // الرصيد قبل الخصم
        'leave_balance_after_deduction',  // الرصيد بعد الخصم
        'leave_deduction_amount',         // المبلغ المالي المخصوم (من الأيام الزائدة)
        'leave_granted_this_month',       // الأيام الممنوحة هذا الشهر
        'used_leave_days',                // الأيام المستخدمة هذا الشهر
        'excess_leave_days',              // الأيام الزائدة (Used - Granted)
        'meta',                           // JSON meta field to store arbitrary metadata (e.g. selected installment ids)
    ];

    /**
     * تحويلات أنواع الحقول عند استرجاعها من قاعدة البيانات.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'month' => 'integer',
        'year' => 'integer',
        'status' => 'string',
        'paid_at' => 'datetime',
        'basic_salary' => 'decimal:2',
        'incentives' => 'decimal:2',
        'bonuses' => 'decimal:2',
        'deductions' => 'decimal:2',
        'active_deductions' => 'decimal:2',
        'loan_installments' => 'decimal:2',
        'leave_deduction' => 'decimal:2',
        'total_salary' => 'decimal:2',
        'net_salary' => 'decimal:2',
        'meta' => 'array',
        // تفاصيل الإجازات
        'leave_balance_before_deduction' => 'decimal:2',
        'leave_balance_after_deduction' => 'decimal:2',
        'leave_deduction_amount' => 'decimal:2',
        'leave_granted_this_month' => 'decimal:2',
        'used_leave_days' => 'decimal:2',
        'excess_leave_days' => 'decimal:2',
    ];

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    /**
     * العلاقة مع نموذج الموظف.
     * كل راتب يخص موظفًا واحدًا.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function employee()
    {
        return $this->belongsTo(Employee::class);
    }

    /**
     * العلاقة مع مكونات الراتب الإضافية.
     * كل راتب يمكن أن يحتوي على العديد من المكونات الإضافية.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function payrollSalaryComponents()
    {
        return $this->hasMany(PayrollSalaryComponent::class);
    }

    /*
    |--------------------------------------------------------------------------
    | Dynamic Accessors (Calculated on-the-fly)
    |--------------------------------------------------------------------------
    */

    /**
     * ✅ (مُعدّل لحل مشكلة مكونات الراتب الإضافية)
     * إجمالي الراتب الخام (الأساسي + الحوافز + المكافآت + مكونات الراتب الإضافية).
     *⚠️ يعتمد الآن على حسابات الموظف (getTotalSalaryWithComponentsAttribute) لجلب المكونات من جدول employees.
     *
     * @return float
     */
    public function getGrossSalaryAttribute(): float
    {
        // ✅ استخدام دالة من نموذج الموظف لحساب الإجمالي (الأساسي + الحوافز + المكونات 1-7)
        if ($this->relationLoaded('employee') && $this->employee) {
            return $this->employee->getTotalSalaryWithComponentsAttribute();
        }

        // ✅ إذا لم يتم تحميل العلاقة، استخدم الحقول المحفوظة في هذا السجل ( backup)
        $base = $this->basic_salary ?? 0;
        $incentives = $this->incentives ?? 0;
        $bonuses = $this->bonuses ?? 0;

        // ✅ محاولة حساب مكونات الراتب الإضافية.
        // إذا كانت العلاقة محملة استخدمها، وإلا استعلم عن المجموع مباشرة من قاعدة البيانات
        // هذا يضمن احتساب المكونات حتى لو لم يتم eager-load للعلاقة بعد الإنشاء.
        if ($this->relationLoaded('payrollSalaryComponents')) {
            $componentsTotal = $this->payrollSalaryComponents->sum('value');
        } else {
            $componentsTotal = $this->payrollSalaryComponents()->sum('value') ?? 0;
        }

        // ✅ محاولة جلب المكونات من الحقول المحفوظة في هذا السجل (مثلاً إذا تم حسابها مسبقًا وحفظها)
        // هذا مفيد إذا لم تكن العلاقة متاحة أو إذا كنت تحفظ القيم المحسوبة مسبقًا في هذا الجدول.
        // مثال: إذا كانت لديك حقول component_1_value ... component_7_value في جدول payrolls
        // $componentsTotal += $this->component_1_value ?? 0;
        // $componentsTotal += $this->component_2_value ?? 0;
        // ... etc

        return $base + $incentives + $bonuses + $componentsTotal;
    }

    /*
    |--------------------------------------------------------------------------
    | Calculation Helper Functions (Used by Controller before saving)
    |--------------------------------------------------------------------------
    */

    /**
     * ✅ (مُعدّل لحل مشكلة مكونات الراتب الإضافية)
     * حساب وإعداد قيمة 'total_salary'.
     * يجب استدعاؤها قبل حفظ النموذج.
     *
     * @return void
     */
    public function calculateTotalSalary(): void
    {
        // ✅ استخدام accessor المُعدّل لحساب الإجمالي
        $this->total_salary = $this->gross_salary;
    }

    /**
     * ✅ (مُعدّل لحل مشكلة الاستقطاعات)
     * حساب وإعداد قيمة 'net_salary'.
     * يجب استدعاؤها بعد 'calculateTotalSalary'.
     *
     * @return void
     */
    public function calculateNetSalary(): void
    {
        $total = $this->total_salary ?? $this->gross_salary;
        
        // ✅ التأكد من أن active_deductions يُحسب بشكل شهري من خلال وظيفة التحكم
        // ✅ استخدام دالة من نموذج الموظف للحصول على الاستقطاعات لهذا الشهر فقط
        // * ملاحظة: هذه القيمة يجب أن تُحسب وتُمر من وظيفة التحكم (Controller) عند إنشاء/تحديث الراتب
        // $this->active_deductions = $this->employee->getTotalActiveDeductionsForMonth($this->year, $this->month);
        
        $deductionsSum = ($this->active_deductions ?? 0) +
                         ($this->loan_installments ?? 0) +
                         ($this->leave_deduction ?? 0) +
                         ($this->deductions ?? 0); // الخصومات اليدوية

        $this->net_salary = max(0, $total - $deductionsSum);
    }

    /*
    |--------------------------------------------------------------------------
    | Support Functions for Display (e.g., in Employee Show View or Payroll List)
    |--------------------------------------------------------------------------
    */

        /**
     * ✅ (مُعدّل لحل مشكلة مكونات الراتب الإضافية)
     * إرجاع تفاصيل مكونات الراتب (الأساسي، الحوافز، المكافآت، المكونات الإضافية).
     *
     * @return \Illuminate\Support\Collection
     */
    public function getSalaryBreakdownAttribute()
    {
        $breakdown = collect([
            ['name' => __('app.basic_salary'), 'value' => $this->basic_salary, 'type' => 'base'],
            ['name' => __('app.incentives'), 'value' => $this->incentives, 'type' => 'incentive'],
            ['name' => __('app.bonuses'), 'value' => $this->bonuses, 'type' => 'bonus'],
        ])->filter(fn($item) => $item['value'] > 0);

        // إضافة مكونات الراتب الإضافية من حسابات الموظف (component_1 ... component_7)
        if ($this->relationLoaded('employee') && $this->employee) {
            $employee = $this->employee;
            $names = ['component_name_1', 'component_name_2', 'component_name_3', 'component_name_4', 'component_name_5', 'component_name_6', 'component_name_7'];
            $values = ['component_1', 'component_2', 'component_3', 'component_4', 'component_5', 'component_6', 'component_7'];

            for ($i = 0; $i < 7; $i++) {
                $name = $employee->{$names[$i]} ?? "المكون " . ($i+1);
                $value = $employee->{$values[$i]} ?? 0;
                if ($value > 0) {
                    $breakdown->push([
                        'name' => $name,
                        'value' => $value,
                        'type' => 'component'
                    ]);
                }
            }
        } else {
            // fallback: إذا لم يتم تحميل العلاقة، استعلم عن مكونات الراتب من قاعدة البيانات
            $components = $this->relationLoaded('payrollSalaryComponents') ? $this->payrollSalaryComponents : $this->payrollSalaryComponents()->get();
            foreach ($components as $comp) {
                $breakdown->push([
                    'name' => $comp->name,
                    'value' => $comp->value,
                    'type' => 'component'
                ]);
            }
        }

        return $breakdown;
    }

    /**
     * ✅ (مُعدّل لحل مشكلة الاستقطاعات)
     * إرجاع تفاصيل الخصومات (الاستقطاعات، القروض، الإجازات، اليدوية).
     * ⚠️ تأكد من أن active_deductions يحتوي فقط على الاستقطاعات *التي تؤثر* على هذا الشهر.
     *
     * @return \Illuminate\Support\Collection
     */
    public function getDeductionsBreakdownAttribute()
    {
        return collect([
            ['name' => 'الاستقطاعات النشطة (هذا الشهر)', 'value' => $this->active_deductions],
            ['name' => 'أقساط القروض (هذا الشهر)', 'value' => $this->loan_installments],
            ['name' => 'خصم الإجازات الزائدة (هذا الشهر)', 'value' => $this->leave_deduction],
            ['name' => 'خصومات إضافية (يدوية)', 'value' => $this->deductions],
        ])->filter(fn($item) => $item['value'] > 0);
    }























    /**
     * إرجاع تفاصيل تتبع الإجازات لهذا الراتب.
     *
     * @return array
     */
    public function getLeaveTrackingDetailsAttribute(): array
    {
        return [
            'used_days_in_month' => $this->used_leave_days ?? 0,
            'granted_days_in_month' => $this->leave_granted_this_month ?? 0,
            'excess_days_deducted' => $this->excess_leave_days ?? 0,
            'deduction_amount' => $this->leave_deduction_amount ?? 0,
            'balance_before_deduction' => $this->leave_balance_before_deduction ?? 0,
            'balance_after_deduction' => $this->leave_balance_after_deduction ?? 0,
        ];
    }

    /*
    |--------------------------------------------------------------------------
    | Utility Functions
    |--------------------------------------------------------------------------
    */

    /**
     * التحقق مما إذا كان الراتب تم دفعه.
     *
     * @return bool
     */
    public function isPaid(): bool
    {
        return $this->status === 'paid';
    }

    /**
     * التحقق مما إذا كان الراتب معلقًا.
     *
     * @return bool
     */
    public function isPending(): bool
    {
        return $this->status === 'pending';
    }

    /**
     * التحقق مما إذا كان الراتب ملغى.
     *
     * @return bool
     */
    public function isCancelled(): bool
    {
        return $this->status === 'cancelled';
    }

    /**
     * تعيين حالة الراتب إلى "مدفوع".
     *
     * @return self
     */
    public function markAsPaid(): self
    {
        $this->update([
            'status' => 'paid',
            'paid_at' => now(),
        ]);
        return $this;
    }

    /**
     * تعيين حالة الراتب إلى "ملغى".
     *
     * @return self
     */
    public function markAsCancelled(): self
    {
        $this->update([
            'status' => 'cancelled',
            'paid_at' => null,
        ]);
        return $this;
    }

    /**
     * ✅ (مُعدّل لحل مشكلة الاستقطاعات)
     * تنسيق الشهر والسنة.
     * ⚠️ تأكد من أن هذا يُستخدم لعرض الشهر الذي *يخص* هذا السجل، وليس لحساب الاستقطاعات.
     *
     * @return string
     */
    public function getFormattedMonthYearAttribute(): string
    {
        try {
            return Carbon::createFromDate($this->year, $this->month, 1)->translatedFormat('F Y');
        } catch (\Throwable $th) {
            return "غير محدد ({$this->month}/{$this->year})";
        }
    }
}