<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Notifications\DatabaseNotificationCollection;

class Notification extends Model
{
    use HasFactory;

    protected $fillable = [
        'type',
        'notifiable_type',
        'notifiable_id',
        'data',
        'read_at',
    ];

    protected $casts = [
        'data' => 'array',
        'read_at' => 'datetime',
    ];

    protected $primaryKey = 'id';
    public $incrementing = false;
    protected $keyType = 'string';

    /**
     * العلاقة مع النموذج القابل للإشعار
     */
    public function notifiable()
    {
        return $this->morphTo();
    }

    /**
     * تحديد ما إذا كان الإشعار قد قُرأ
     */
    public function isRead()
    {
        return !is_null($this->read_at);
    }

    /**
     * تحديد ما إذا كان الإشعار غير مقروء
     */
    public function isUnread()
    {
        return is_null($this->read_at);
    }

    /**
     * وضع علامة على الإشعار كمقروء
     */
    public function markAsRead()
    {
        if (is_null($this->read_at)) {
            $this->forceFill(['read_at' => $this->freshTimestamp()])->save();
        }
    }

    /**
     * وضع علامة على الإشعار كغير مقروء
     */
    public function markAsUnread()
    {
        if (!is_null($this->read_at)) {
            $this->forceFill(['read_at' => null])->save();
        }
    }

    /**
     * الحصول على محتوى الإشعار
     */
    public function getDataAttribute($value)
    {
        return json_decode($value, true);
    }

    /**
     * تعيين محتوى الإشعار
     */
    public function setDataAttribute($value)
    {
        $this->attributes['data'] = json_encode($value);
    }

    /**
     * الحصول على العنوان
     */
    public function getTitleAttribute()
    {
        return $this->data['title'] ?? 'إشعار جديد';
    }

    /**
     * الحصول على الرسالة
     */
    public function getMessageAttribute()
    {
        return $this->data['message'] ?? '';
    }

    /**
     * الحصول على الرابط
     */
    public function getUrlAttribute()
    {
        return $this->data['url'] ?? '#';
    }

    /**
     * الحصول على النوع
     */
    public function getTypeAttribute()
    {
        return $this->data['type'] ?? 'info';
    }

    /**
     * تحديد ما إذا كان الإشعار مهم
     */
    public function isImportant()
    {
        return $this->data['important'] ?? false;
    }

    /**
     * تحديد ما إذا كان الإشعار يتطلب إجراء
     */
    public function requiresAction()
    {
        return $this->data['requires_action'] ?? false;
    }

    /**
     * الحصول على الإجراءات المتاحة
     */
    public function getActionsAttribute()
    {
        return $this->data['actions'] ?? [];
    }

    /**
     * نطاق للإشعارات المقروءة
     */
    public function scopeRead($query)
    {
        return $query->whereNotNull('read_at');
    }

    /**
     * نطاق للإشعارات غير المقروءة
     */
    public function scopeUnread($query)
    {
        return $query->whereNull('read_at');
    }

    /**
     * نطاق للإشعارات المهمة
     */
    public function scopeImportant($query)
    {
        return $query->where('data->important', true);
    }

    /**
     * نطاق للإشعارات التي تتطلب إجراء
     */
    public function scopeRequiresAction($query)
    {
        return $query->where('data->requires_action', true);
    }

    /**
     * نطاق للبحث عن إشعارات حسب النوع
     */
    public function scopeOfType($query, $type)
    {
        return $query->where('type', $type);
    }

    /**
     * نطاق للإشعارات الحديثة
     */
    public function scopeRecent($query, $days = 7)
    {
        return $query->where('created_at', '>=', now()->subDays($days));
    }

    /**
     * الحصول على عدد الأيام منذ الإنشاء
     */
    public function getAgeInDaysAttribute()
    {
        return $this->created_at->diffInDays(now());
    }

    /**
     * تحديد ما إذا كان الإشعار قديم
     */
    public function isOld()
    {
        return $this->age_in_days > 30;
    }

    /**
     * تحويل الإشعار إلى مصفوفة
     */
    public function toArray()
    {
        return [
            'id' => $this->id,
            'type' => $this->type,
            'title' => $this->title,
            'message' => $this->message,
            'url' => $this->url,
            'read' => $this->isRead(),
            'important' => $this->isImportant(),
            'requires_action' => $this->requiresAction(),
            'created_at' => $this->created_at->toISOString(),
            'created_at_human' => $this->created_at->diffForHumans(),
        ];
    }
}