<?php

namespace App\Models;

use App\Scopes\CompanyScope;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Loan extends Model
{
    use HasFactory;

    protected static function boot()
    {
        parent::boot();
        static::addGlobalScope(new CompanyScope());
    }

    protected $fillable = [
        'employee_id', 'company_id', 'loan_number', 'title', 'description',
        'loan_type', 'amount', 'interest_rate', 'interest_amount', 'total_amount',
        'installment_count',
        'installment_amount', 'start_date', 'end_date', 'status',
        'approved_by', 'approved_at', 'approval_notes',
        'reason', 'notes',
        'paid_amount', 'remaining_amount',
        'rejection_reason', 'rejected_by', 'rejected_at',
        'activated_by', 'activated_at',
        'cancelled_by', 'cancelled_at'
    ];

    protected $casts = [
        'start_date' => 'date',
        'end_date' => 'date',
        'approved_at' => 'date',
        'rejected_at' => 'date',
        'activated_at' => 'date',
        'cancelled_at' => 'date',
        'amount' => 'decimal:2',
        'interest_rate' => 'decimal:2',
        'interest_amount' => 'decimal:2',
        'total_amount' => 'decimal:2',
        'installment_amount' => 'decimal:2',
        'paid_amount' => 'decimal:2',
        'remaining_amount' => 'decimal:2',
        'installment_count' => 'integer', // ✅ Cast لحقل installment_count (اختياري، لكن يساعد على التعامل معه بشكل صحيح)
        'approved_by' => 'integer',
        'rejected_by' => 'integer',
        'activated_by' => 'integer',
        'cancelled_by' => 'integer',
    ];

    public function employee()
    {
        return $this->belongsTo(Employee::class);
    }

    public function approver()
    {
        return $this->belongsTo(User::class, 'approved_by');
    }

    public function installments()
    {
        return $this->hasMany(LoanInstallment::class)->orderBy('installment_number'); // ✅ ترتيب الأقساط حسب الرقم
    }

    public function paidInstallments()
    {
        return $this->installments()->where('status', 'paid');
    }

    public function pendingInstallments()
    {
        return $this->installments()->where('status', 'pending');
    }

    public function overdueInstallments()
    {
        return $this->installments()->where('status', 'overdue');
    }

    public function getTotalPaidAttribute()
    {
        // ✅ استخدام sum('amount') للحصول على المبلغ الإجمالي المدفوع من الأقساط المدفوعة
        return $this->installments()->where('status', 'paid')->sum('amount');
    }

    public function getRemainingAmountAttribute()
    {
        // ✅ استخدام paid_amount بدلاً من total_paid (الذي يعتمد على العلاقة)
        return $this->total_amount - $this->paid_amount;
    }

    public function getProgressPercentageAttribute()
    {
        // ✅ استخدام paid_amount و total_amount
        if ($this->total_amount > 0) {
            return round(($this->paid_amount / $this->total_amount) * 100, 2);
        }
        return 0;
    }

    public function updateIfCompleted()
    {
        // ⚠️ استخدام الاسم الجديد 'installment_count' للتحقق من عدد الأقساط
        if ($this->paidInstallments()->count() >= $this->installment_count) {
            $this->update(['status' => 'paid']); // ✅ تغيير الحالة إلى 'paid' عند اكتمال الدفع
        }
    }
}