<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class LeaveBalance extends Model
{
    protected $fillable = [
        'employee_id',
        // 'annual_granted' الآن يمثل مجموع الأيام الممنوحة حتى تاريخ معين (ليس بالضرورة 12 شهرًا)
        // على سبيل المثال، إذا كان الشهري 2، وبعد 6 أشهر يكون 12.
        'annual_granted',
        // 'used_annual' الآن يمثل مجموع الأيام المستخدمة حتى تاريخ معين من الإجازات المدفوعة (annual/emergency).
        'used_annual',
        // 'carry_forward' الآن يُستخدم لتخزين الأيام الزائدة غير المستخدمة من الأشهر السابقة التي تم تجميعها.
        // هذا يمكن استخدامه لاحقاً إذا احتجنا لتتبع الأيام المتراكمة بشكل منفصل.
        // في المنطق الجديد، التركيز سيكون على حساب الرصيد شهريًا، ولكن هذا الحقل لا يزال مفيدًا.
        'carry_forward',
    ];

    protected $casts = [
        'annual_granted' => 'integer',
        'used_annual' => 'integer',
        'carry_forward' => 'integer',
    ];

    public function employee(): BelongsTo
    {
        return $this->belongsTo(Employee::class);
    }

    /**
     * حساب الرصيد المتبقي المتاح.
     * هذا الرصيد يعتمد على المنطق السابق (الممنوح - المستخدم).
     * في المنطق الجديد، قد لا يُستخدم هذا الحساب بشكل مباشر في الخصومات الشهرية،
     * لكنه لا يزال يوفر نظرة عامة على "الحق النظري" مقابل "الاستخدام الفعلي".
     *
     * @return int
     */
    public function getAvailableBalanceAttribute(): int
    {
        // الحماية من القيم السالبة
        return max(0, $this->annual_granted - $this->used_annual);
    }

     /**
     * [اختياري] تحديث الرصيد الشهري المتراكم.
     * هذه الدالة مفيدة لتحديث قيم `annual_granted` و `used_annual` بشكل دوري (مثلاً شهريًا).
     * لكن الحساب الفعلي للخصم سيكون في PayrollController بناءً على الشهر المُعالج.
     *
     * @param int $newGranted إجمالي الأيام الممنوحة حتى الآن
     * @param int $newUsed إجمالي الأيام المستخدمة حتى الآن
     * @return void
     */
    public function updateCumulativeBalance(int $newGranted, int $newUsed): void
    {
        $this->update([
            'annual_granted' => $newGranted,
            'used_annual' => $newUsed,
            // carry_forward يمكن تحديثه أيضًا إذا لزم
        ]);
    }
}