<?php

namespace App\Models;

use App\Scopes\CompanyScope;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;

class Leave extends Model
{
    use HasFactory;

    protected $fillable = [
        'employee_id',
        'company_id',
        'leave_type',
        'leave_type_id',
        'start_date',
        'end_date',
        'reason',
        'status',
        'approved_by',
        'days_count',
        'contact_details',
        'rejection_reason',
        'rejected_by',
        'rejected_at',
        'cancelled_by',
        'cancelled_at',
        // ✅ إضافة الحقول المفقودة للتعامل مع الخصم الشهري
        'deduction_date', // التاريخ الذي تم فيه خصم الإجازة (مثلاً، نهاية الشهر)
        'is_deducted',     // علامة تشير إلى ما إذا تم خصم هذه الإجازة من الراتب أم لا
        'applied_balance_change_id',
        'deducted_days',
        'unpaid_deducted_days',
        'paid_days',
        'unpaid_days',
        'is_unpaid_deduction_applied',
        'balance_snapshot',
        // ✅ أيام مستثناة من الخصم (عطل أسبوعية / أيام رسمية في منتصف الإجازة)
        'excluded_days',
        'excluded_days_count',
        'deductible_days',
        // indicate approval was modified after initial approval
        'approval_modified',
        'approval_modified_at',
        'approval_modified_by',
    ];

    protected $casts = [
        'start_date' => 'date',
        'end_date' => 'date',
        'days_count' => 'integer',
        'approved_by' => 'integer',
        'rejected_by' => 'integer',
        'cancelled_by' => 'integer',
        'rejected_at' => 'datetime',
        'cancelled_at' => 'datetime',
        'deduction_date' => 'date', // ✅ تحويل الحقل إلى تاريخ
        'is_deducted' => 'boolean',  // ✅ تحويل الحقل إلى قيمة منطقية
        'is_unpaid_deduction_applied' => 'boolean',
        'balance_snapshot' => 'array',
        // ✅ أيام مستثناة من الخصم
        'excluded_days' => 'array',
        'excluded_days_count' => 'integer',
        'deductible_days' => 'integer',
        'unpaid_deducted_days' => 'integer',
        'paid_days' => 'integer',
        'unpaid_days' => 'integer',
        // approval modified markers
        'approval_modified' => 'boolean',
        'approval_modified_at' => 'datetime',
        'approval_modified_by' => 'integer',
    ];

    public function employee()
    {
        return $this->belongsTo(Employee::class);
    }

    public function leaveTypeModel()
    {
        return $this->belongsTo(LeaveType::class, 'leave_type_id');
    }

    public function appliedBalanceChange()
    {
        return $this->belongsTo(LeaveBalanceChange::class, 'applied_balance_change_id');
    }

    /**
     * Scope: filter leaves by leave type code (single)
     */
    public function scopeOfCode($query, $code)
    {
        return $query->whereHas('leaveTypeModel', function ($q) use ($code) {
            $q->where('code', $code);
        });
    }

    /**
     * Scope: filter leaves by multiple leave type codes
     */
    public function scopeOfCodes($query, array $codes)
    {
        return $query->whereHas('leaveTypeModel', function ($q) use ($codes) {
            $q->whereIn('code', $codes);
        });
    }

    public function approver()
    {
        // ✅ التصحيح: العلاقة يجب أن تكون مع نموذج User، ليس Employee
        // لأن approved_by يشير إلى معرف المستخدم في جدول users
        return $this->belongsTo(\App\Models\User::class, 'approved_by');
    }

     /**
     * العلاقة مع المستخدم الذي رفض الطلب.
     */
    public function rejecter()
    {
        // ✅ إضافة العلاقة مع المستخدم الذي قام بالرفض
        return $this->belongsTo(\App\Models\User::class, 'rejected_by');
    }

    /**
     * العلاقة مع المستخدم الذي ألغى الطلب.
     */
    public function canceller()
    {
        // ✅ إضافة العلاقة مع المستخدم الذي قام بالإلغاء
        return $this->belongsTo(\App\Models\User::class, 'cancelled_by');
    }

    /**
     * حساب عدد الأيام عند حفظ الإجازة أو تحديثها.
     * ✅ التحسين: التأكد من أن عدد الأيام يُحسب فقط إذا تغيرت التواريخ
     * أو إذا كانت القيمة الحالية فارغة/صفر.
     * ✅ حساب أيام مستثناة (عطل أسبوعية / أيام رسمية)
     */
    protected static function boot()
    {
        parent::boot();
        
        // تطبيق CompanyScope
        static::addGlobalScope(new CompanyScope());

        // ✅ استخدام creating و updating بدلاً من saving لتحسين الأداء قليلاً
        // أو الاحتفاظ بـ saving إذا كنت بحاجة للتحقق في كل مرة
        static::saving(function ($leave) {
            // التحقق من وجود التواريخ
            if ($leave->start_date && $leave->end_date) {
                $startDate = Carbon::instance($leave->start_date)->startOfDay();
                $endDate = Carbon::instance($leave->end_date)->startOfDay();

                // التأكد من أن تاريخ البدء ليس بعد تاريخ الانتهاء
                if ($startDate->lte($endDate)) {
                    // حساب عدد الأيام (+1 ليشمل اليوم الأول والأخير)
                    $calculatedDays = $startDate->diffInDays($endDate) + 1;
                    
                    // فقط حدّث days_count إذا كان الحساب مختلفًا أو إذا كانت القيمة الحالية غير صحيحة
                    // هذا يمنع الكتابة فوق قيمة days_count يدويًا في حالات خاصة
                    if ($leave->isDirty(['start_date', 'end_date']) || is_null($leave->days_count) || $leave->days_count <= 0) {
                         $leave->days_count = $calculatedDays;
                    }
                    
                    // ✅ حساب أيام مستثناة (عطل أسبوعية / أيام رسمية)
                    $leave->calculateExcludedDays();
                   
                } else {
                    // إذا كان تاريخ البدء بعد تاريخ الانتهاء، اجعل عدد الأيام صفر
                    $leave->days_count = 0;
                    $leave->excluded_days = [];
                    $leave->excluded_days_count = 0;
                    $leave->deductible_days = 0;
                }
            }
        });
    }

    /**
     * ✅ حساب الأيام المستثناة من الخصم (أيام عطل أسبوعية / أيام رسمية في منتصف الإجازة)
     */
    public function calculateExcludedDays()
    {
        if (!$this->start_date || !$this->end_date) {
            $this->excluded_days = [];
            $this->excluded_days_count = 0;
            $this->deductible_days = $this->days_count ?? 0;
            return;
        }

        // Ensure we have the employee object (may not be loaded during saving)
        $employee = $this->employee ?? \App\Models\Employee::find($this->employee_id);
        if (!$employee) {
            $this->excluded_days = [];
            $this->excluded_days_count = 0;
            $this->deductible_days = $this->days_count ?? 0;
            return;
        }

        $excludedDates = [];
        $startDate = Carbon::instance($this->start_date)->startOfDay();
        $endDate = Carbon::instance($this->end_date)->startOfDay();

        // احصل على أيام العطل الأسبوعية للموظف (توقع array من المفاتيح مثل ['friday','saturday'])
        $weeklyOffDays = array_map('strtolower', (array) ($employee->weekly_off_days ?? []));

        // خريطة الأسماء إلى أرقام أيام الأسبوع (متوافقة مع Carbon::dayOfWeek: 0=Sunday)
        $dayIndexToKey = [
            0 => 'sunday',
            1 => 'monday',
            2 => 'tuesday',
            3 => 'wednesday',
            4 => 'thursday',
            5 => 'friday',
            6 => 'saturday',
        ];

        // ابدأ بحلقة على جميع الأيام بين البداية والنهاية
        $currentDate = $startDate->copy();
        while ($currentDate->lessThanOrEqualTo($endDate)) {
            $dayIndex = $currentDate->dayOfWeek; // رقم اليوم (0=Sunday)
            $dayKey = $dayIndexToKey[$dayIndex] ?? null;

            // تحقق إذا كان اليوم عطلة أسبوعية بناءً على المفتاح
            if ($dayKey && in_array($dayKey, $weeklyOffDays, true)) {
                $excludedDates[] = $currentDate->format('Y-m-d');
            }
            // تحقق إذا كان اليوم عطلة رسمية (Holiday)
            elseif ($this->isOfficalHoliday($currentDate)) {
                $excludedDates[] = $currentDate->format('Y-m-d');
            }

            $currentDate->addDay();
        }

        // ✅ تخزين النتائج
        $this->excluded_days = $excludedDates;
        $this->excluded_days_count = count($excludedDates);
        $this->deductible_days = max(0, ($this->days_count ?? 0) - $this->excluded_days_count);
    }

    /**
     * ✅ التحقق من ما إذا كان اليوم عطلة رسمية
     */
    public function isOfficalHoliday($date)
    {
        // ابحث عن عطلة رسمية في جدول holidays للعام الحالي
        return Holiday::whereDate('date', '=', $date)->exists();
    }
}