<?php

namespace App\Models;

use App\Models\Traits\TenantScoped;
use App\Scopes\CompanyScope;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Collection;
use App\Models\Payroll;

/**
 * Class Employee
 *
 * نموذج الموظف مع وظائف متقدمة لحساب الإجازات، والعلاقات، وحسابات مرتبطة بالبايرول.
 *
 * ملاحظة: هذا الملف يحافظ على كل الدوال القديمة (المعلمة بـ @deprecated)
 * كما طلبت — لم يتم حذف أي دوال، فقط تم تصحيحها وتحسين التوثيق والتناغم بينها.
 *
 * @package App\Models
 */
class Employee extends Model
{
    use HasFactory;

    use TenantScoped; // استخدام الـ Trait



    /**
     * الحقول القابلة للملء
     *
     * @var array
     */
    protected $fillable = [
        'employee_id',
        'name',
        'email',
        'phone',
        'position_id',
        'department_id',
        'hire_date',
        'salary',
        'incentives',
        'status',
        'user_id',
        'company_id',
        'tenant_id',
        'address',
        'photo',
        'birth_date',
        'national_id',
        'age',
        'qualification',
        'cv_files',
        'scheduled_check_in',
        'scheduled_check_out',
        'monthly_leave_days_allowed',
        'overtime_hourly_rate',
        'deduction_hourly_rate',
        'overtime_hours_total',
        'overtime_pay_total',
        'accrue_leaves',
        'deduct_if_underworked',
        'overtime_paid',
        'component_1',
        'component_2',
        'component_3',
        'component_4',
        'component_5',
        'component_6',
        'component_7',
        'component_name_1',
        'component_name_2',
        'component_name_3',
        'component_name_4',
        'component_name_5',
        'component_name_6',
        'component_name_7',
        'annual_entitlement',
        'weekly_off_days',
        'default_paid_off_amount',
    ];

    /**
     * تشغيل الدوال عند تحميل النموذج (Model Booting)
     * تستخدم لتسجيل أحداث النموذج
     *
     * @return void
     */
    protected static function boot()
    {
        parent::boot();
        
        // تطبيق CompanyScope
        static::addGlobalScope(new CompanyScope());
        
        // حدث التوليد التلقائي: يتم تشغيله قبل عملية حفظ موظف جديد
        static::creating(function ($employee) {
            // التحقق مما إذا كان حقل employee_id فارغًا أو null
            if (empty($employee->employee_id)) {
                // تعيين الرقم الوظيفي الجديد المولد تلقائياً
                $employee->employee_id = self::generateUniqueEmployeeId();
            }
        });
    }

    /**
     * دالة لتوليد رقم وظيفي فريد بناءً على التسلسل والسنة.
     * مثال للتنسيق: EMP20250001
     *
     * @return string
     */
    private static function generateUniqueEmployeeId(): string
    {
        // 1. جلب آخر سجل موظف تم إدخاله
        $lastEmployee = self::orderBy('id', 'desc')->first();
        $lastNumber = 0;
        // 2. استخلاص الرقم التسلسلي
        if ($lastEmployee) {
            $lastId = $lastEmployee->employee_id;
            // استخلاص الجزء الرقمي فقط ثم آخر 4 أرقام تسلسلية
            $numericPart = preg_replace('/[^0-9]/', '', $lastId);
            $lastNumber = (int) substr($numericPart, -4);
        }
        // 3. زيادة وتنسيق الرقم الجديد
        $newNumber = $lastNumber + 1;
        $formattedNumber = str_pad($newNumber, 4, '0', STR_PAD_LEFT);
        // 4. بناء الرقم الوظيفي الكامل
        $year = date('Y');
        return "EMP{$year}{$formattedNumber}";
    }

    /**
     * casts
     *
     * @var array
     */
    protected $casts = [
        'hire_date' => 'date',
        'birth_date' => 'date',
        'salary' => 'decimal:2',
        'incentives' => 'decimal:2', // ✅ تم تغيير 'allowances' إلى 'incentives'
        'cv_files' => 'array',
        'scheduled_check_in' => 'string',
        'scheduled_check_out' => 'string',
        'monthly_leave_days_allowed' => 'integer',
        // ✅ إضافة التحويلات للحقول الجديدة
        'component_1' => 'decimal:2',
        'component_2' => 'decimal:2',
        'component_3' => 'decimal:2',
        'component_4' => 'decimal:2',
        'component_5' => 'decimal:2',
        'component_6' => 'decimal:2',
        'component_7' => 'decimal:2',
        // ✅ إضافة التحويلات لأسماء المكونات
        'component_name_1' => 'string',
        'component_name_2' => 'string',
        'component_name_3' => 'string',
        'component_name_4' => 'string',
        'component_name_5' => 'string',
        'component_name_6' => 'string',
        'component_name_7' => 'string',
        'annual_entitlement' => 'integer',
        'weekly_off_days' => 'array',
        'overtime_hourly_rate' => 'decimal:2',
        'deduction_hourly_rate' => 'decimal:2',
        'overtime_hours_total' => 'decimal:2',
        'overtime_pay_total' => 'decimal:2',
        'accrue_leaves' => 'boolean',
        'deduct_if_underworked' => 'boolean',
        'overtime_paid' => 'boolean',
        'default_paid_off_amount' => 'decimal:2',
    ];

    /**
     * Recalculate and persist overtime totals for this employee.
     * If $since is provided (Carbon or date string), totals are calculated from that date to now; otherwise all attendances are used.
     *
     * @param \DateTime|string|null $since
     * @return void
     */
    public function recalcOvertimeTotals($since = null): void
    {
        $query = $this->attendances();
        if ($since) {
            $sinceDate = \Carbon\Carbon::parse($since)->startOfDay();
            $query->where('date', '>=', $sinceDate->toDateString());
        }

        $attendances = $query->get();
        $hours = $attendances->sum(fn($a) => $a->getOvertimeHours());
        // If this employee should not be paid overtime, set pay to 0.
        $pay = 0;
        if (($this->overtime_paid ?? true) && ($this->overtime_hourly_rate ?? 0) > 0) {
            $pay = $hours * (float) ($this->overtime_hourly_rate ?? 0);
        }

        $this->overtime_hours_total = round($hours, 2);
        $this->overtime_pay_total = round($pay, 2);
        $this->saveQuietly();
    }

    /**
     * Ensure pending payrolls for this employee respect the current `overtime_paid` flag.
     * If overtime is disabled, zero out any existing "Overtime" payroll components on pending payrolls.
     * If overtime is enabled, recalculate and update (or create) the Overtime component for pending payrolls
     * based on attendances recorded for each payroll month.
     *
     * @return void
     */
    public function applyOvertimePaidFlagToPendingPayrolls(): void
    {
        try {
            $pendingPayrolls = $this->payrolls()->where('status', 'pending')->get();
            foreach ($pendingPayrolls as $payroll) {
                $payroll->load('payrollSalaryComponents');

                $existing = $payroll->payrollSalaryComponents->firstWhere('name', 'Overtime');

                if (!($this->overtime_paid ?? true)) {
                    // disable overtime: remove or zero the component
                    if ($existing) {
                        // set value to 0 to preserve record
                        $existing->value = 0;
                        $existing->save();
                    }
                } else {
                    // enable overtime: compute overtime for this payroll month and update/create component
                    $start = \Carbon\Carbon::create($payroll->year, $payroll->month, 1)->startOfMonth();
                    $end = (clone $start)->endOfMonth();
                    $hours = $this->attendances()
                        ->whereBetween('date', [$start, $end])
                        ->get()
                        ->sum(fn($a) => $a->getOvertimeHours());
                    $rate = (float) ($this->overtime_hourly_rate ?? 0);
                    $amount = ($hours > 0 && $rate > 0) ? round($hours * $rate, 2) : 0;

                    if ($existing) {
                        $existing->value = $amount;
                        $existing->save();
                    } else {
                        if ($amount > 0) {
                            $payroll->payrollSalaryComponents()->create([
                                'salary_component_id' => null,
                                'name' => 'Overtime',
                                'value' => $amount,
                            ]);
                        }
                    }
                }

                // Recalculate payroll totals after change
                $payroll->calculateTotalSalary();
                $payroll->calculateNetSalary();
                $payroll->save();
            }
        } catch (\Throwable $e) {
            // swallow errors to avoid breaking user flow; log if needed
            \Log::error('applyOvertimePaidFlagToPendingPayrolls failed for emp=' . ($this->id ?? 'null') . ': ' . $e->getMessage());
        }
    }

    // - العلاقات -
    public function department()
    {
        return $this->belongsTo(Department::class);
    }

    public function position()
    {
        return $this->belongsTo(Position::class);
    }

    public function attendances()
    {
        return $this->hasMany(Attendance::class, 'employee_id');
    }

    public function leaves()
    {
        return $this->hasMany(Leave::class, 'employee_id');
    }

    public function payrolls()
    {
        return $this->hasMany(Payroll::class, 'employee_id');
    }

    public function user()
    {
        return $this->hasOne(User::class, 'employee_id');
    }

    public function loans()
    {
        return $this->hasMany(Loan::class, 'employee_id');
    }

    public function deductions()
    {
        return $this->hasMany(Deduction::class, 'employee_id');
    }

    public function assignedAssets()
    {
        return $this->hasMany(Asset::class, 'assigned_to');
    }

    public function enrollments()
    {
        return $this->hasMany(TrainingEnrollment::class, 'employee_id');
    }

    public function leaveBalance()
    {
        return $this->hasOne(LeaveBalance::class, 'employee_id');
    }

    /**
     * علاقة بسجل التعويضات مع ترتيب من الأحدث للأقدم.
     */
    public function salaryHistories()
    {
        return $this->hasMany(EmployeeSalaryHistory::class, 'employee_id')->orderByDesc('effective_from');
    }

    /**
     * احصل على سجل التعويضات المعمول به لتاريخ معين.
     * ستُرجع السجل الأقرب (الأحدث) الذي effective_from <= بداية الشهر المطلوب.
     * 
     * @param \Carbon\Carbon|null $date التاريخ المطلوب (سيتم استخدام بداية الشهر)
     * @return \App\Models\EmployeeSalaryHistory|null
     */
    public function getCompensationForDate(\Carbon\Carbon $date = null)
    {
        $date = $date ?? now();
        // استخدام بداية الشهر للبحث
        $startOfMonth = $date->copy()->startOfMonth();
        
        return EmployeeSalaryHistory::where('employee_id', $this->id)
            ->where('effective_from', '<=', $startOfMonth->toDateString())
            ->orderByDesc('effective_from')
            ->first();
    }

    /**
     * احصل على التعويض الحالي المعمول به (للعرض أو لحسابات حالية).
     * يختار السجل ذو effective_from <= اليوم.
     */
    public function getCurrentCompensationAttribute()
    {
        return $this->getCompensationForDate(now());
    }

    public function leaveBalanceChanges()
    {
        return $this->hasMany(LeaveBalanceChange::class, 'employee_id');
    }

    // ✅ علاقة جديدة مع مكونات الراتب
    public function salaryComponents()
    {
        return $this->belongsToMany(SalaryComponent::class, 'employee_salary_components')
            ->withPivot('value')
            ->withTimestamps();
    }

    // - الصفات والوظائف -
    /**
     * إرجاع العمر (Attribute accessor)
     *
     * @param mixed $value
     * @return int|null
     */
    public function getAgeAttribute($value)
    {
        if ($value) {
            return $value;
        }
        if ($this->birth_date) {
            return Carbon::parse($this->birth_date)->age;
        }
        return null;
    }

    /**
     * إجمالي الراتب (الأساسي + الحوافز)
     *
     * @return float
     */
    public function getTotalSalaryAttribute()
    {
        return (float) (($this->salary ?? 0) + ($this->incentives ?? 0)); // ✅ تم تغيير 'allowances' إلى 'incentives'
    }

    /**
     * ✅ (مُعدّل) خاصية محسوبة لإرجاع إجمالي الراتب مع جميع المكونات (الأساسي + الحوافز + المكونات من العلاقة + المكونات من الحقول الثابتة).
     *
     * @return float
     */
    public function getTotalSalaryWithComponentsAttribute(): float
    {
        // 1. ابدأ بجمع الراتب الأساسي والحوافز
        $total = ($this->salary ?? 0) + ($this->incentives ?? 0);
        // 2. أضف المكونات من العلاقة (employee_salary_components)
        $relationalComponentsTotal = $this->salaryComponents()->sum('employee_salary_components.value');
        $total += $relationalComponentsTotal;
        // 3. أضف المكونات من الحقول الثابتة (component_1, component_2, ..., component_7)
        for ($i = 1; $i <= 7; $i++) {
            $fieldName = "component_$i";
            $value = $this->{$fieldName} ?? 0;
            $total += $value;
        }
        return $total;
    }

    /**
     * احصل على إجمالي الراتب (الأساسي + الحوافز + المكونات) لِتاريخ مُحدد.
     * يستخدم سجل `EmployeeSalaryHistory` إذا وُجد سجل effective_from <= التاريخ.
     */
    public function getTotalSalaryWithComponentsForDate(\Carbon\Carbon $date): float
    {
        $history = $this->getCompensationForDate($date);
        if ($history) {
            $base = (float) ($history->base_salary ?? 0);
            $incentives = (float) ($history->incentives ?? 0);
            $componentsSum = 0;
            if (is_array($history->components) && !empty($history->components)) {
                $componentsSum = array_sum(array_map(fn($v) => (float) $v, $history->components));
            } else {
                // fall back to relational components + named component fields
                try {
                    $relComponents = $this->salaryComponents()->get()->map(fn($c) => (float) ($c->pivot->value ?? 0))->sum();
                } catch (\Throwable $e) {
                    $relComponents = 0;
                }
                $namedSum = 0;
                for ($i = 1; $i <= 7; $i++) {
                    $namedSum += (float) ($this->{"component_$i"} ?? 0);
                }
                $componentsSum = $relComponents + $namedSum;
            }
            return $base + $incentives + $componentsSum;
        }
        // fallback to current computed attribute
        return $this->getTotalSalaryWithComponentsAttribute();
    }

    /**
     * هل تراكُم الإجازات مفعّل لهذا الموظف؟
     *
     * @return bool
     */
    public function isAccrualEnabled(): bool
    {
        return (bool) ($this->accrue_leaves ?? true);
    }

    /**
     * إرجاع الحصة الشهرية الممنوحة مع مراعاة خيار التراكم
     *
     * @return int
     */
    public function getMonthlyGrant(): int
    {
        if (!$this->isAccrualEnabled()) {
            return 0;
        }
        return (int) ($this->monthly_leave_days_allowed ?? 0);
    }

    /**
     * الراتب السنوي (الأساسي + الحوافز)
     *
     * @return float
     */
    public function getAnnualSalaryAttribute()
    {
        return (float) ((($this->salary ?? 0) + ($this->incentives ?? 0)) * 12); // ✅ تم تغيير 'allowances' إلى 'incentives'
    }

    /**
     * الراتب السنوي (الأساسي + الحوافز + المكونات - الطريقة القديمة)
     *
     * @return float
     */
    public function getAnnualSalaryWithComponentsAttribute()
    {
        return (float) ($this->getTotalSalaryWithComponentsAttribute() * 12);
    }

    /**
     * الراتب السنوي (الأساسي + الحوافز + المكونات - الطريقة الجديدة)
     *
     * @return float
     */
    public function getAnnualSalaryWithComponentsNewAttribute()
    {
        return (float) ($this->getTotalSalaryWithComponentsAttribute() * 12);
    }

    public function isActive(): bool
    {
        return $this->status === 'active';
    }

    // - Accessor لحل مشكلة relation not found -
    /**
     * Accessor لحل مشكلة RelationNotFoundException عند محاولة تحميل 'addition_components'.
     * يتم استدعاء هذه الدالة عندما يتم الوصول إلى $employee->addition_components
     * في العرض أو المتحكم. هذا يحل مشكلة RelationNotFoundException عند محاولة تحميلها.
     *
     * @return \Illuminate\Support\Collection
     */
    public function getAdditionComponentsAttribute()
    {
        $components = collect();
        // 1. إضافة الراتب الأساسي والحوافز كأول مكونات (أساسية)
        $components->push([
            'name' => __('app.basic_salary'),
            'value' => (float) ($this->salary ?? 0),
            'is_base' => true,
        ]);
        $components->push([
            'name' => __('app.incentives'), // ✅ تم تغيير "البدلات" إلى "الحوافز"
            'value' => (float) ($this->incentives ?? 0), // ✅ تم تغيير 'allowances' إلى 'incentives'
            'is_base' => true,
        ]);
        // 2. إضافة المكونات القابلة للتسمية (1-7) (الطريقة القديمة)
        for ($i = 1; $i <= 7; $i++) {
            $nameField = "component_name_$i";
            $valueField = "component_$i";
            $name = $this->{$nameField};
            $value = (float) ($this->{$valueField} ?? 0);
            if (!empty($name) && $value > 0) {
                $components->push([
                    'name' => $name,
                    'value' => $value,
                    'is_base' => false,
                ]);
            }
        }
        // 3. إضافة مكونات الراتب من العلاقة (employee_salary_components) (الطريقة الجديدة)
        if ($this->relationLoaded('salaryComponents')) {
            foreach ($this->salaryComponents as $comp) {
                $components->push([
                    'name' => $comp->name,
                    'value' => (float) $comp->pivot->value,
                    'is_base' => false,
                ]);
            }
        }
        return $components;
    }

    /**
     * ✅ (مُعدّل جذريًا) حساب تفاصيل خصم الإجازات حسب التراكم الشهري من تاريخ التعيين.
     * ⚠️ تم تعديله لحل مشكلة الخصم في الشهور التي لم يأخذ فيها الموظف إجازات.
     * - يحسب الرصيد الشهري المتاح (السابق + الحصة الجديدة - المستخدمة).
     * - إذا كان الرصيد سالبًا بعد الاستخدام، يُعاد إلى 0.
     * - يحسب الخصم المالي فقط للأيام الزائدة.
     *
     * @param int $year
     * @param int $month
     * @return \Illuminate\Support\Collection
     */
    public function calculateLeaveDeductionDetailsNew(int $year, int $month): Collection
    {
        if (!$this->hire_date || $this->monthly_leave_days_allowed === null) {
            return collect();
        }
        
        // ✅ إذا كان التراكم معطل، لا تحسب الأرصدة التراكمية
        if (!$this->isAccrualEnabled()) {
            return collect();
        }
        
        $leaveMonthlyDetails = [];
        // ✅ تحديد تاريخ البدء من تاريخ التعيين
        $startYear = Carbon::parse($this->hire_date)->year;
        $startMonth = Carbon::parse($this->hire_date)->month;
        // ✅ التحقق من أن الشهر/السنة المطلوبين بعد تاريخ التعيين
        if ($year < $startYear || ($year == $startYear && $month < $startMonth)) {
            return collect($leaveMonthlyDetails);
        }
        // ✅ التكرار من تاريخ التعيين إلى الشهر/السنة المطلوبين
        $current = Carbon::create($startYear, $startMonth, 1);
        $end = Carbon::create($year, $month, 1);
        // ✅ متغيرات لتخزين الأرصدة التراكمية:
        // - للإجازات الشهرية: previousMonthlyBalance
        // - للإجازات السنوية: previousAnnualBalance
        $previousMonthlyBalance = 0;
        $previousAnnualBalance = 0;
        
        while ($current->lessThanOrEqualTo($end)) {
            $y = $current->year;
            $m = $current->month;
            
            // ✅ حساب الحصة الممنوحة *الشهرية* في هذا الشهر
            $grantedMonthlyThisMonth = $this->monthly_leave_days_allowed;
            
            // ✅ حساب الأيام المستخدمة *من الشهرية* في هذا الشهر فقط
            $usedMonthlyThisMonth = $this->getUsedLeaveDaysInMonth($y, $m, 'monthly');
            
            // ✅ حساب الأيام المستخدمة *من السنوية* في هذا الشهر فقط
            $usedAnnualThisMonth = $this->getUsedLeaveDaysInMonth($y, $m, 'annual');
            
            // ========== حساب رصيد الشهرية ==========
            $availableMonthlyThisMonth = $previousMonthlyBalance + $grantedMonthlyThisMonth;
            $balanceAfterUsingMonthly = $availableMonthlyThisMonth - $usedMonthlyThisMonth;
            $excessMonthly = max(0, -$balanceAfterUsingMonthly); // إذا كان الرصيد سالباً، هناك فائض
            
            // حساب خصم الشهرية (على أساس الأيام الزائدة عن الرصيد)
            $monthlyDeductionAmount = 0;
            if ($excessMonthly > 0) {
                $totalSalaryWithComponents = $this->getTotalSalaryWithComponentsAttribute();
                $dailyRate = $totalSalaryWithComponents > 0 ? ($totalSalaryWithComponents / 30) : 0;
                $monthlyDeductionAmount = $dailyRate * $excessMonthly;
            }
            // الرصيد بعد الخصم: إذا كان موجب يُحفظ، وإذا كان سالب أو صفر يصير صفر
            $balanceAfterDeductionMonthly = max(0, $balanceAfterUsingMonthly);
            
            // ========== حساب رصيد السنوية ==========
            // السنوية تُعطى مرة واحدة فقط في السنة (في يناير أو الشهر الأول من التعيين)
            $grantedAnnualThisMonth = 0;
            $isFirstMonthOfYear = ($m === 1); // يناير
            $isFirstMonthOfHire = ($y === $startYear && $m === $startMonth); // الشهر الأول من التعيين
            
            // امنح السنوي في يناير أو في الشهر الأول من التعيين
            if ($isFirstMonthOfYear || $isFirstMonthOfHire) {
                $grantedAnnualThisMonth = (float) ($this->annual_entitlement ?? 0);
            }
            
            $availableAnnualThisMonth = $previousAnnualBalance + $grantedAnnualThisMonth;
            $balanceAfterUsingAnnual = $availableAnnualThisMonth - $usedAnnualThisMonth;
            $excessAnnual = max(0, -$balanceAfterUsingAnnual); // إذا كان الرصيد سالباً، هناك فائض
            
            // حساب خصم السنوية (على أساس الراتب الإجمالي)
            $annualDeductionAmount = 0;
            if ($excessAnnual > 0) {
                $totalSalaryWithComponents = $this->getTotalSalaryWithComponentsAttribute();
                $dailyRate = $totalSalaryWithComponents > 0 ? ($totalSalaryWithComponents / 30) : 0;
                $annualDeductionAmount = $dailyRate * $excessAnnual;
            }
            // الرصيد بعد الخصم: إذا كان موجب يُحفظ، وإذا كان سالب أو صفر يصير صفر
            $balanceAfterDeductionAnnual = max(0, $balanceAfterUsingAnnual);
            
            // ========== إجمالي الخصم والأيام ==========
            $totalUsedThisMonth = $usedMonthlyThisMonth + $usedAnnualThisMonth;
            $totalExcessDays = $excessMonthly + $excessAnnual;
            $totalDeductionAmount = $monthlyDeductionAmount + $annualDeductionAmount;
            
            $leaveMonthlyDetails[] = [
                'year' => $y,
                'month' => $m,
                'month_name' => Carbon::create($y, $m, 1)->translatedFormat('F Y'),
                
                // ========== بيانات الإجازات الشهرية ==========
                'granted_monthly' => (float) $grantedMonthlyThisMonth,
                'used_monthly' => (float) $usedMonthlyThisMonth,
                'balance_before_monthly' => (float) $availableMonthlyThisMonth,
                'balance_after_monthly' => (float) $balanceAfterDeductionMonthly,
                'excess_monthly' => (float) $excessMonthly,
                'deduction_amount_monthly' => (float) $monthlyDeductionAmount,
                
                // ========== بيانات الإجازات السنوية ==========
                'granted_annual' => (float) $grantedAnnualThisMonth, // السنوية مرة واحدة في السنة
                'used_annual' => (float) $usedAnnualThisMonth,
                'balance_before_annual' => (float) $availableAnnualThisMonth,
                'balance_after_annual' => (float) $balanceAfterDeductionAnnual,
                'excess_annual' => (float) $excessAnnual,
                'deduction_amount_annual' => (float) $annualDeductionAmount,
                
                // ========== المجموع الكلي ==========
                'granted_this_month' => (float) $grantedMonthlyThisMonth, // للتوافق مع العرض القديم
                'used_this_month' => (float) $totalUsedThisMonth,
                'excess_days' => (float) $totalExcessDays,
                'balance_before_deduction' => (float) ($availableMonthlyThisMonth + $availableAnnualThisMonth),
                'balance_after_deduction' => (float) ($balanceAfterDeductionMonthly + $balanceAfterDeductionAnnual),
                'leave_deduction_amount' => (float) $totalDeductionAmount,
                
                'debug_info' => [
                    'monthly_available' => $availableMonthlyThisMonth,
                    'monthly_used' => $usedMonthlyThisMonth,
                    'monthly_excess' => $excessMonthly,
                    'annual_available' => $availableAnnualThisMonth,
                    'annual_used' => $usedAnnualThisMonth,
                    'annual_excess' => $excessAnnual,
                ]
            ];
            
            // Add a debug log for traceability
            try {
                Log::debug('Employee leave accrual month details (separated)', [
                    'employee_id' => $this->id ?? null,
                    'year' => $y,
                    'month' => $m,
                    'granted_annual' => $grantedAnnualThisMonth,
                    'monthly_available' => $availableMonthlyThisMonth,
                    'monthly_used' => $usedMonthlyThisMonth,
                    'monthly_excess' => $excessMonthly,
                    'annual_available' => $availableAnnualThisMonth,
                    'annual_used' => $usedAnnualThisMonth,
                    'annual_excess' => $excessAnnual,
                    'total_deduction' => $totalDeductionAmount,
                ]);
            } catch (\Throwable $e) {
                // ignore logging failures
            }
            
            // ✅ تحديث الأرصدة للاستخدام في الشهر القادم
            $previousMonthlyBalance = $balanceAfterDeductionMonthly;
            $previousAnnualBalance = $balanceAfterDeductionAnnual;
            $current->addMonth();
        }
        return collect($leaveMonthlyDetails);
    }

    /**
     * ✅ (مُعدّل بدقة عالية) حساب خصم الإجازات حسب الرصيد التراكمي *المتاح قبل الاستخدام* في هذا الشهر.
     * ⚠️ هذا هو المنطق المطلوب بدقة:
     * 1. يحسب الرصيد التراكمي المتاح *قبل استخدام أي إجازات هذا الشهر*، أي:
     *    = (الرصيد التراكمي حتى نهاية الشهر السابق) + (الحصة الممنوحة هذا الشهر)
     * 2. يحسب الأيام المستخدمة *في هذا الشهر فقط*.
     * 3. إذا كانت الأيام المستخدمة ≤ الرصيد المتاح → لا خصم (0).
     * 4. إذا تجاوزت → يُخصم فقط الفرق من الراتب الإجمالي (الأساسي + الحوافز + المكونات).
     *
     * @param int $year
     * @param int $month
     * @return float
     */
    public function calculateLeaveDeductionForMonth(int $year, int $month): float
    {
        // ✅ إذا كان التراكم معطل، لا تحسب خصم الإجازات
        if (!$this->isAccrualEnabled()) {
            return 0.0;
        }
        
        // If deduction-on-underwork flag is disabled, no leave deduction should be applied.
        if (!($this->deduct_if_underworked ?? false)) {
            return 0.0;
        }

        if (!$this->hire_date || $this->getMonthlyGrant() === null) {
            return 0.0;
        }
        // ✅ 1. حساب الرصيد التراكمي حتى نهاية الشهر *السابق*
        //    مثال: ديسمبر 2024 → نحسب الرصيد حتى نوفمبر 2024
        $balanceAtEndOfPreviousMonth = ($month > 1)
            ? $this->getLeaveBalanceAtEndOfMonthNew($year, $month - 1)
            : ($year > Carbon::parse($this->hire_date)->year
                ? $this->getLeaveBalanceAtEndOfMonthNew($year - 1, 12)
                : 0);
        // ✅ 2. الحصة الممنوحة في *هذا الشهر*
        $grantedThisMonth = $this->getMonthlyGrant();
        // ✅ 3. إجمالي الرصيد المتاح *قبل استخدام الإجازات في هذا الشهر*
        $availableBalanceThisMonth = $balanceAtEndOfPreviousMonth + $grantedThisMonth;
        // ✅ 4. الأيام المستخدمة *فعلياً* في هذا الشهر (باستثناء عطلات نهاية الأسبوع)
        $usedThisMonth = $this->getUsedLeaveDaysInMonth($year, $month);
        // ✅ 5. الأيام الزائدة = max(0, المستخدمة − المتاح)
        $excessDays = max(0, $usedThisMonth - $availableBalanceThisMonth);
        // ✅ 6. حساب مبلغ الخصم (إن وُجد) — استخدم لقطة التعويض المعتمدة لهذا الشهر إن وُجدت
        $deductionAmount = 0.0;
        if ($excessDays > 0) {
            $snapshotDate = \Carbon\Carbon::create($year, $month, 1)->startOfMonth();
            $totalSalary = $this->getTotalSalaryWithComponentsForDate($snapshotDate);
            $dailyRate = $totalSalary / 30.0;
            $deductionAmount = $excessDays * $dailyRate;
        }
        return (float) $deductionAmount;
    }










/**
     * ✅ (إضافة الحل) Accessor للرصيد المتاح *قبل* خصم الاستخدام في الشهر الحالي.
     * يستخدم لعرض قيمة "رصيد قبل الخصم" في واجهة المستخدم (حسب الشهر الحالي).
     *
     * @return float
     */
    public function getLeaveBalanceBeforeDeductionAttribute(): float
    {
        if (!$this->hire_date || !$this->isAccrualEnabled()) {
            return 0.0;
        }

        $currentDate = Carbon::now();
        $year = $currentDate->year;
        $month = $currentDate->month;

        // 1. حساب الرصيد التراكمي حتى نهاية الشهر *السابق*
        //    نستخدم منطق getLeaveBalanceAtEndOfMonthNew للتراكم: (الرصيد يصفر إذا أصبح سالباً)
        $balanceAtEndOfPreviousMonth = 0.0;
        if ($month > 1) {
            $balanceAtEndOfPreviousMonth = $this->getLeaveBalanceAtEndOfMonthNew($year, $month - 1);
        } elseif ($year > Carbon::parse($this->hire_date)->year) {
            $balanceAtEndOfPreviousMonth = $this->getLeaveBalanceAtEndOfMonthNew($year - 1, 12);
        }

        // 2. الحصة الممنوحة في *هذا الشهر*
        $grantedThisMonth = $this->getMonthlyGrant();

        // 3. إجمالي الرصيد المتاح *قبل استخدام الإجازات في هذا الشهر*
        return (float) ($balanceAtEndOfPreviousMonth + $grantedThisMonth);
    }









    /**
     * ✅ (مُعدّل) حساب خصم الإجازات بالتراكمي الشهري من تاريخ التعيين.
     * ⚠️ تم تعديله لحل مشكلة الخصم في الشهور التي لم يأخذ فيها الموظف إجازات.
     *
     * @param int $year
     * @param int $month
     * @return float
     */
    public function calculateLeaveDeductionBasedOnCumulativeFromHireDate(int $year, int $month): float
    {
        if (!$this->hire_date || !$this->isAccrualEnabled()) {
            return 0;
        }
        // ✅ حساب الرصيد التراكمي *قبل* استخدام الإجازات في هذا الشهر
        $balanceBeforeThisMonth = $this->getLeaveBalanceAtEndOfMonth($year, $month - 1);
        // ✅ حساب الحصة الممنوحة في هذا الشهر
        $grantedThisMonth = $this->getMonthlyGrant();
        // ✅ حساب الأيام المستخدمة في هذا الشهر
        $usedThisMonth = $this->getUsedLeaveDaysInMonth($year, $month);
        // ✅ حساب الرصيد *بعد* استخدام الإجازات (قبل الخصم المالي)
        $balanceAfterUsingThisMonth = $balanceBeforeThisMonth + $grantedThisMonth - $usedThisMonth;
        // ✅ حساب الأيام الزائدة التي سيتم خصمها مالياً
        $actualDeductedDays = max(0, $usedThisMonth - ($balanceBeforeThisMonth + $grantedThisMonth));
        // ✅ حساب مبلغ الخصم بناءً على الأيام الزائدة
        $deductionAmount = 0;
        $dailyRate = 0;
        // ✅ استخدام "إجمالي الراتب مع المكونات" بدلًا من "الأساسي + الحوافز"
        $totalSalaryWithComponents = $this->getTotalSalaryWithComponentsAttribute();
        if ($totalSalaryWithComponents > 0) {
            $dailyRate = $totalSalaryWithComponents / 30;
        }
        $deductionAmount = $actualDeductedDays * $dailyRate;
        return $deductionAmount; // ✅ نُعيد فقط مبلغ الخصم
    }

    /**
     * ✅ (مُعدّل) حساب خصم الإجازات الشهري الجديد.
     * ⚠️ هذا هو المنطق الجديد الذي يتوافق مع تفضيلاتك.
     * - يحسب الخصم فقط إذا استخدم الموظف *في الشهر الحالي* أيامًا أكثر من *الحصة الشهرية*.
     * - إذا حدث ذلك، يُحسب الخصم ثم يُعاد الرصيد إلى 0.
     *
     * @param int $year
     * @param int $month
     * @return float
     */
    public function calculateMonthlyLeaveDeduction(int $year, int $month): float
    {
        // ✅ التحقق من وجود الحصة الشهرية
        // if flag not enabled, no monthly leave deduction
        if (!($this->deduct_if_underworked ?? false)) {
            return 0.0;
        }

        $monthlyGranted = $this->getMonthlyGrant();
        // ✅ حساب الأيام المستخدمة في هذا الشهر
        $usedThisMonth = $this->getUsedLeaveDaysInMonth($year, $month);
        // ✅ حساب الخصم المالي: فقط إذا تجاوزت الأيام المستخدمة الحصة الشهرية
        $excessDays = max(0, $usedThisMonth - $monthlyGranted);
        // ✅ حساب الأجر اليومي من إجمالي الراتب مع المكونات
        $totalSalary = $this->getTotalSalaryWithComponentsAttribute();
        $dailyRate = $totalSalary / 30;
        // ✅ حساب الخصم
        $leaveDeduction = $excessDays * $dailyRate;
        return $leaveDeduction;
    }

    /**
     * ✅ (مُعدّل) حساب رصيد الإجازات الشهري الجديد.
     * ⚠️ هذا هو المنطق الجديد الذي يتوافق مع تفضيلاتك.
     * - يحسب الرصيد الشهري المتاح (السابق + الحصة الجديدة - المستخدمة).
     * - إذا كان الرصيد سالبًا بعد الاستخدام، يُعاد إلى 0.
     *
     * @param int $year
     * @param int $month
     * @return int
     */
    public function getMonthlyLeaveBalance(int $year, int $month): int
    {
        // ✅ حساب الرصيد من الشهر الماضي
        $previousBalance = ($month > 1) ? $this->getMonthlyLeaveBalance($year, $month - 1) : 0;
        // ✅ الحصة الجديدة
        $grantedThisMonth = $this->getMonthlyGrant();
        // ✅ الأيام المستخدمة في هذا الشهر
        $usedThisMonth = $this->getUsedLeaveDaysInMonth($year, $month);
        // ✅ حساب الرصيد بعد الاستخدام
        $balanceAfterUsing = $previousBalance + $grantedThisMonth - $usedThisMonth;
        // ✅ تصفير الرصيد إذا كان سالبًا
        $finalBalance = max(0, $balanceAfterUsing);
        return $finalBalance;
    }

    /**
     * ✅ (مُعدّل) حساب الرصيد التراكمي الشهري من تاريخ التعيين.
     * ⚠️ تم تعديله لحل مشكلة الخصم في الشهور التي لم يأخذ فيها الموظف إجازات.
     * - يحسب الرصيد الشهري المتاح (السابق + الحصة الجديدة - المستخدمة).
     * - إذا كان الرصيد سالبًا بعد الاستخدام، يُعاد إلى 0.
     *
     * @param int $year
     * @param int $month
     * @return int
     */
    public function getLeaveBalanceAtEndOfMonthNew(int $year, int $month): int
    {
        if (!$this->hire_date || !$this->isAccrualEnabled()) {
            return 0;
        }
        $hireDate = Carbon::parse($this->hire_date);
        $targetDate = Carbon::create($year, $month, 1)->endOfMonth();
        // ✅ التحقق من أن التاريخ المطلوب بعد تاريخ التعيين
        if ($targetDate->lt($hireDate)) {
            return 0;
        }
        // ✅ التكرار من تاريخ التعيين إلى الشهر/السنة المطلوبين
        $current = Carbon::create($hireDate->year, $hireDate->month, 1);
        $end = Carbon::create($year, $month, 1);
        // ✅ متغير لتخزين الرصيد التراكمي *قبل* استخدام الإجازات في كل شهر
        $previousBalanceAfterDeduction = 0;
        while ($current->lessThanOrEqualTo($end)) {
            $y = $current->year;
            $m = $current->month;
            // ✅ حساب الرصيد *المتاح* *قبل* استخدام الإجازات في هذا الشهر
            $balanceBeforeThisMonth = $previousBalanceAfterDeduction;
            $grantedThisMonth = $this->getMonthlyGrant();
            $availableBalanceThisMonth = $balanceBeforeThisMonth + $grantedThisMonth;
            // ✅ حساب الأيام المستخدمة في هذا الشهر فقط
            $usedThisMonth = $this->getUsedLeaveDaysInMonth($y, $m);
            // ✅ حساب الرصيد *بعد* استخدام الإجازات في هذا الشهر (قبل الخصم المالي)
            $balanceAfterUsingThisMonth = $availableBalanceThisMonth - $usedThisMonth;
            // ✅ حساب الرصيد *بعد* الخصم المالي (الذي يُستخدم كرصيد جديد للشهر القادم)
            $balanceAfterDeductionThisMonth = max(0, $balanceAfterUsingThisMonth);
            // ✅ تحديث الرصيد للاستخدام في الشهر القادم
            $previousBalanceAfterDeduction = $balanceAfterDeductionThisMonth;
            $current->addMonth();
        }
        return $previousBalanceAfterDeduction;
    }

    // - دوال مساعدة -
    /**
     * ✅ (مُعدّل) حساب الأيام المستخدمة في شهر معين.
     * ⚠️ تم تعديله لحساب الأيام الفعلية (باستثناء عطلات نهاية الأسبوع).
     *
     * @param int $year
     * @param int $month
     * @return float
     */
    public function getUsedLeaveDaysInMonth(int $year, int $month, string $leaveType = 'all'): float
    {
        $startOfMonth = Carbon::create($year, $month, 1)->startOfMonth();
        $endOfMonth = (clone $startOfMonth)->endOfMonth();
        
        $query = $this->leaves()
            ->where(function ($q) {
                $q->where('status', 'approved')
                  ->orWhere('status', 'modified')
                  ->orWhere('approval_modified', true);
            })
            ->where(function ($q) use ($startOfMonth, $endOfMonth) {
                $q->where(function ($qq) use ($startOfMonth, $endOfMonth) {
                    $qq->where('start_date', '<=', $endOfMonth)
                        ->where('end_date', '>=', $startOfMonth);
                });
            });
        
        // حدد نوع الإجازة المطلوب
        if ($leaveType === 'monthly') {
            $query->ofCode('monthly');
        } elseif ($leaveType === 'annual') {
            $query->ofCodes(['annual', 'emergency', 'sick']);
        } else {
            // 'all' — جميع الإجازات
            $query->ofCodes(['annual', 'emergency', 'sick', 'monthly']);
        }
        
        $leaves = $query->get();
        
        // جلب الأيام الرسمية في هذا الشهر
        $holidays = Holiday::whereBetween('date', [$startOfMonth->format('Y-m-d'), $endOfMonth->format('Y-m-d')])
            ->pluck('date')
            ->map(fn($date) => $date instanceof \DateTime ? $date : \Carbon\Carbon::parse($date))
            ->toArray();
        
        // احصل على أيام العطل الأسبوعية للموظف
        $weeklyOffDays = array_map('strtolower', (array) ($this->weekly_off_days ?? []));
        
        // خريطة الأسماء إلى أرقام أيام الأسبوع (Carbon::dayOfWeek: 0=Sunday)
        $dayIndexToKey = [
            0 => 'sunday',
            1 => 'monday',
            2 => 'tuesday',
            3 => 'wednesday',
            4 => 'thursday',
            5 => 'friday',
            6 => 'saturday',
        ];
        
        $totalDays = 0;
        foreach ($leaves as $leave) {
            $start = Carbon::parse($leave->start_date)->max($startOfMonth);
            $end = Carbon::parse($leave->end_date)->min($endOfMonth);
            $currentDate = $start->copy();
            while ($currentDate->lessThanOrEqualTo($end)) {
                // استثنِ أيام عطل أسبوعية
                $dayIndex = $currentDate->dayOfWeek;
                $dayKey = $dayIndexToKey[$dayIndex] ?? null;
                
                if ($dayKey && in_array($dayKey, $weeklyOffDays, true)) {
                    $currentDate->addDay();
                    continue;
                }
                
                // استثنِ الأيام الرسمية (الـ Holidays)
                $isHoliday = in_array($currentDate->format('Y-m-d'), array_map(fn($h) => $h instanceof \DateTime ? $h->format('Y-m-d') : $h, $holidays));
                if (!$isHoliday) {
                    $totalDays += 1;
                }
                
                $currentDate->addDay();
            }
        }
        return $totalDays;
    }

    /**
     * ✅ (مُعدّل) حساب الأيام المستخدمة في سنة معينة حتى نهاية شهر معين.
     * ⚠️ تم تعديله لحساب الأيام الفعلية (باستثناء عطلات نهاية الأسبوع والأيام الرسمية).
     *
     * @param int $year
     * @param int $month
     * @return float
     */
    public function getUsedLeaveDaysForYearUntilMonth(int $year, int $month, string $leaveType = 'all'): float
    {
        $startDate = Carbon::create($year, 1, 1)->startOfYear();
        $endDate = Carbon::create($year, $month, 1)->endOfMonth();
        
        $query = $this->leaves()
            ->where(function ($q) {
                $q->where('status', 'approved')
                  ->orWhere('status', 'modified')
                  ->orWhere('approval_modified', true);
            })
            ->where(function ($q) use ($startDate, $endDate) {
                $q->where(function ($qq) use ($startDate, $endDate) {
                    $qq->where('start_date', '<=', $endDate)
                        ->where('end_date', '>=', $startDate);
                });
            });
        
        // حدد نوع الإجازة المطلوب
        if ($leaveType === 'monthly') {
            $query->ofCode('monthly');
        } elseif ($leaveType === 'annual') {
            $query->ofCodes(['annual', 'emergency', 'sick']);
        } else {
            // 'all' — جميع الإجازات
            $query->ofCodes(['annual', 'emergency', 'sick', 'monthly']);
        }
        
        $leaves = $query->get();
        
        // جلب الأيام الرسمية في السنة حتى نهاية الشهر المطلوب
        $holidays = Holiday::whereBetween('date', [$startDate->format('Y-m-d'), $endDate->format('Y-m-d')])
            ->pluck('date')
            ->map(fn($date) => $date instanceof \DateTime ? $date : \Carbon\Carbon::parse($date))
            ->toArray();
        
        // احصل على أيام العطل الأسبوعية للموظف
        $weeklyOffDays = array_map('strtolower', (array) ($this->weekly_off_days ?? []));
        
        // خريطة الأسماء إلى أرقام أيام الأسبوع (Carbon::dayOfWeek: 0=Sunday)
        $dayIndexToKey = [
            0 => 'sunday',
            1 => 'monday',
            2 => 'tuesday',
            3 => 'wednesday',
            4 => 'thursday',
            5 => 'friday',
            6 => 'saturday',
        ];
        
        $totalDays = 0;
        foreach ($leaves as $leave) {
            $start = Carbon::parse($leave->start_date)->max($startDate);
            $end = Carbon::parse($leave->end_date)->min($endDate);
            $currentDate = $start->copy();
            while ($currentDate->lessThanOrEqualTo($end)) {
                // استثنِ أيام عطل أسبوعية
                $dayIndex = $currentDate->dayOfWeek;
                $dayKey = $dayIndexToKey[$dayIndex] ?? null;
                
                if ($dayKey && in_array($dayKey, $weeklyOffDays, true)) {
                    $currentDate->addDay();
                    continue;
                }
                
                // استثنِ الأيام الرسمية (الـ Holidays)
                $isHoliday = in_array($currentDate->format('Y-m-d'), array_map(fn($h) => $h instanceof \DateTime ? $h->format('Y-m-d') : $h, $holidays));
                if (!$isHoliday) {
                    $totalDays += 1;
                }
                
                $currentDate->addDay();
            }
        }
        return $totalDays;
    }

    /**
 * ✅ (مُعدّل لحل مشكلة تكرار الاستقطاعات الشهرية) 
 * حساب إجمالي الاستقطاعات النشطة لهذا الشهر.
 * @param int $year
 * @param int $month
 * @return float
 */
public function getTotalActiveDeductionsForMonth(int $year, int $month): float
{
    $startOfMonth = Carbon::create($year, $month, 1)->startOfMonth();
    $endOfMonth = $startOfMonth->copy()->endOfMonth();

    // استقطاعات شهرية نشطة في هذا الشهر (قيمتها تُخصم هذا الشهر)
    // - 'is_monthly' = true
    // - 'status' = 'applied'
    // - 'deduction_date' <= نهاية الشهر (أو بداية الشهر، حسب التعريف)
    // - ('end_date' IS NULL OR 'end_date' >= بداية الشهر)
    // * المهم: * تجنب حساب نفس الاستقطاع الشهري عدة مرات.
    // الطريقة: نختار الاستقطاعات *الشهرية* التي *بدأ* تطبيقها قبل أو في هذا الشهر،
    // و*انتهى* تطبيقها بعد أو في هذا الشهر (أو لا تنتهي أبداً)،
    // *و* نجمع *قيمتها* مرة واحدة فقط.
    // * الافتراض: * إذا كان 'is_monthly' = 1، فإن 'amount' هو المبلغ *الشهري*.
    $monthlyDeductionsSum = $this->deductions()
        ->where('status', 'applied')
        ->where('is_monthly', true)
        ->where('deduction_date', '<=', $endOfMonth) // بدأت قبل أو في هذا الشهر
        ->where(function ($query) use ($startOfMonth) {
            $query->whereNull('end_date') // لا تنتهي أبداً
                  ->orWhere('end_date', '>=', $startOfMonth); // أو تنتهي بعد أو في هذا الشهر
        })
        ->sum('amount'); // نجمع *قيم* الاستقطاعات الشهرية *النشطة* في هذا الشهر (النظام يفترض أن 'amount' هو المبلغ الشهري)

    // استقطاعات لمرة واحدة تم تطبيقها *بالفعل* في هذا الشهر
    $oneTimeDeductions = $this->deductions()
        ->where('status', 'applied')
        ->where('is_monthly', false) // تأكد من أنها لمرة واحدة
        ->whereBetween('deduction_date', [$startOfMonth, $endOfMonth]) // تم تطبيقها في هذا الشهر بالتحديد
        ->sum('amount');

    // مجموع الاستقطاعات *التي تؤثر* على الراتب في هذا الشهر
    return $monthlyDeductionsSum + $oneTimeDeductions;
}








/**
     * ✅ (جديد) حساب إجمالي أقساط القروض النشطة لهذا الشهر.
     * @param int $year
     * @param int $month
     * @return float
     */
    public function getTotalLoanInstallmentsForMonth(int $year, int $month): float
    {
        $start = Carbon::create($year, $month, 1)->startOfMonth();
        $end = (clone $start)->endOfMonth();

        return $this->loans()
            ->where('status', 'active')
            ->with(['installments' => function ($q) use ($start, $end) {
                $q->whereBetween('due_date', [$start, $end])
                  ->where('status', 'pending'); // فقط الأقساط غير المدفوعة
            }])
            ->get()
            ->pluck('installments')
            ->flatten()
            ->sum('amount');
    }







    /**
     * ✅ (جديد) Accessor للرصيد المتاح.
     * يُستخدم لعرض الرصيد الحالي في واجهة المستخدم.
     *
     * @return int
     */
    public function getAvailableBalanceAttribute()
    {
        return $this->getLeaveBalanceAtEndOfMonthNew(Carbon::now()->year, Carbon::now()->month);
    }

    // - حساب الرصيد (قديم - مُحتفظ به للتوافق) -
    /**
     * @deprecated
     * حساب الرصيد الحالي للإجازات (نظريًا).
     *
     * @param Carbon|null $date
     * @return int
     */
    public function getCurrentLeaveBalance(\Carbon\Carbon $date = null): int
    {
        if (!$this->hire_date) {
            return 0;
        }
        $date = $date ?? now();
        $granted = $this->getAnnualGranted($date->year);
        $used = $this->getAnnualUsed($date->year);
        return (int) max(0, $granted - $used);
    }

    /**
     * @deprecated
     * حساب الرصيد الفعلي المتبقي حتى شهر معين (يطبق حد 12 شهرًا).
     * ✅ تم تعديل هذه الدالة لحساب الأيام الفعلية من start_date إلى end_date بدلاً من days_count فقط.
     *
     * @param int|null $year
     * @param int|null $month
     * @return float
     */
    public function getActualAnnualRemaining($year = null, $month = null): float
    {
        $year = $year ?? Carbon::now()->year;
        $month = $month ?? Carbon::now()->month;
        if (!$this->hire_date) {
            return 0;
        }
        $hireDate = Carbon::parse($this->hire_date)->startOfMonth();
        $currentMonthEnd = Carbon::createFromDate($year, $month, 1)->endOfMonth();
        if ($currentMonthEnd->lt($hireDate)) {
            return 0;
        }
        $monthsWorked = $hireDate->diffInMonths($currentMonthEnd) + 1;
        $monthsWorked = min($monthsWorked, 12);
        $grantedDays = ($this->monthly_leave_days_allowed ?? 0) * $monthsWorked;
        // ✅ حساب الأيام المستخدمة من سجل الإجازات (leaves) حتى نهاية الشهر المحدد
        $usedDays = $this->getUsedLeaveDaysForYearUntilMonth($year, $month);
        return (float) max(0, $grantedDays - $usedDays);
    }

    /**
     * @deprecated
     * حساب عدد أيام الإجازة الممنوحة شهريًا.
     *
     * @param int|null $year
     * @param int|null $month
     * @return float
     */
    public function getMonthlyGranted($year = null, $month = null)
    {
        $year = $year ?? Carbon::now()->year;
        $month = $month ?? Carbon::now()->month;
        if (!$this->hire_date) {
            return 0;
        }
        $hireDate = Carbon::parse($this->hire_date)->startOfMonth();
        $targetMonthStart = Carbon::createFromDate($year, $month, 1)->startOfMonth();
        $targetMonthEnd = (clone $targetMonthStart)->endOfMonth();
        if ($targetMonthEnd->lt($hireDate)) {
            return 0;
        }
        // إذا كان تاريخ التعيين داخل هذا الشهر، يتم حساب جزء من الحصة
        if ($hireDate->between($targetMonthStart, $targetMonthEnd)) {
            return (float) ($this->monthly_leave_days_allowed ?? 0);
        }
        // إذا كان الشهر بعد التعيين بالكامل
        if ($targetMonthStart->gte($hireDate)) {
            return (float) ($this->monthly_leave_days_allowed ?? 0);
        }
        // إذا كان الشهر قبل التعيين
        return 0;
    }

    /**
     * @deprecated
     * حساب عدد أيام الإجازة الممنوحة سنويًا.
     *
     * @param int|null $year
     * @return float
     */
    public function getAnnualGranted($year = null)
    {
        $year = $year ?? Carbon::now()->year;
        if (!$this->hire_date) {
            return 0;
        }
        $monthsWorkedInYear = $this->getMonthsWorkedForYear($year);
        $grantedDays = $monthsWorkedInYear * ($this->monthly_leave_days_allowed ?? 0);
        return round($grantedDays, 2);
    }

    /**
     * @deprecated
     * حساب عدد الأشهر المنقضية في سنة معينة منذ التعيين.
     *
     * @param int $year
     * @return int
     */
    private function getMonthsWorkedForYear($year)
    {
        if (!$this->hire_date) {
            return 0;
        }
        $hireDate = Carbon::parse($this->hire_date)->startOfDay();
        $startOfYear = Carbon::create($year, 1, 1)->startOfDay();
        $endOfYear = Carbon::create($year, 12, 31)->endOfDay();
        if ($hireDate->gt($endOfYear)) {
            return 0;
        }
        $effectiveStart = $hireDate->gt($startOfYear) ? $hireDate : $startOfYear;
        $monthsWorked = $effectiveStart->diffInMonths($endOfYear) + 1;
        if ($monthsWorked > 12) {
            $monthsWorked = 12;
        }
        return intval($monthsWorked);
    }

    /**
     * @deprecated
     * حساب عدد أيام الإجازة المستخدمة سنويًا.
     *
     * @param int|null $year
     * @return int
     */
    public function getAnnualUsed($year = null)
    {
        $year = $year ?? Carbon::now()->year;
        return $this->getUsedLeaveDaysForTypeInYear('annual', $year);
    }

    /**
     * @deprecated
     * مساعدة: حساب الأيام المستخدمة لنوع إجازة في سنة محددة.
     * ✅ تم تعديل هذه الدالة لحساب الأيام الفعلية من start_date إلى end_date بدلاً من days_count فقط.
     *
     * @param string $leaveType
     * @param int $year
     * @return int
     */
    private function getUsedLeaveDaysForTypeInYear($leaveType, $year)
    {
        $startDate = Carbon::createFromDate($year, 1, 1)->startOfYear();
        $endDate = (clone $startDate)->endOfYear();
        $usedDays = 0;
        $relevantLeaves = $this->leaves()
            ->where('status', 'approved')
            ->whereHas('leaveTypeModel', function ($q) use ($leaveType) {
                $q->where('code', $leaveType);
            })
            ->where(function ($query) use ($startDate, $endDate) {
                $query->whereBetween('start_date', [$startDate, $endDate])
                    ->orWhereBetween('end_date', [$startDate, $endDate])
                    ->orWhere(function ($q) use ($startDate, $endDate) {
                        $q->where('start_date', '<=', $startDate)
                            ->where('end_date', '>=', $endDate);
                    });
            })
            ->get();
        foreach ($relevantLeaves as $leave) {
            $leaveStart = Carbon::parse($leave->start_date)->startOfDay();
            $leaveEnd = Carbon::parse($leave->end_date)->startOfDay();
            $periodStart = $leaveStart->copy()->max($startDate);
            $periodEnd = $leaveEnd->copy()->min($endDate);
            if ($periodStart->lte($periodEnd)) {
                $usedDays += $periodStart->diffInDays($periodEnd) + 1;
            }
        }
        return (int) $usedDays;
    }

    /**
     * @deprecated
     * حساب الرصيد الفعلي المتبقي للسنة الحالية.
     *
     * @return float
     */
    public function getActualAnnualRemainingForCurrentYear(): float
    {
        $year = Carbon::now()->year;
        $granted = $this->getAnnualGranted($year);
        $used = $this->getAnnualUsed($year);
        $carryForward = 0;
        return max(0, $granted - $used + $carryForward);
    }

    // - التراكم الشهري منذ التعيين (لأغراض العرض) -
    /**
     * حساب تفاصيل التراكم الشهري منذ التعيين.
     *
     * ✅ تم تعديل هذه الدالة لربطها بدقة بسجل الإجازات (leaves) وحساب الرصيد بناءً على التراكم الفعلي.
     * ✅ هذا الحساب *يعرض* التراكم فقط، وليس له علاقة مباشرة بخصم الراتب الشهري.
     * ✅ تم إضافة الحقل 'leave_deduction_amount' لحل مشكلة Undefined array key.
     *
     * @return \Illuminate\Support\Collection
     */
    public function getMonthlyLeaveAccrualDetails()
    {
        if (!$this->hire_date) {
            return collect();
        }
        $leaveMonthlyDetails = [];
        $hireDate = Carbon::parse($this->hire_date);
        $currentDate = Carbon::now();
        $startYear = $hireDate->year;
        $startMonth = $hireDate->month;
        $endYear = $currentDate->year;
        $endMonth = $currentDate->month;
        // ✅ متغير لتخزين الرصيد من الشهر السابق
        $previousBalanceAfterDeduction = 0;
        for ($y = $startYear; $y <= $endYear; $y++) {
            $mStart = ($y == $startYear) ? $startMonth : 1;
            $mEnd = ($y == $endYear) ? $endMonth : 12;
            for ($m = $mStart; $m <= $mEnd; $m++) {
                // ✅ حساب الحصة الممنوحة في هذا الشهر
                $grantedThisMonth = $this->getMonthlyGrant();
                // ✅ حساب الأيام المستخدمة في هذا الشهر
                $usedThisMonth = $this->getUsedLeaveDaysInMonth($y, $m);
                // ✅ حساب الرصيد *بعد* استخدام الإجازات (قبل الخصم المالي)
                $balanceAfterUsing = $previousBalanceAfterDeduction + $grantedThisMonth - $usedThisMonth;
                // ✅ تصفير الرصيد إذا كان سالبًا
                $finalBalance = max(0, $balanceAfterUsing);
                // ✅ إضافة البيانات إلى المصفوفة
                $leaveMonthlyDetails[] = [
                    'year' => $y,
                    'month' => $m,
                    'month_name' => Carbon::create($y, $m, 1)->translatedFormat('F Y'),
                    'granted_this_month' => $grantedThisMonth,
                    'used_this_month' => $usedThisMonth,
                    'balance_before' => $previousBalanceAfterDeduction, // ✅ تم إضافة هذا المفتاح لحل الخطأ
                    'balance_after' => $finalBalance,
                    'notes' => 'ملاحظات شهر ' . $m . '/' . $y,
                ];
                // ✅ تحديث المتغير لاستخدامه في الشهر التالي
                $previousBalanceAfterDeduction = $finalBalance;
            }
        }
        return collect($leaveMonthlyDetails);
    }

    /**
     * ✅ (مُعدّل) حساب خصم الإجازات حسب التراكم الشهري.
     * ⚠️ تم تعديله لحل مشكلة الخصم في الشهور التي لم يأخذ فيها الموظف إجازات.
     *
     * @param int $year
     * @param int $month
     * @return float
     */
    public function calculateLeaveDeductionForMonthNew(int $year, int $month): float
    {
        if (!$this->hire_date || !$this->isAccrualEnabled()) {
            return 0;
        }
        // ✅ حساب الحصة الممنوحة في هذا الشهر
        $grantedThisMonth = $this->getMonthlyGrant();
        // ✅ حساب عدد الأيام الممنوحة حتى نهاية هذا الشهر (من التعيين)
        $hireYear = $this->hire_date->year;
        $hireMonth = $this->hire_date->month;
        $totalGrantedToDate = $grantedThisMonth * (($year - $hireYear) * 12 + ($month - $hireMonth) + 1);
        // ✅ حساب الأيام المستخدمة في هذا الشهر
        $usedThisMonth = $this->getUsedLeaveDaysInMonth($year, $month);
        // ✅ حساب الرصيد التراكمي قبل الخصم في هذا الشهر (الرصيد المتاح للاستخدام في هذا الشهر)
        $usedUntilLastMonth = $this->getUsedLeaveDaysForYearUntilMonth($year, $month - 1);
        $availableBalanceThisMonth = $totalGrantedToDate - $usedUntilLastMonth;
        // ✅ حساب الأيام الزائدة: الفرق بين المستخدمة هذا الشهر والرصيد المتاح
        $excessDays = max(0, $usedThisMonth - $availableBalanceThisMonth);
        // ✅ حساب مبلغ الخصم بناءً على الأيام الزائدة
        $deductionAmount = 0;
        if ($excessDays > 0) {
            $totalSalaryWithComponents = $this->getTotalSalaryWithComponentsAttribute();
            $dailyRate = $totalSalaryWithComponents / 30;
            $deductionAmount = $dailyRate * $excessDays;
        }
        return (float) $deductionAmount;
    }

    /**
     * حساب رصيد الإجازات التراكمي حتى نهاية شهر معين.
     *
     * هذا الحساب متماشٍ مع منطق PayrollController:
     * - الرصيد = (الحصة الشهرية * عدد الشهور من التعيين إلى نهاية الشهر المطلوب) - (إجمالي الأيام المأخوذة حتى نهاية الشهر المطلوب)
     * - إذا كان الرصيد < 0، نعتبره 0 (لا رصيد متوفر، لكن لا يُخصم مالياً من هنا، بل في دالة أخرى)
     *
     * @param int $year
     * @param int $month
     * @return int
     */
    public function getLeaveBalanceAtEndOfMonth(int $year, int $month): int
    {
        if (!$this->hire_date) {
            return 0;
        }
        $hireDate = Carbon::parse($this->hire_date);
        $targetDate = Carbon::create($year, $month, 1)->endOfMonth();
        // ✅ التحقق من أن التاريخ المطلوب بعد تاريخ التعيين
        if ($targetDate->lt($hireDate)) {
            return 0;
        }
        // ✅ حساب عدد الشهور من التعيين إلى نهاية الشهر المطلوب
        $monthsFromHire = $hireDate->diffInMonths($targetDate) + 1;
        // ✅ حساب إجمالي الأيام الممنوحة حتى نهاية هذا الشهر
        $totalGrantedDays = $monthsFromHire * $this->getMonthlyGrant();
        // ✅ حساب إجمالي الأيام المستخدمة حتى نهاية هذا الشهر
        $totalUsedDays = $this->getUsedLeaveDaysForYearUntilMonth($year, $month);
        // ✅ حساب الرصيد
        $balance = $totalGrantedDays - $totalUsedDays;
        return (int) $balance;
    }

    /**
     * ✅ (مُعدّل) حساب رصيد الإجازات التراكمي حتى نهاية شهر معين (طريقة بديلة).
     * ⚠️ تم تعديله لحل مشكلة الخصم في الشهور التي لم يأخذ فيها الموظف إجازات.
     *
     * @param int $year
     * @param int $month
     * @return array
     */
    public function getLeaveBalanceSummary(int $year, int $month): array
    {
        if (!$this->hire_date) {
            return [
                'total_granted' => 0,
                'total_used' => 0,
                'cumulative_balance' => 0,
                'current_month_allowance' => $this->getMonthlyGrant(),
                'current_month_used' => 0,
                'current_month_excess' => 0,
                'monthly_granted' => 0,
                'monthly_used' => 0,
                'monthly_balance' => 0,
                'annual_granted' => 0,
                'annual_used' => 0,
                'annual_balance' => 0,
            ];
        }
        $targetDate = Carbon::createFromDate($year, $month, 1)->endOfMonth();
        $hireDate = Carbon::parse($this->hire_date)->startOfMonth();
        if ($hireDate->greaterThan($targetDate)) {
            return [
                'total_granted' => 0,
                'total_used' => 0,
                'cumulative_balance' => 0,
                'current_month_allowance' => $this->monthly_leave_days_allowed ?? 0,
                'current_month_used' => 0,
                'current_month_excess' => 0,
                'monthly_granted' => 0,
                'monthly_used' => 0,
                'monthly_balance' => 0,
                'annual_granted' => 0,
                'annual_used' => 0,
                'annual_balance' => 0,
            ];
        }
        $monthsWorked = $hireDate->diffInMonths($targetDate) + 1;
        $monthsWorked = min($monthsWorked, 12);
        
        // ========== حساب الإجازات الشهرية ==========
        $monthlyGranted = $monthsWorked * $this->getMonthlyGrant();
        $monthlyUsed = $this->getUsedLeaveDaysForYearUntilMonth($year, $month, 'monthly');
        $monthlyBalance = max(0, $monthlyGranted - $monthlyUsed);
        
        // ========== حساب الإجازات السنوية ==========
        // السنويات تُعطى مرة واحدة في السنة الأولى بقيمة annual_entitlement
        $annualGranted = (float) ($this->annual_entitlement ?? 0);
        $annualUsed = $this->getUsedLeaveDaysForYearUntilMonth($year, $month, 'annual');
        $annualBalance = max(0, $annualGranted - $annualUsed);
        
        // ========== المجموع الكلي (للتوافق مع الحسابات القديمة) ==========
        $totalGranted = $monthlyGranted + $annualGranted;
        $totalUsed = $monthlyUsed + $annualUsed;
        $calculatedBalance = $totalGranted - $totalUsed;
        
        return [
            'total_granted' => (int) round($totalGranted),
            'total_used' => (int) round($totalUsed),
            'cumulative_balance' => max(0, (int) round($calculatedBalance)),
            'current_month_allowance' => $this->getMonthlyGrant(),
            'current_month_used' => $this->getUsedLeaveDaysInMonth($year, $month),
            'current_month_excess' => max(0, $this->getUsedLeaveDaysInMonth($year, $month) - $this->getMonthlyGrant()),
            
            // ========== تفاصيل الإجازات الشهرية ==========
            'monthly_granted' => (int) round($monthlyGranted),
            'monthly_used' => (int) round($monthlyUsed),
            'monthly_balance' => (int) round($monthlyBalance),
            
            // ========== تفاصيل الإجازات السنوية ==========
            'annual_granted' => (int) round($annualGranted),
            'annual_used' => (int) round($annualUsed),
            'annual_balance' => (int) round($annualBalance),
        ];
    }

    /**
     * Return an annual entitlement summary for a given year/month.
     *
     * Keys returned:
     * - annual_entitlement: float
     * - used_year_to_month: float
     * - annual_remaining: float
     * - annual_excess: float
     * - annual_excess_amount: float (monetary, based on daily rate)
     * - total_granted_year: int (from getLeaveBalanceSummary total_granted)
     * - total_used_year: int
     *
     * @param int $year
     * @param int $month
     * @return array
     */
    public function getAnnualEntitlementSummary(int $year, int $month): array
    {
        // entitlement stored on employee or fallback to monthly * 12
        $annual_entitlement = (float) ($this->annual_entitlement ?? ($this->monthly_leave_days_allowed * 12));

        // used days in the year up to given month
        $used_year_to_month = (float) $this->getUsedLeaveDaysForYearUntilMonth($year, $month);

        $annual_remaining = max(0, $annual_entitlement - $used_year_to_month);
        $annual_excess = max(0, $used_year_to_month - $annual_entitlement);

        $totalSalaryWithComponents = $this->getTotalSalaryWithComponentsAttribute();
        $dailyRate = $totalSalaryWithComponents > 0 ? ($totalSalaryWithComponents / 30.0) : 0;
        $annual_excess_amount = $annual_excess * $dailyRate;

        $leaveSummary = $this->getLeaveBalanceSummary($year, $month);
        $total_granted_year = $leaveSummary['total_granted'] ?? 0;
        $total_used_year = $leaveSummary['total_used'] ?? 0;

        return [
            'annual_entitlement' => $annual_entitlement,
            'used_year_to_month' => $used_year_to_month,
            'annual_remaining' => $annual_remaining,
            'annual_excess' => $annual_excess,
            'annual_excess_amount' => $annual_excess_amount,
            'total_granted_year' => $total_granted_year,
            'total_used_year' => $total_used_year,
        ];
    }

    /**
     * ✅ (مُعدّل) حساب ملخص الرواتب الشهرية (من التعيين حتى الآن) لعرضها في صفحة الموظف.
     * يُستخدم في دالة show في EmployeeController.
     *
     * @return \Illuminate\Support\Collection
     */
    public function getMonthlyPayrollSummary()
    {
        if (!$this->hire_date) {
            return collect();
        }
        $hireDate = Carbon::parse($this->hire_date);
        $currentDate = Carbon::now();
        $startYear = $hireDate->year;
        $startMonth = $hireDate->month;
        $payrollSummary = [];
        for ($y = $startYear; $y <= $currentDate->year; $y++) {
            $mStart = ($y == $startYear) ? $startMonth : 1;
            $mEnd = ($y == $currentDate->year) ? $currentDate->month : 12;
            for ($m = $mStart; $m <= $mEnd; $m++) {
                // ✅ جلب معرف الراتب إذا كان موجودًا أولاً
                $payrollRecord = Payroll::where('employee_id', $this->id)
                    ->where('year', $y)
                    ->where('month', $m)
                    ->with('payrollSalaryComponents')
                    ->first();
                
                // ✅ إذا كان هناك راتب محفوظ، استخدم القيم المحفوظة (الرواتب القديمة تحتفظ بقيمها)
                if ($payrollRecord) {
                    $grossSalary = ($payrollRecord->basic_salary ?? 0) + ($payrollRecord->incentives ?? 0);
                    // إضافة المكونات الإضافية من الراتب المحفوظ
                    $componentsTotal = $payrollRecord->payrollSalaryComponents
                        ->filter(fn($c) => ($c->value ?? 0) > 0)
                        ->sum('value');
                    $grossSalary += $componentsTotal;
                    $deductionsTotal = $payrollRecord->active_deductions ?? 0;
                    $loanInstallmentsTotal = $payrollRecord->loan_installments ?? 0;
                    $leaveDeduction = $payrollRecord->leave_deduction ?? 0;
                    $netSalary = $payrollRecord->net_salary ?? ($grossSalary - $deductionsTotal - $loanInstallmentsTotal - $leaveDeduction);
                } else {
                    // ✅ إذا لم يكن هناك راتب محفوظ، استخدم سجل التعويض للشهر المحدد
                    $snapshotDate = \Carbon\Carbon::create($y, $m, 1)->startOfMonth();
                    $grossSalary = $this->getTotalSalaryWithComponentsForDate($snapshotDate);
                    $deductionsTotal = $this->getTotalActiveDeductionsForMonth($y, $m);
                    $loanInstallmentsTotal = $this->getTotalLoanInstallmentsForMonth($y, $m);
                    $leaveDeduction = $this->calculateLeaveDeductionForMonth($y, $m);
                    $netSalary = $grossSalary - $deductionsTotal - $loanInstallmentsTotal - $leaveDeduction;
                }
                
                $payrollSummary[] = [
                    'year' => $y,
                    'month' => $m,
                    'month_name' => Carbon::create($y, $m, 1)->translatedFormat('F Y'),
                    'gross_salary' => (float) $grossSalary,
                    'deductions_total' => (float) $deductionsTotal,
                    'loan_installments_total' => (float) $loanInstallmentsTotal,
                    'leave_deduction' => (float) $leaveDeduction,
                    'net_salary' => (float) $netSalary,
                    'employee_id' => $this->id,
                    'payroll_id' => $payrollRecord ? $payrollRecord->id : null,
                ];
            }
        }
        return collect($payrollSummary);
    }
}