<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Company extends Model
{
    use SoftDeletes;

    protected $fillable = [
        'name',
        'email',
        'phone',
        'address',
        'city',
        'country',
        'tax_id',
        'status',
        'settings',
        // حقول الفوترة والاشتراك
        'subscription_start_date',
        'subscription_end_date',
        'subscription_fee',
        'billing_cycle',
        'payment_status',
        'last_payment_date',
        'next_payment_date',
        'outstanding_amount',
        'grace_period_days',
        'payment_notes',
        'billing_settings',
        // حقول التحكم والقيود
        'can_add_employees',
        'can_process_payroll',
        'can_access_reports',
        'max_employees',
        'current_employee_count',
        'suspended_at',
        'suspension_reason',
        'suspended_by',
        // حقول الإحصائيات
        'total_logins',
        'last_login_at',
        'usage_statistics',
    ];

    protected $casts = [
        'settings' => 'json',
        'billing_settings' => 'json',
        'usage_statistics' => 'json',
        'subscription_start_date' => 'date',
        'subscription_end_date' => 'date',
        'last_payment_date' => 'date',
        'next_payment_date' => 'date',
        'suspended_at' => 'datetime',
        'last_login_at' => 'datetime',
        'can_add_employees' => 'boolean',
        'can_process_payroll' => 'boolean',
        'can_access_reports' => 'boolean',
    ];

    // العلاقات
    public function users()
    {
        return $this->hasMany(User::class);
    }

    public function branches()
    {
        return $this->hasMany(Branch::class);
    }

    public function employees()
    {
        return $this->hasMany(Employee::class);
    }

    public function payrolls()
    {
        return $this->hasMany(Payroll::class);
    }

    public function loans()
    {
        return $this->hasMany(Loan::class);
    }

    public function deductions()
    {
        return $this->hasMany(Deduction::class);
    }

    public function leaves()
    {
        return $this->hasMany(Leave::class);
    }

    public function attendances()
    {
        return $this->hasMany(Attendance::class);
    }

    // دوال مساعدة للتحكم في الشركة
    public function isActive()
    {
        return $this->status === 'active' && !$this->isSuspended();
    }

    public function isSuspended()
    {
        return !is_null($this->suspended_at);
    }

    public function isPaymentOverdue()
    {
        if (!$this->next_payment_date) return false;
        return now()->greaterThan($this->next_payment_date) && $this->payment_status !== 'paid';
    }

    public function canAddMoreEmployees()
    {
        if (!$this->can_add_employees) return false;
        if ($this->max_employees && $this->current_employee_count >= $this->max_employees) {
            return false;
        }
        return true;
    }

    public function suspend($reason, $suspendedBy)
    {
        $this->suspended_at = now();
        $this->suspension_reason = $reason;
        $this->suspended_by = $suspendedBy;
        $this->status = 'suspended';
        $this->save();
    }

    public function unsuspend()
    {
        $this->suspended_at = null;
        $this->suspension_reason = null;
        $this->suspended_by = null;
        $this->status = 'active';
        $this->save();
    }

    public function updateEmployeeCount()
    {
        $this->current_employee_count = $this->employees()->count();
        $this->save();
    }

    public function recordLogin()
    {
        $this->total_logins++;
        $this->last_login_at = now();
        $this->save();
    }

    public function getDaysUntilPaymentDue()
    {
        if (!$this->next_payment_date) return null;
        return now()->diffInDays($this->next_payment_date, false);
    }

    public function getPaymentStatusColor()
    {
        return match($this->payment_status) {
            'paid' => 'success',
            'pending' => 'warning',
            'overdue' => 'danger',
            'cancelled' => 'secondary',
            default => 'primary'
        };
    }
}

