<?php

namespace App\Models;

use App\Scopes\CompanyScope;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

/**
 * نموذج سجل الحضور والانصراف.
 *
 * @property int $id
 * @property int $employee_id
 * @property \Illuminate\Support\Carbon $date
 * @property string|null $check_in
 * @property string|null $check_out
 * @property string|null $scheduled_check_in
 * @property string|null $scheduled_check_out
 * @property string $status
 * @property string|null $work_duration
 * @property string|null $scheduled_work_duration
 * @property string|null $notes
 * @property \Illuminate\Support\Carbon|null $created_at
 * @property \Illuminate\Support\Carbon|null $updated_at
 *
 * @property-read Employee $employee
 */
class Attendance extends Model
{
    use HasFactory;

    /**
     * اسم الجدول في قاعدة البيانات.
     *
     * @var string
     */
    protected $table = 'attendances';

    /**
     * الحقول القابلة للتعبئة.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'employee_id',
        'company_id',
        'date',
        'check_in',
        'check_out',
        'scheduled_check_in',
        'scheduled_check_out',
        'status',
        'work_duration',
        'scheduled_work_duration',
        'leave_id',
        'paid_for_off',
        'holiday_pay_amount',
        'notes'
    ];

    /**
     * التحويلات التلقائية للحقول.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'date' => 'date',
        'paid_for_off' => 'boolean',
        'holiday_pay_amount' => 'decimal:2',
        'check_in' => 'datetime:H:i:s',
        'check_out' => 'datetime:H:i:s',
        'scheduled_check_in' => 'datetime:H:i:s',
        'scheduled_check_out' => 'datetime:H:i:s',
        'work_duration' => 'string',
        'scheduled_work_duration' => 'string',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    /**
     * العلاقة مع نموذج الموظف.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function employee()
    {
        return $this->belongsTo(Employee::class);
    }

    /**
     * العلاقة مع سجل الإجازة المرتبط (إن وُجد)
     */
    public function leave()
    {
        return $this->belongsTo(\App\Models\Leave::class, 'leave_id');
    }

    /**
     * Boot model to listen for changes and update employee overtime totals.
     */
    protected static function boot()
    {
        parent::boot();
        
        // تطبيق CompanyScope
        static::addGlobalScope(new CompanyScope());

        // When creating an attendance record, if no leave_id provided but an
        // approved leave exists for the employee covering the attendance date,
        // link the attendance to that leave. This centralizes the logic so all
        // Attendance::create(...) call sites automatically get a consistent
        // `leave_id` when applicable.
        static::creating(function ($attendance) {
            try {
                if (empty($attendance->leave_id) && !empty($attendance->employee_id) && !empty($attendance->date)) {
                    $d = $attendance->date instanceof \Carbon\Carbon ? $attendance->date->toDateString() : (string) $attendance->date;
                    $lv = \App\Models\Leave::where('employee_id', $attendance->employee_id)
                        ->where('status', 'approved')
                        ->whereDate('start_date', '<=', $d)
                        ->whereDate('end_date', '>=', $d)
                        ->orderBy('id', 'desc')
                        ->first();
                    if ($lv) {
                        $attendance->leave_id = $lv->id;
                    }
                }
                // mark paid_for_off if this attendance is present and falls on a weekly off or official holiday
                try {
                    $status = $attendance->status ?? null;
                    if (empty($attendance->paid_for_off) && ($status === 'present' || !empty($attendance->check_in))) {
                        $d = $attendance->date instanceof \Carbon\Carbon ? $attendance->date->toDateString() : (string) $attendance->date;
                        // check official holiday
                        $holiday = \App\Models\Holiday::whereDate('date', $d)->first();
                        $isWeeklyOff = false;
                        $emp = null;
                        try {
                            $emp = \App\Models\Employee::find($attendance->employee_id);
                            if ($emp) {
                                $dayKey = \Carbon\Carbon::parse($d)->format('l');
                                $dayKey = strtolower($dayKey);
                                $weekly = $emp->weekly_off_days ?? [];
                                if (is_array($weekly) && in_array($dayKey, $weekly)) {
                                    $isWeeklyOff = true;
                                }
                            }
                        } catch (\Throwable $_) {
                            $isWeeklyOff = false;
                        }

                        if ($holiday || $isWeeklyOff) {
                            $attendance->paid_for_off = true;
                        }
                    }
                } catch (\Throwable $_) {
                    // swallow
                }
            } catch (\Throwable $_) {
                // Do not block creating attendance on observer failures
            }
        });

        static::created(function ($attendance) {
            try {
                if ($attendance->employee) {
                    $attendance->employee->recalcOvertimeTotals();
                }
            } catch (\Throwable $_) {
            }
        });

        static::updated(function ($attendance) {
            try {
                if ($attendance->employee) {
                    $attendance->employee->recalcOvertimeTotals();
                }
            } catch (\Throwable $_) {
            }
        });

        static::deleted(function ($attendance) {
            try {
                if ($attendance->employee) {
                    $attendance->employee->recalcOvertimeTotals();
                }
            } catch (\Throwable $_) {
            }
        });
    }

    // ------------------------------
    // 🔧 وظائف مساعدة اختيارية
    // ------------------------------

    /**
     * هل الموظف حاضر؟
     *
     * @return bool
     */
    public function isPresent(): bool
    {
        return $this->status === 'present';
    }

    /**
     * هل الموظف متأخر؟
     *
     * @return bool
     */
    public function isLate(): bool
    {
        return $this->status === 'late';
    }

    /**
     * هل الموظف غائب؟
     *
     * @return bool
     */
    public function isAbsent(): bool
    {
        return $this->status === 'absent';
    }

    /**
     * هل تم تسجيل الحضور؟
     *
     * @return bool
     */
    public function hasCheckIn(): bool
    {
        return !empty($this->check_in);
    }

    /**
     * هل تم تسجيل الانصراف؟
     *
     * @return bool
     */
    public function hasCheckOut(): bool
    {
        return !empty($this->check_out);
    }

    /**
     * احسب مدة العمل يدويًا (إذا لم تُحسب).
     *
     * @return string|null
     */
    public function calculateWorkDuration(): ?string
    {
        if (!$this->check_in || !$this->check_out) {
            return null;
        }

        try {
            $in = \Carbon\Carbon::parse($this->check_in);
            $out = \Carbon\Carbon::parse($this->check_out);
            $duration = $out->diff($in);
            return $duration->format('%H:%I:%S');
        } catch (\Exception $e) {
            return null;
        }
    }

    /**
     * Get worked hours as float (hours.decimal) based on check_in/check_out or work_duration.
     * Returns 0 if cannot calculate.
     *
     * @return float
     */
    public function getWorkedHours(): float
    {
        // If work_duration is present as H:i:s, use it
        if ($this->work_duration) {
            try {
                $parts = explode(':', $this->work_duration);
                $hours = (float) ($parts[0] ?? 0);
                $minutes = (float) ($parts[1] ?? 0);
                $seconds = (float) ($parts[2] ?? 0);
                return round($hours + $minutes / 60 + $seconds / 3600, 4);
            } catch (\Throwable $e) {
                // fallback
            }
        }

        if ($this->check_in && $this->check_out) {
            try {
                $in = \Carbon\Carbon::parse($this->check_in);
                $out = \Carbon\Carbon::parse($this->check_out);

                // ensure overnight shifts are handled
                if ($out->lt($in)) {
                    // assume overnight shift: add 1 day
                    $out = $out->addDay();
                }
                $seconds = $out->diffInSeconds($in);
                return abs(round($seconds / 3600, 4));
            } catch (\Throwable $e) {
                return 0.0;
            }
        }

        return 0.0;
    }

    /**
     * Get scheduled work hours as float (hours.decimal) from scheduled_check_in/out fields.
     * If not set, attempts to use scheduled_work_duration field.
     *
     * @return float
     */
    public function getScheduledHours(): float
    {
        if ($this->scheduled_work_duration) {
            try {
                $parts = explode(':', $this->scheduled_work_duration);
                $hours = (float) ($parts[0] ?? 0);
                $minutes = (float) ($parts[1] ?? 0);
                $seconds = (float) ($parts[2] ?? 0);
                return round($hours + $minutes / 60 + $seconds / 3600, 4);
            } catch (\Throwable $e) {
                // fallback
            }
        }

        if ($this->scheduled_check_in && $this->scheduled_check_out) {
            try {
                $in = \Carbon\Carbon::parse($this->scheduled_check_in);
                $out = \Carbon\Carbon::parse($this->scheduled_check_out);

                if ($out->lt($in)) {
                    $out = $out->addDay();
                }
                $seconds = $out->diffInSeconds($in);
                return abs(round($seconds / 3600, 4));
            } catch (\Throwable $e) {
                return 0.0;
            }
        }

        // fallback: if neither provided, return 0
        return 0.0;
    }

    /**
     * Overtime hours for this attendance record: max(0, worked - scheduled)
     *
     * @return float
     */
    public function getOvertimeHours(): float
    {
        $worked = $this->getWorkedHours();
        $scheduled = $this->getScheduledHours();
        $ot = max(0, $worked - $scheduled);
        return round($ot, 4);
    }

    /**
     * Hours arrived early: scheduled_check_in - actual check_in (positive if earlier)
     * @return float
     */
    public function getEarlyHours(): float
    {
        if (!$this->check_in || !$this->scheduled_check_in) return 0.0;
        try {
            $actual = \Carbon\Carbon::parse($this->check_in);
            $scheduled = \Carbon\Carbon::parse($this->scheduled_check_in);
            if ($actual->lt($scheduled)) {
                $seconds = $scheduled->diffInSeconds($actual);
                return round($seconds / 3600, 4);
            }
        } catch (\Throwable $e) {
            return 0.0;
        }
        return 0.0;
    }

    /**
     * Hours arrived late: actual check_in - scheduled_check_in (positive if later)
     * @return float
     */
    public function getLateHours(): float
    {
        if (!$this->check_in || !$this->scheduled_check_in) return 0.0;
        try {
            $actual = \Carbon\Carbon::parse($this->check_in);
            $scheduled = \Carbon\Carbon::parse($this->scheduled_check_in);
            if ($actual->gt($scheduled)) {
                $seconds = $actual->diffInSeconds($scheduled);
                return round($seconds / 3600, 4);
            }
        } catch (\Throwable $e) {
            return 0.0;
        }
        return 0.0;
    }

    /**
     * Monetary value for early/late hours based on employee's total monthly salary.
     * Positive value for early (earnings), negative for late (deduction).
     * @return float
     */
    public function getEarlyLateValue(): float
    {
        if (!$this->employee) return 0.0;
        $early = $this->getEarlyHours();
        $late = $this->getLateHours();
        if ($early == 0 && $late == 0) return 0.0;

            // employee monthly total (includes components) - use history snapshot for this attendance date if available
            $snapshotDate = $this->date ? \Carbon\Carbon::parse($this->date) : \Carbon\Carbon::now();
            $monthlyTotal = (float) $this->employee->getTotalSalaryWithComponentsForDate($snapshotDate);
        if ($monthlyTotal <= 0) return 0.0;

        // daily rate approximation (30 days month)
        $dailyRate = $monthlyTotal / 30.0;
        // scheduled hours for day
        $schedHours = $this->getScheduledHours() ?: 8.0;
        $hourlyRate = $dailyRate / max(1, $schedHours);

        $value = ($early * $hourlyRate) - ($late * $hourlyRate);
        return round($value, 2);
    }

    /**
     * Underworked hours: scheduled - worked (positive if worked less than scheduled)
     * @return float
     */
    public function getUnderworkedHours(): float
    {
        try {
            $scheduled = $this->getScheduledHours();
            $worked = $this->getWorkedHours();
            $under = max(0, round($scheduled - $worked, 4));
            return $under;
        } catch (\Throwable $e) {
            return 0.0;
        }
    }

    /**
     * Monetary deduction for underworked hours based on employee salary.
     * Returns positive number representing deduction amount.
     * @return float
     */
    public function getUnderworkedValue(): float
    {
        if (!$this->employee) return 0.0;
        $under = $this->getUnderworkedHours();
        if ($under <= 0) return 0.0;

        // Only apply deduction if employee has deduct_if_underworked enabled
        if (!($this->employee->deduct_if_underworked ?? false)) {
            return 0.0;
        }

        // Use custom deduction hourly rate if set, otherwise calculate based on salary
        $customRate = (float) ($this->employee->deduction_hourly_rate ?? 0);
        if ($customRate > 0) {
            // Use custom rate specified by employee
            $deduction = $under * $customRate;
            return round($deduction, 2);
        }

        // Fallback to original calculation based on salary
        $snapshotDate = $this->date ? \Carbon\Carbon::parse($this->date) : \Carbon\Carbon::now();
        $monthlyTotal = (float) $this->employee->getTotalSalaryWithComponentsForDate($snapshotDate);
        if ($monthlyTotal <= 0) return 0.0;

        $dailyRate = $monthlyTotal / 30.0;
        $schedHours = $this->getScheduledHours() ?: 8.0;
        $hourlyRate = $dailyRate / max(1, $schedHours);

        $deduction = $under * $hourlyRate;
        return round($deduction, 2);
    }
}