<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
// ❌ إزالة استيراد SoftDeletes
// use Illuminate\Database\Eloquent\SoftDeletes;

class Asset extends Model
{
    // ❌ إزالة SoftDeletes من التريتات المستخدمة
    use HasFactory; // <-- فقط HasFactory

    protected $fillable = [
        'name',
        'description',
        'category',
        'serial_number',
        'value',
        'purchase_date',
        'warranty_expiry',
        'location',
        'status',
        'assigned_to',
        'photo',
        'notes'
    ];

    protected $casts = [
        'value' => 'decimal:2',
        'purchase_date' => 'date',
        'warranty_expiry' => 'date',
        'is_active' => 'boolean'
        // ❌ إزالة deleted_at من casts إن وُجد (ليس موجودًا في الكود الأصلي، لكن للتأكيد)
        // 'deleted_at' => 'datetime',
    ];

    // ✅ تحديث مصفوفة dates لإزالة deleted_at
    protected $dates = [
        'purchase_date',
        'warranty_expiry',
        'created_at',
        'updated_at'
        // ❌ إزالة deleted_at
        // 'deleted_at'
    ];

    // العلاقة مع الموظف المُسنَد إليه
    public function assignedTo()
    {
        return $this->belongsTo(Employee::class, 'assigned_to');
    }

    // العلاقة مع الموظف المالك (تأكد من وجود هذا الحقل في الجدول)
    // إذا لم يكن لديك عمود employee_id في جدول assets، يجب إزالة هذه العلاقة أو إضافته.
    // public function employee()
    // {
    //     return $this->belongsTo(Employee::class, 'employee_id');
    // }

    // العلاقة مع سجل الإسناد
    public function assignments()
    {
        return $this->hasMany(AssetAssignment::class, 'asset_id');
    }

    // العلاقة مع سجل الصيانة
    public function maintenanceRecords()
    {
        return $this->hasMany(AssetMaintenance::class, 'asset_id');
    }

    // الحصول على اسم الفئة
    public function getCategoryNameAttribute()
    {
        $categories = [
            'electronic' => 'إلكترونيات',
            'furniture' => 'أثاث',
            'vehicle' => 'مركبات',
            'equipment' => 'معدات',
            'software' => 'برمجيات'
        ];

        return $categories[$this->category] ?? $this->category;
    }

    // الحصول على اسم الحالة
    public function getStatusNameAttribute()
    {
        $statuses = [
            'available' => 'متاح',
            'assigned' => 'مُسنَد',
            'maintenance' => 'صيانة',
            'retired' => 'متقاعد',
            'damaged' => 'تالف'
        ];

        return $statuses[$this->status] ?? $this->status;
    }

    // نطاق للبحث
    public function scopeSearch($query, $search)
    {
        return $query->where('name', 'like', "%{$search}%")
                    ->orWhere('serial_number', 'like', "%{$search}%")
                    ->orWhere('description', 'like', "%{$search}%");
    }

    // نطاق للأصول النشطة
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    // نطاق للأصول المتاحة
    public function scopeAvailable($query)
    {
        return $query->where('status', 'available');
    }

    // نطاق للأصول المُسنَدة
    public function scopeAssigned($query)
    {
        return $query->where('status', 'assigned');
    }

    // نطاق حسب الفئة
    public function scopeByCategory($query, $category)
    {
        return $query->where('category', $category);
    }

    // نطاق حسب الحالة
    public function scopeByStatus($query, $status)
    {
        return $query->where('status', $status);
    }
}