<?php

namespace App\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\Rule;
use App\Models\LeaveType;

class LeaveRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {
        // جلب الأكواد من قاعدة البيانات، أو استخدام القيم الافتراضية إذا لم تكن موجودة
        $codes = LeaveType::pluck('code')->toArray();
        if (empty($codes)) {
            // القيم الافتراضية إذا لم تكن هناك بيانات في قاعدة البيانات
            $codes = ['annual', 'sick', 'emergency', 'maternity', 'paternity', 'unpaid', 'monthly'];
        }

        // قواعد أساسية مشتركة
        $rules = [
            'employee_id' => 'required|exists:employees,id',
            'leave_type' => ['required', Rule::in($codes)],
            'end_date' => 'required|date|after_or_equal:start_date',
            'reason' => 'required|string|max:1000',
            'contact_details' => 'nullable|string|max:500',
        ];

        // عند الإنشاء نمنع تحديد تاريخ بداية في الماضي
        if ($this->isMethod('post')) {
            $rules['start_date'] = 'required|date|after_or_equal:today';
            // عند الإنشاء الحالة اختيارية (سيتم تعيينها إلى 'pending' في الـ Controller)
            $rules['status'] = 'nullable|in:pending,approved,rejected,cancelled,modified';
        } else {
            // عند التحديث: نسمح بتعديل طلبات قد تبدأ في الماضي، لذا نحذف قيد after_or_equal:today
            $rules['start_date'] = 'required|date';
            // لا نجبر على وجود حقل الحالة في نموذج التعديل (قد تكون صفحة التحرير لا تتضمنه)
            $rules['status'] = 'nullable|in:pending,approved,rejected,cancelled,modified';
        }

        return $rules;
    }

    /**
     * Get the error messages for the defined validation rules.
     *
     * @return array
     */
    public function messages()
    {
        return [
            'employee_id.required' => 'الموظف مطلوب',
            'employee_id.exists' => 'الموظف المحدد غير موجود',
            'leave_type.required' => 'نوع الإجازة مطلوب',
            'leave_type.in' => 'نوع الإجازة غير صحيح',
            'start_date.required' => 'تاريخ البدء مطلوب',
            'start_date.date' => 'تاريخ البدء غير صحيح',
            'end_date.required' => 'تاريخ الانتهاء مطلوب',
            'end_date.date' => 'تاريخ الانتهاء غير صحيح',
            'end_date.after_or_equal' => 'تاريخ الانتهاء يجب أن يكون أكبر من أو يساوي تاريخ البدء',
            'reason.required' => 'السبب مطلوب',
            'reason.max' => 'السبب يجب ألا يتجاوز 1000 حرف',
            'contact_details.max' => 'تفاصيل الاتصال يجب ألا تتجاوز 500 حرف',
            'status.required' => 'الحالة مطلوبة',
            'status.in' => 'الحالة يجب أن تكون واحدة من القيم المسموحة'
        ];
    }
}
